<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Shipment;
use App\Models\Trip;
use App\Models\TripCompanion;
use App\Models\OnlinePurchaseRequest;
use App\Models\OnlinePurchaseShopper;
use App\Models\Ticket;
use Carbon\Carbon;
use Illuminate\Http\JsonResponse;

class AdminDashboardController extends Controller
{
    public function index(): JsonResponse
    {
        // ✅ آمار کلی
        $stats = [
            'total_users' => User::count(),
            'pending_shipments' => Shipment::where('status', 'pending')->count(),
            'pending_trips' => Trip::where('status', 'pending')->count(),
            'pending_companions' => TripCompanion::where('status', 'pending')->count(),
            'pending_purchases' => OnlinePurchaseRequest::where('status', 'pending')->count(),
            'pending_shoppers' => OnlinePurchaseShopper::where('status', 'pending')->count(),
            'open_tickets' => Ticket::where('status', 'open')->count(),
        ];

        // ✅ آخرین درخواست‌های در انتظار (از همه بخش‌ها)
        $pendingRequests = collect();

        // مرسولات
        $shipments = Shipment::with('user:id,name')
            ->where('status', 'pending')
            ->latest()
            ->limit(3)
            ->get()
            ->map(fn($item) => [
                'id' => 'SH-' . $item->id,
                'request_id' => $item->id,
                'type' => 'shipment',
                'type_label' => 'مرسوله',
                'user' => $item->user?->name ?? 'نامشخص',
                'detail' => $item->origin_city . ' ← ' . $item->destination_city,
                'created_at' => $item->created_at,
            ]);

        // سفرها
        $trips = Trip::with('user:id,name')
            ->where('status', 'pending')
            ->latest()
            ->limit(3)
            ->get()
            ->map(fn($item) => [
                'id' => 'TR-' . $item->id,
                'request_id' => $item->id,
                'type' => 'trip',
                'type_label' => 'سفر',
                'user' => $item->user?->name ?? 'نامشخص',
                'detail' => $item->origin_city . ' ← ' . $item->destination_city,
                'created_at' => $item->created_at,
            ]);

        // همسفری
        $companions = TripCompanion::with('user:id,name')
            ->where('status', 'pending')
            ->latest()
            ->limit(2)
            ->get()
            ->map(fn($item) => [
                'id' => 'TC-' . $item->id,
                'request_id' => $item->id,
                'type' => 'companion',
                'type_label' => 'همسفری',
                'user' => $item->user?->name ?? 'نامشخص',
                'detail' => $item->origin_city . ' ← ' . $item->destination_city,
                'created_at' => $item->created_at,
            ]);

        // خریدهای اینترنتی
        $purchases = OnlinePurchaseRequest::with('user:id,name')
            ->where('status', 'pending')
            ->latest()
            ->limit(2)
            ->get()
            ->map(fn($item) => [
                'id' => 'PU-' . $item->id,
                'request_id' => $item->id,
                'type' => 'purchase',
                'type_label' => 'خرید اینترنتی',
                'user' => $item->user?->name ?? 'نامشخص',
                'detail' => 'ارسال به: ' . ($item->delivery_city ?? 'نامشخص'),
                'created_at' => $item->created_at,
            ]);

        // خریداران
      // خریداران
$shoppers = OnlinePurchaseShopper::with('user:id,name')
    ->where('status', 'pending')
    ->latest()
    ->limit(2)
    ->get()
    ->map(function($item) {
        // categories به دلیل cast به آرایه، دیگر نیازی به json_decode ندارد
        $categories = is_array($item->categories) ? $item->categories : [];

        // ترجمه دسته‌بندی‌ها به فارسی
        $categoryTranslations = [
            'electronics' => 'الکترونیک',
            'books' => 'کتاب',
            'clothing' => 'پوشاک',
            'home' => 'خانه و آشپزخانه',
            'beauty' => 'زیبایی و سلامت',
            'toys' => 'اسباب بازی',
            'sports' => 'ورزشی',
            'food' => 'مواد غذایی',
        ];

        $persianCategories = array_map(
            fn($cat) => $categoryTranslations[$cat] ?? $cat,
            $categories
        );

        $categoriesText = !empty($persianCategories) 
            ? implode('، ', $persianCategories) 
            : 'عمومی';

        return [
            'id' => 'SHP-' . $item->id,
            'request_id' => $item->id,
            'type' => 'shopper',
            'type_label' => 'خریدار',
            'user' => $item->user?->name ?? 'نامشخص',
            'detail' => 'دسته‌بندی: ' . $categoriesText,
            'created_at' => $item->created_at,
        ];
    });

        // ترکیب و مرتب‌سازی بر اساس تاریخ
        $pendingRequests = $pendingRequests
            ->concat($shipments)
            ->concat($trips)
            ->concat($companions)
            ->concat($purchases)
            ->concat($shoppers)
            ->sortByDesc('created_at')
            ->take(10)
            ->values();

        // ✅ فعالیت‌های اخیر
        $recentActivities = $this->getRecentActivities();

        return response()->json([
            'stats' => $stats,
            'pending_requests' => $pendingRequests,
            'recent_activities' => $recentActivities,
        ]);
    }

    private function getRecentActivities(): array
    {
        $activities = [];

        // آخرین تأییدها
        $approvedShipments = Shipment::with('user:id,name')
            ->where('status', 'approved')
            ->where('updated_at', '>=', Carbon::now()->subDays(7))
            ->latest('updated_at')
            ->limit(3)
            ->get()
            ->map(fn($item) => [
                'type' => 'approved',
                'description' => 'مرسوله ' . $item->title . ' تأیید شد',
                'time' => Carbon::parse($item->updated_at)->diffForHumans(),
                'user' => [
                    'name' => $item->user?->name ?? 'نامشخص',
                    'avatar' => 'https://ui-avatars.com/api/?name=' . urlencode($item->user?->name ?? 'نامشخص') . '&background=random',
                ],
            ]);

        $approvedTrips = Trip::with('user:id,name')
            ->where('status', 'approved')
            ->where('updated_at', '>=', Carbon::now()->subDays(7))
            ->latest('updated_at')
            ->limit(2)
            ->get()
            ->map(fn($item) => [
                'type' => 'approved',
                'description' => 'سفر به ' . $item->destination_city . ' تأیید شد',
                'time' => Carbon::parse($item->updated_at)->diffForHumans(),
                'user' => [
                    'name' => $item->user?->name ?? 'نامشخص',
                    'avatar' => 'https://ui-avatars.com/api/?name=' . urlencode($item->user?->name ?? 'نامشخص') . '&background=random',
                ],
            ]);

        // کاربران جدید
        $newUsers = User::where('created_at', '>=', Carbon::now()->subDays(7))
            ->latest('created_at')
            ->limit(2)
            ->get()
            ->map(fn($user) => [
                'type' => 'new_user',
                'description' => 'کاربر جدید ' . $user->name . ' ثبت‌نام کرد',
                'time' => Carbon::parse($user->created_at)->diffForHumans(),
                'user' => [
                    'name' => $user->name,
                    'avatar' => 'https://ui-avatars.com/api/?name=' . urlencode($user->name) . '&background=random',
                ],
            ]);

        // ترکیب و مرتب‌سازی
        $activities = collect($activities)
            ->concat($approvedShipments)
            ->concat($approvedTrips)
            ->concat($newUsers)
            ->take(5)
            ->toArray();

        return $activities;
    }
}
