<?php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\OnlinePurchaseRequest;
use App\Models\OnlinePurchaseShopper;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class AdminOnlinePurchaseController extends Controller
{
    // ========== درخواست‌های خرید ==========

    public function purchaseIndex(Request $request): JsonResponse
    {
        $query = OnlinePurchaseRequest::with('user:id,name,phone');

        if ($request->has('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        $purchases = $query->get()->map(fn($item) => [
            'id'               => $item->id,
            'title'            => $item->title,
            'user'             => ['name' => $item->user?->name ?? 'نامشخص','phone' => $item->user?->phone ?? 'نامشخص'],
            'purchase_country' => 'اینترنت',
            'purchase_city'    => '',
            'delivery_country' => $item->delivery_country,
            'delivery_city'    => $item->delivery_city,
            'item_price'       => (float) $item->item_price,
            'proposed_fee'     => (float) $item->proposed_fee,
            'currency'         => $item->currency ?? 'IRT',
            'status'           => $item->status,
            'created_at'       => $item->created_at,
        ]);

        return response()->json($purchases);
    }

    public function purchaseShow(string $id): JsonResponse
    {
        try {
            $item = OnlinePurchaseRequest::with(['user:id,name', 'images'])->findOrFail($id);

            return response()->json(['data' => [
                'id'               => $item->id,
                'type'             => 'purchase',
                'title'            => $item->title,
                'link'             => $item->link,
                'description'      => $item->description,
                'user'             => ['name' => $item->user?->name ?? 'نامشخص','phone' => $item->user?->phone ?? 'نامشخص'],
                'purchaseLocation' => ['country' => 'اینترنت', 'city' => ''],
                'deliveryLocation' => [
                    'country' => $item->delivery_country,
                    'city'    => $item->delivery_city,
                    'address' => $item->delivery_address,
                    'lat'     => $item->delivery_lat,
                    'lng'     => $item->delivery_lng,
                ],
                'itemPrice'        => (float) $item->item_price,
                'proposedFee'      => (float) $item->proposed_fee,
                'currency'         => $item->currency ?? 'IRT',
                'showPhone'        => $item->show_phone,
                'images'           => $item->images->map(fn($img) => [
                    'id' => $img->id,
                    'url' => $img->path // فقط مسیر نسبی: uploads/online-purchase-requests/...
                ]),
                'status'           => $item->status,
                'rejection_reason' => $item->rejection_reason,
                'created_at'       => $item->created_at,
                'updated_at'       => $item->updated_at,
            ]]);
        } catch (\Exception $e) {
            \Log::error('Error in purchaseShow: ' . $e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    public function purchaseApprove(string $id): JsonResponse
    {
        $item = OnlinePurchaseRequest::findOrFail($id);
        $item->update(['status' => 'approved']);
        return response()->json(['message' => 'درخواست خرید با موفقیت تأیید شد']);
    }

    public function purchaseReject(Request $request, string $id): JsonResponse
    {
        $request->validate(['reason' => 'required|string|max:500']);
        $item = OnlinePurchaseRequest::findOrFail($id);
        $item->update(['status' => 'rejected', 'rejection_reason' => $request->reason]);
        return response()->json(['message' => 'درخواست خرید با موفقیت رد شد']);
    }

    // ========== ثبت‌نام خریداران ==========

    public function shopperIndex(Request $request): JsonResponse
    {
        $query = OnlinePurchaseShopper::with('user:id,name,phone');

        if ($request->has('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        $shoppers = $query->get()->map(fn($item) => [
            'id'             => $item->id,
            'service_title'  => $item->service_title,
            'user'             => ['name' => $item->user?->name ?? 'نامشخص','phone' => $item->user?->phone ?? 'نامشخص'],
            'categories'     => $item->categories ?? [],
            'supportedSites' => $item->supported_sites ?? [],
            'status'         => $item->status,
            'created_at'     => $item->created_at,
        ]);

        return response()->json($shoppers);
    }

    public function shopperShow(string $id): JsonResponse
    {
        try {
            $item = OnlinePurchaseShopper::with(['user:id,name,phone', 'images'])->findOrFail($id);

            return response()->json(['data' => [
                'id'               => $item->id,
                'type'             => 'shopper',
                'title'            => $item->service_title,
                'user'             => ['name' => $item->user?->name ?? 'نامشخص','phone' => $item->user?->phone ?? 'نامشخص'],
                'location'         => [
                    'country' => $item->country,
                    'city'    => $item->city
                ],
                'serviceTitle'     => $item->service_title,
                'categories'       => $item->categories ?? [],
                'supportedSites'   => $item->supported_sites ?? [],
                'feeDescription'   => $item->fee_description,
                'capabilities'     => $item->capabilities,
                'showPhone'        => $item->show_phone,
                'images'           => $item->images->map(fn($img) => [
                    'id' => $img->id,
                    'url' => $img->path // فقط مسیر نسبی: uploads/online-purchase-shoppers/...
                ]),
                'status'           => $item->status,
                'rejection_reason' => $item->rejection_reason,
                'created_at'       => $item->created_at,
                'updated_at'       => $item->updated_at,
            ]]);
        } catch (\Exception $e) {
            \Log::error('Error in shopperShow: ' . $e->getMessage());
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    public function shopperApprove(string $id): JsonResponse
    {
        $item = OnlinePurchaseShopper::findOrFail($id);
        $item->update(['status' => 'approved']);
        return response()->json(['message' => 'ثبت‌نام خریدار با موفقیت تأیید شد']);
    }

    public function shopperReject(Request $request, string $id): JsonResponse
    {
        $request->validate(['reason' => 'required|string|max:500']);
        $item = OnlinePurchaseShopper::findOrFail($id);
        $item->update(['status' => 'rejected', 'rejection_reason' => $request->reason]);
        return response()->json(['message' => 'ثبت‌نام خریدار با موفقیت رد شد']);
    }
}
