<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Admin;
use App\Models\Shipment;
use Illuminate\Http\Response;
use Illuminate\Http\Request;

class AdminShipmentController extends Controller
{
    private function safeCoordinates($coords, $defaultLat = 35.6892, $defaultLng = 51.3890)
    {
        if (!$coords || !is_string($coords)) {
            return [$defaultLat, $defaultLng];
        }

        $parts = explode(',', $coords);
        $lat = isset($parts[0]) ? (float) trim($parts[0]) : $defaultLat;
        $lng = isset($parts[1]) ? (float) trim($parts[1]) : $defaultLng;

        return [$lat, $lng];
    }

    /**
     * لیست درخواست‌ها (با فیلتر status و type)
     */
    public function index(Request $request)
    {
        $admin = auth()->user();
        if (!$admin instanceof Admin) {
            return response()->json(['error' => 'Unauthorized'], Response::HTTP_FORBIDDEN);
        }

        $query = Shipment::with(['user', 'images']);

        // ✅ اصلاح: فقط اگر status ارسال شده باشد و 'all' نباشد، فیلتر اعمال شود
        if ($request->has('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        // فیلتر type
        if ($request->has('type') && in_array($request->type, ['send', 'carry'])) {
            $query->where('type', $request->type);
        }

        // جستجو
        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                    ->orWhere('description', 'like', "%{$search}%")
                    ->orWhere('trip_description', 'like', "%{$search}%")
                    ->orWhere('origin_city', 'like', "%{$search}%")
                    ->orWhere('destination_city', 'like', "%{$search}%");
            });
        }

        $shipments = $query->orderBy('created_at', 'desc')->get()->map(function ($shipment) {
            return [
                'id' => $shipment->id,
                'type' => $shipment->type,
                'title' => $this->getTitle($shipment),
                'user' => [
                    'id' => $shipment->user->id ?? null,
                    'name' => $shipment->user->name ?? 'نامشخص',
                    'phone' => $shipment->user->phone ?? 'نامشخص',
                    'avatar' => $shipment->user->name
                        ? 'https://ui-avatars.com/api/?name=' . urlencode($shipment->user->name) . '&background=random'
                        : 'https://ui-avatars.com/api/?name=نامشخص&background=gray'
                ],
                'origin' => [
                    'country' => $shipment->origin_country ?? 'نامشخص',
                    'city' => $shipment->origin_city ?? 'نامشخص'
                ],
                'destination' => [
                    'country' => $shipment->destination_country ?? 'نامشخص',
                    'city' => $shipment->destination_city ?? 'نامشخص'
                ],
                'weight' => $shipment->weight,
                'carry_weight_capacity' => $shipment->carry_weight_capacity,
                'fee' => (float) ($shipment->amount ?? 0),
                'currency' => $shipment->currency ?? 'IRT',
                'price_type' => $shipment->price_type,
                'status' => $shipment->status,
                'created_at' => $shipment->created_at->format('Y-m-d H:i'),
                'description' => $shipment->description ?? $shipment->trip_description,
                'has_images' => $shipment->images->count() > 0,
            ];
        });

        return response()->json([
            'success' => true,
            'data' => $shipments
        ]);
    }

    /**
     * جزئیات کامل یک درخواست
     */
    public function show(string $id)
    {
        $admin = auth()->user();
        if (!$admin instanceof Admin) {
            return response()->json(['error' => 'Unauthorized'], Response::HTTP_FORBIDDEN);
        }

        $shipment = Shipment::with(['user', 'images'])->find($id);
        if (!$shipment) {
            return response()->json(['error' => 'مرسوله یافت نشد'], Response::HTTP_NOT_FOUND);
        }

        [$originLat, $originLng] = $this->safeCoordinates($shipment->origin_coordinates);
        [$destLat, $destLng] = $this->safeCoordinates($shipment->destination_coordinates);

        // تصاویر واقعی از دیتابیس
        $images = $shipment->images->map(function ($img) {
            return [
                'id' => $img->id,
                'url' => $img->url ?? asset($img->image_path),
                'path' => $img->image_path
            ];
        })->toArray();

        // اگر تصویری نداره، یک تصویر پیش‌فرض
        if (empty($images)) {
            $images = [[
                'id' => null,
                'url' => 'https://via.placeholder.com/800x600?text=No+Image',
                'path' => null
            ]];
        }

        $data = [
            'id' => $shipment->id,
            'type' => $shipment->type,
            'title' => $this->getTitle($shipment),
            'description' => $this->getDescription($shipment),
            'status' => $shipment->status,
            'rejection_reason' => $shipment->rejection_reason, // ✅ نمایش علت رد
            'created_at' => $shipment->created_at->format('Y-m-d H:i'),
            'updated_at' => $shipment->updated_at->format('Y-m-d H:i'),
            'origin' => [
                'country' => $shipment->origin_country ?? 'نامشخص',
                'city' => $shipment->origin_city ?? 'نامشخص',
                'address' => $shipment->origin_address ?? '',
                'coordinates' => $shipment->origin_coordinates,
                'lat' => $originLat,
                'lng' => $originLng,
            ],
            'destination' => [
                'country' => $shipment->destination_country ?? 'نامشخص',
                'city' => $shipment->destination_city ?? 'نامشخص',
                'address' => $shipment->destination_address ?? '',
                'coordinates' => $shipment->destination_coordinates,
                'lat' => $destLat,
                'lng' => $destLng,
            ],
            'user' => [
                'id' => $shipment->user->id ?? null,
                'name' => $shipment->user->name ?? 'نامشخص',
                'email' => $shipment->user->email ?? null,
                'phone' => $shipment->user->phone ?? null,
                'show_phone' => $shipment->show_phone,
                'avatar' => $shipment->user->name
                    ? 'https://ui-avatars.com/api/?name=' . urlencode($shipment->user->name) . '&background=random'
                    : 'https://ui-avatars.com/api/?name=نامشخص&background=gray',
                'rating' => 4.5,
            ],
            'price_type' => $shipment->price_type,
            'fee' => (float) ($shipment->amount ?? 0),
            'currency' => $shipment->currency ?? 'IRT',
            'images' => $images,
        ];

        // فیلدهای مخصوص send
        if ($shipment->type === 'send') {
            $data['category'] = $shipment->category ?? 'نامشخص';
            $data['weight'] = $shipment->weight;
        }

        // فیلدهای مخصوص carry
        if ($shipment->type === 'carry') {
            $data['carry_weight_capacity'] = $shipment->carry_weight_capacity;
            $data['departure_date'] = $shipment->departure_date ? $shipment->departure_date->format('Y-m-d') : null;
            $data['arrival_date'] = $shipment->arrival_date ? $shipment->arrival_date->format('Y-m-d') : null;
        }

        return response()->json([
            'success' => true,
            'data' => $data
        ]);
    }

    /**
     * تایید درخواست
     */
    public function approve(string $id)
    {
        $admin = auth()->user();
        if (!$admin instanceof Admin) {
            return response()->json(['error' => 'Unauthorized'], Response::HTTP_FORBIDDEN);
        }

        $shipment = Shipment::find($id);
        if (!$shipment) {
            return response()->json(['error' => 'درخواست یافت نشد'], Response::HTTP_NOT_FOUND);
        }

        if ($shipment->status === 'approved') {
            return response()->json([
                'success' => false,
                'message' => 'این درخواست قبلاً تایید شده است'
            ], 400);
        }

        $shipment->update(['status' => 'approved']);
        
        return response()->json([
            'success' => true,
            'message' => 'درخواست با موفقیت تأیید شد'
        ]);
    }

    /**
     * رد کردن درخواست
     */
    public function reject(Request $request, string $id)
    {
        $admin = auth()->user();
        if (!$admin instanceof Admin) {
            return response()->json(['error' => 'Unauthorized'], Response::HTTP_FORBIDDEN);
        }

        $shipment = Shipment::find($id);
        if (!$shipment) {
            return response()->json(['error' => 'درخواست یافت نشد'], Response::HTTP_NOT_FOUND);
        }

        if ($shipment->status === 'rejected') {
            return response()->json([
                'success' => false,
                'message' => 'این درخواست قبلاً رد شده است'
            ], 400);
        }

        $shipment->update([
            'status' => 'rejected',
            'rejection_reason' => $request->input('reason')
        ]);
        
        return response()->json([
            'success' => true,
            'message' => 'درخواست با موفقیت رد شد'
        ]);
    }

    /**
     * متد کمکی برای عنوان
     */
    private function getTitle($shipment)
    {
        if ($shipment->type === 'send') {
            return $shipment->title ?? 'بدون عنوان';
        } else {
            return 'سفر از ' . ($shipment->origin_city ?? 'نامشخص') . ' به ' . ($shipment->destination_city ?? 'نامشخص');
        }
    }

    /**
     * متد کمکی برای توضیحات
     */
    private function getDescription($shipment)
    {
        if ($shipment->type === 'send') {
            return $shipment->description ?? '';
        } else {
            return $shipment->trip_description ?? '';
        }
    }
}