<?php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Ticket;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class AdminTicketController extends Controller
{
    /**
     * لیست تمام تیکت‌ها
     */
    public function index(Request $request)
    {
        $query = Ticket::with(['user', 'replies.user', 'attachments']);

        // فیلتر وضعیت
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // فیلتر دپارتمان
        if ($request->has('department')) {
            $query->where('department', $request->department);
        }

        // فیلتر اولویت
        if ($request->has('priority')) {
            $query->where('priority', $request->priority);
        }

        // جستجو
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                    ->orWhere('ticket_number', 'like', "%{$search}%")
                    ->orWhereHas('user', function ($q) use ($search) {
                        $q->where('name', 'like', "%{$search}%");
                    });
            });
        }

        // مرتب‌سازی
        $sortBy    = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        $tickets = $query->paginate($request->get('per_page', 15));

        // تبدیل داده‌ها
        $tickets->getCollection()->transform(function ($ticket) {
            return [
                'id'                   => $ticket->id,
                'ticket_number'        => $ticket->ticket_number,
                'title'                => $ticket->title,
                'status'               => $ticket->status,
                'priority'             => $ticket->priority,
                'department'           => $ticket->department,
                'user'                 => [
                    'id'    => $ticket->user->id,
                    'name'  => $ticket->user->name,
                    'email' => $ticket->user->email,
                ],
                'replies_count'        => $ticket->replies->count(),
                'unread_replies_count' => $ticket->replies()->whereNull('read_at')->where('is_admin', false)->count(),
                'last_reply_at'        => $ticket->replies->last()?->created_at,
                'created_at'           => $ticket->created_at,
                'updated_at'           => $ticket->updated_at,
            ];
        });

        return response()->json([
            'success' => true,
            'data'    => $tickets->items(),
            'meta'    => [
                'current_page' => $tickets->currentPage(),
                'last_page'    => $tickets->lastPage(),
                'per_page'     => $tickets->perPage(),
                'total'        => $tickets->total(),
            ],
        ]);
    }

    /**
     * نمایش جزئیات یک تیکت
     */
    public function show($id)
    {
        try {
            $ticket = Ticket::with([
                'user',
                'attachments',
                'replies.user',
                'replies.attachments',
            ])->findOrFail($id);

            return response()->json([
                'success' => true,
                'data'    => $ticket,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'تیکت مورد نظر یافت نشد',
            ], 404);
        }
    }

    /**
     * پاسخ ادمین به تیکت
     */
    public function reply(Request $request, $id)
    {
        $validator = \Validator::make($request->all(), [
            'message'       => 'required_without:attachments|string',
            'attachments'   => 'array|max:5',
            'attachments.*' => 'file|mimes:jpg,jpeg,png,pdf,doc,docx|max:10240',
        ], [
            'message.required_without' => 'متن پیام یا فایل پیوست الزامی است',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در اعتبارسنجی',
                'errors'  => $validator->errors(),
            ], 422);
        }

        try {
            $ticket = Ticket::findOrFail($id);

            \DB::beginTransaction();

            // ایجاد پاسخ ادمین
            $reply = \App\Models\TicketReply::create([
                'ticket_id' => $ticket->id,
                'user_id'   => auth()->id(), // ادمین
                'message'   => $request->message,
                'is_admin'  => true, // مهم: این پاسخ از طرف ادمین است
            ]);

            // آپلود فایل‌های پیوست
            if ($request->hasFile('attachments')) {
                $uploadPath = public_path('uploads/tickets');
                if (! file_exists($uploadPath)) {
                    mkdir($uploadPath, 0755, true);
                }

                foreach ($request->file('attachments') as $file) {
                    $originalName = $file->getClientOriginalName();
                    $fileSize     = $file->getSize();
                    $mimeType     = $file->getMimeType();
                    $extension    = $file->getClientOriginalExtension();

                    $fileName = time() . '_' . uniqid() . '.' . $extension;
                    $file->move($uploadPath, $fileName);
                    $relativePath = 'uploads/tickets/' . $fileName;

                    \App\Models\TicketAttachment::create([
                        'ticket_id'       => $ticket->id,
                        'ticket_reply_id' => $reply->id,
                        'file_path'       => $relativePath,
                        'file_name'       => $originalName,
                        'file_size'       => $fileSize,
                        'mime_type'       => $mimeType,
                    ]);
                }
            }

            // آپدیت وضعیت تیکت
            $ticket->update(['status' => 'in_progress']);

            \DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'پاسخ با موفقیت ارسال شد',
                'data'    => $reply->load(['attachments', 'user']),
            ]);

        } catch (\Exception $e) {
            \DB::rollBack();
            Log::error('Admin reply error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'خطا در ارسال پاسخ',
                'error'   => config('app.debug') ? $e->getMessage() : 'خطای سرور',
            ], 500);
        }
    }

    /**
     * تغییر وضعیت تیکت
     */
    public function updateStatus(Request $request, $id)
    {
        $validator = \Validator::make($request->all(), [
            'status' => 'required|in:open,in_progress,resolved,closed',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors'  => $validator->errors(),
            ], 422);
        }

        try {
            $ticket = Ticket::findOrFail($id);
            $ticket->update(['status' => $request->status]);

            return response()->json([
                'success' => true,
                'message' => 'وضعیت تیکت با موفقیت تغییر کرد',
                'data'    => $ticket,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در تغییر وضعیت',
            ], 500);
        }
    }

    /**
     * آمار تیکت‌ها
     */
    public function stats()
    {
        try {
            $stats = [
                'total'         => Ticket::count(),
                'open'          => Ticket::where('status', 'open')->count(),
                'in_progress'   => Ticket::where('status', 'in_progress')->count(),
                'resolved'      => Ticket::where('status', 'resolved')->count(),
                'closed'        => Ticket::where('status', 'closed')->count(),

                'by_department' => [
                    'technical'   => Ticket::where('department', 'technical')->count(),
                    'financial'   => Ticket::where('department', 'financial')->count(),
                    'suggestions' => Ticket::where('department', 'suggestions')->count(),
                ],

                'by_priority'   => [
                    'low'    => Ticket::where('priority', 'low')->count(),
                    'medium' => Ticket::where('priority', 'medium')->count(),
                    'high'   => Ticket::where('priority', 'high')->count(),
                ],
            ];

            return response()->json([
                'success' => true,
                'data'    => $stats,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت آمار',
            ], 500);
        }
    }
}
