<?php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\TripCompanion;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class AdminTripCompanionController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $query = TripCompanion::with('user:id,name,phone');

        if ($request->has('status') && in_array($request->status, ['pending', 'approved', 'rejected'])) {
            $query->where('status', $request->status);
        }

        $companions = $query->get()->map(function ($item) {
            return [
                'id'                    => $item->id,
                'type'                  => 'companion',
                'user' => [
                    'name' => $item->user?->name ?? 'نامشخص',
                    'phone' => $item->user?->phone ?? 'نامشخص',
                ],
                'title'                 => $item->description ?? 'درخواست همسفری',
                'origin'                => [
                    'country' => $item->origin_country ?? 'نامشخص',
                    'city'    => $item->origin_city ?? 'نامشخص',
                ],
                'destination'           => [
                    'country' => $item->destination_country ?? 'نامشخص',
                    'city'    => $item->destination_city ?? 'نامشخص',
                ],
                'vehicle_types_label'   => $item->vehicle_types,
                'available_from_shamsi' => $item->available_from ? \Carbon\Carbon::parse($item->available_from)->format('Y-m-d') : null,
                'available_to_shamsi'   => $item->available_to ? \Carbon\Carbon::parse($item->available_to)->format('Y-m-d') : null,
                'capacity'              => $item->capacity,
                'status'                => $item->status ?? 'pending',
                'rejection_reason'      => $item->rejection_reason,
                'created_at'     => $item->created_at ?? 'نامشخص',
                'updated_at'     => $item->updated_at ?? 'نامشخص',
            ];
        });

        return response()->json([
            'success' => true,
            'data'    => $companions,
        ]);
    }

    public function show(string $id): JsonResponse
    {
        $item = TripCompanion::with('user:id,name')->findOrFail($id);

        return response()->json([
            'success' => true,
            'data'    => [
                'id'                    => $item->id,
                'type'                  => 'companion',
                'user' => [
                    'name' => $item->user?->name ?? 'نامشخص',
                    'phone' => $item->user?->phone ?? 'نامشخص',
                ],
                'title'                 => $item->description ?? 'درخواست همسفری',
                'origin'                => [
                    'country'     => $item->origin_country ?? 'نامشخص',
                    'city'        => $item->origin_city ?? 'نامشخص',
                    'address'     => $item->origin_address ?? '',
                    'coordinates' => $item->origin_coordinates,
                ],
                'destination'           => [
                    'country'     => $item->destination_country ?? 'نامشخص',
                    'city'        => $item->destination_city ?? 'نامشخص',
                    'address'     => $item->destination_address ?? '',
                    'coordinates' => $item->destination_coordinates,
                ],
                'vehicle_types_label'   => $item->vehicle_types,
                'available_from_shamsi' => $item->available_from ? \Carbon\Carbon::parse($item->available_from)->format('Y-m-d') : null,
                'available_to_shamsi'   => $item->available_to ? \Carbon\Carbon::parse($item->available_to)->format('Y-m-d') : null,
                'capacity'              => $item->capacity,
                'status'                => $item->status ?? 'pending',
                'rejection_reason'      => $item->rejection_reason,
                'created_at'     => $item->created_at ?? 'نامشخص',
                'updated_at'     => $item->updated_at ?? 'نامشخص',
            ],
        ]);
    }

    public function approve(string $id): JsonResponse
    {
        $item = TripCompanion::findOrFail($id);
        $item->update(['status' => 'approved']);
        return response()->json(['success' => true, 'message' => 'درخواست همسفری با موفقیت تأیید شد']);
    }

    public function reject(Request $request, string $id): JsonResponse
    {
        $request->validate(['reason' => 'required|string|max:500']);
        $item = TripCompanion::findOrFail($id);
        $item->update(['status' => 'rejected', 'rejection_reason' => $request->reason]);
        return response()->json(['success' => true, 'message' => 'درخواست همسفری با موفقیت رد شد']);
    }
}
