<?php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Trip;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class AdminTripController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $query = Trip::with('user:id,name,phone');

        if ($request->has('status') && in_array($request->status, ['pending', 'approved', 'rejected'])) {
            $query->where('status', $request->status);
        }

        $trips = $query->get()->map(function ($trip) {
            return [
                'id'                    => $trip->id,
                'type'                  => 'trip',
                'user'                  => [
                    'name'  => $trip->user?->name ?? 'نامشخص',
                    'phone' => $trip->user?->phone ?? 'نامشخص',
                ],
                'title'                 => $trip->description ?? 'درخواست سفر',
                'origin'                => [
                    'country' => $trip->origin_country ?? 'نامشخص',
                    'city'    => $trip->origin_city ?? 'نامشخص',
                ],
                'destination'           => [
                    'country' => $trip->destination_country ?? 'نامشخص',
                    'city'    => $trip->destination_city ?? 'نامشخص',
                ],
                'vehicle_type_label'    => $trip->vehicle_type,
                'departure_date_shamsi' => $trip->departure_date,
                'arrival_date_shamsi'   => $trip->arrival_date,
                'capacity'              => $trip->capacity,
                'status'                => $trip->status,
                'rejection_reason'      => $trip->rejection_reason,
                'created_at'     => $trip->created_at ?? 'نامشخص',
                'updated_at'     => $trip->updated_at ?? 'نامشخص',
            ];
        });

        return response()->json([
            'success' => true,
            'data'    => $trips,
        ]);
    }

    public function show(string $id): JsonResponse
    {
        $trip = Trip::with('user:id,name')->findOrFail($id);

        return response()->json([
            'success' => true,
            'data'    => [
                'id'                    => $trip->id,
                'type'                  => 'trip',
                'user'                  => [
                    'name'  => $trip->user?->name ?? 'نامشخص',
                    'phone' => $trip->user?->phone ?? 'نامشخص',
                ],
                'title'                 => $trip->description ?? 'درخواست سفر',
                'origin'                => [
                    'country'     => $trip->origin_country ?? 'نامشخص',
                    'city'        => $trip->origin_city ?? 'نامشخص',
                    'address'     => $trip->origin_address ?? '',
                    'coordinates' => $trip->origin_coordinates,
                ],
                'destination'           => [
                    'country'     => $trip->destination_country ?? 'نامشخص',
                    'city'        => $trip->destination_city ?? 'نامشخص',
                    'address'     => $trip->destination_address ?? '',
                    'coordinates' => $trip->destination_coordinates,
                ],
                'vehicle_type_label'    => $trip->vehicle_type,
                'departure_date_shamsi' => $trip->departure_date,
                'arrival_date_shamsi'   => $trip->arrival_date,
                'capacity'              => $trip->capacity,
                'status'                => $trip->status,
                'rejection_reason'      => $trip->rejection_reason,
                'created_at'     => $trip->created_at ?? 'نامشخص',
                'updated_at'     => $trip->updated_at ?? 'نامشخص',
            ],
        ]);
    }

    public function approve(string $id): JsonResponse
    {
        $trip = Trip::findOrFail($id);
        $trip->update(['status' => 'approved']);
        return response()->json(['success' => true, 'message' => 'درخواست سفر تأیید شد']);
    }

    public function reject(Request $request, string $id): JsonResponse
    {
        $request->validate(['reason' => 'required|string|max:500']);
        $trip = Trip::findOrFail($id);
        $trip->update(['status' => 'rejected', 'rejection_reason' => $request->reason]);
        return response()->json(['success' => true, 'message' => 'درخواست سفر رد شد']);
    }
}
