<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Resources\ArticleCategoryResource;
use App\Models\ArticleCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class ArticleCategoryController extends Controller
{
    public function index()
    {
        $categories = ArticleCategory::latest()->get();
        return ArticleCategoryResource::collection($categories);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:article_categories',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        $image = $request->file('image');
        $filename = time() . '_' . uniqid() . '.' . $image->extension();
        $image->move(public_path('uploads/acategories'), $filename);

        $category = ArticleCategory::create([
            'name' => $request->name,
            'image_url' => 'uploads/acategories/' . $filename,
        ]);

        return new ArticleCategoryResource($category);
    }

    public function update(Request $request, $id)
    {
        $category = ArticleCategory::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:article_categories,name,' . $id,
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        // حذف تصویر قبلی (اگر آپلود جدید داشته باشیم)
        if ($request->hasFile('image')) {
            if ($category->image_url && file_exists(public_path($category->image_time))) {
                unlink(public_path($category->image_url));
            }

            $image = $request->file('image');
            $filename = time() . '_' . uniqid() . '.' . $image->extension();
            $image->move(public_path('uploads/acategories'), $filename);

            $category->image_url = 'uploads/acategories/' . $filename;
        }

        $category->name = $request->name;
        $category->save();

        return new ArticleCategoryResource($category);
    }

    public function destroy($id)
    {
        $category = ArticleCategory::findOrFail($id);

        if ($category->image_url && file_exists(public_path($category->image_url))) {
            unlink(public_path($category->image_url));
        }

        $category->delete();

        return response()->noContent();
    }
}