<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Resources\ArticleResource;
use App\Models\Article;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ArticleController extends Controller
{
    public function index(Request $request)
    {
        $query = Article::with('category');
        
        if ($search = $request->get('search')) {
            $query->where('title', 'like', "%{$search}%");
        }
        
        if ($status = $request->get('status')) {
            $query->where('status', $status);
        }
        
        if ($categoryId = $request->get('category_id')) {
            $query->where('category_id', $categoryId);
        }
        
        $articles = $query->latest()->paginate(20);
        
        return ArticleResource::collection($articles);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'status' => 'required|in:draft,published',
            'category_id' => 'required|exists:article_categories,id',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $imagePath = null;
        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $filename = time() . '_' . uniqid() . '.' . $file->extension();
            $file->move(public_path('articles'), $filename);
            $imagePath = 'articles/' . $filename;
        }

        $article = Article::create([
            'title' => $request->title,
            'content' => $request->content,
            'status' => $request->status,
            'category_id' => $request->category_id,
            'image_url' => $imagePath,
        ]);

        return new ArticleResource($article->load('category'));
    }

    public function show($id)
    {
        $article = Article::with('category')->findOrFail($id);
        return new ArticleResource($article);
    }

    public function update(Request $request, $id)
    {
        $article = Article::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'status' => 'required|in:draft,published',
            'category_id' => 'required|exists:article_categories,id',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $data = [
            'title' => $request->title,
            'content' => $request->content,
            'status' => $request->status,
            'category_id' => $request->category_id,
        ];

        if ($request->hasFile('image')) {
            if ($article->image_url && file_exists(public_path($article->image_url))) {
                unlink(public_path($article->image_url));
            }

            $file = $request->file('image');
            $filename = time() . '_' . uniqid() . '.' . $file->extension();
            $file->move(public_path('articles'), $filename);
            $data['image_url'] = 'articles/' . $filename;
        }

        $article->update($data);

        return new ArticleResource($article->load('category'));
    }

    public function destroy($id)
    {
        $article = Article::findOrFail($id);

        if ($article->image_url && file_exists(public_path($article->image_url))) {
            unlink(public_path($article->image_url));
        }

        $article->delete();
        
        return response()->noContent();
    }
}
