<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class TicketReply extends Model
{
    use HasFactory;

    protected $fillable = [
        'ticket_id',
        'user_id',
        'message',
        'is_admin',
        'read_at', // اضافه شد
    ];

    protected $casts = [
        'is_admin' => 'boolean',
        'read_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Get the ticket that owns the reply.
     */
    public function ticket()
    {
        return $this->belongsTo(Ticket::class);
    }

    /**
     * Get the user who created the reply.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the attachments for the reply.
     */
    public function attachments()
    {
        return $this->hasMany(TicketAttachment::class);
    }

    /**
     * Scope a query to only include admin replies.
     */
    public function scopeAdminReplies($query)
    {
        return $query->where('is_admin', true);
    }

    /**
     * Scope a query to only include user replies.
     */
    public function scopeUserReplies($query)
    {
        return $query->where('is_admin', false);
    }

    /**
     * Scope a query to filter by read status for a specific user.
     */
    public function scopeReadBy($query, $userId = null)
    {
        if ($userId === null) {
            return $query->whereNotNull('read_at');
        }
        
        // می‌توانید منطق پیچیده‌تری اضافه کنید
        return $query->whereNotNull('read_at');
    }

    /**
     * Mark reply as read.
     */
    public function markAsRead()
    {
        $this->update(['read_at' => now()]);
    }

    /**
     * Check if reply has been read.
     */
    public function isRead()
    {
        return !is_null($this->read_at);
    }
}
