// src/App.tsx
import React, { useState, useEffect } from 'react';
import Sidebar from './components/Sidebar';
import Header from './components/Header';
import AdminDashboardPage from './components/pages/DashboardPage';
import { API_BASE_URL } from '@/src/config.js';
// ⚠️ توجه: این فایل در واقع صفحه "مدیریت کاربران" است
import UserListPage from './components/pages/MyPurchaseRequestsPage';

import AdminShipmentRequestsPage from './components/pages/AcceptShipmentPage';
import AdminTripRequestsPage from './components/pages/AcceptCompanionPage';
import AdminPurchaseRequestsPage from './components/pages/AcceptOnlinePurchasePage';
import AllRequestsListPage from './components/pages/AllRequestsListPage';
import AdminTicketListPage from './components/pages/TicketsPage';
import AdminTicketDetailPage from './components/pages/TicketDetailPage';
import AccountPage from './components/pages/AccountPage';
import LoginPage from './components/auth/LoginPage';
import VerificationPage from './components/auth/VerificationPage';
import ShipmentDetailPage from './components/pages/ShipmentDetailPage';
import TripDetailPage from './components/pages/TripDetailPage';
import PurchaseDetailPage from './components/pages/PurchaseDetailPage';
import ArticlesListPage from './components/pages/ArticlesListPage';
import ArticleCategoriesPage from './components/pages/ArticleCategoriesPage';
import AddOrEditArticlePage from './components/pages/AddOrEditArticlePage';
import AllRequestsDetailPage from './components/pages/AllRequestsDetailPage';

const App: React.FC = () => {
  const [isSidebarOpen, setSidebarOpen] = useState(false);
  const [theme, setTheme] = useState('dark');
  const [currentPage, setCurrentPage] = useState('dashboard');
  const [selectedTicketId, setSelectedTicketId] = useState<string | null>(null);
  const [selectedShipmentId, setSelectedShipmentId] = useState<string | null>(null);
  const [selectedTripId, setSelectedTripId] = useState<string | null>(null);
  const [selectedTripType, setSelectedTripType] = useState<'trip' | 'companion' | null>(null);
  const [selectedPurchaseId, setSelectedPurchaseId] = useState<string | null>(null);
  const [selectedPurchaseType, setSelectedPurchaseType] = useState<'purchase' | 'shopper' | null>(null);
  const [selectedArticleId, setSelectedArticleId] = useState<string | null>(null);
  const [selectedAllRequestId, setSelectedAllRequestId] = useState<number | null>(null);
  const [selectedAllRequestType, setSelectedAllRequestType] = useState<'shipment' | 'trip' | 'purchase' | 'shopper' | 'ticket' | null>(null);

  const [isAuthenticated, setIsAuthenticated] = useState(false);
  const [loginStep, setLoginStep] = useState<'phone' | 'code'>('phone');

  const [message, setMessage] = useState<{ text: string; type: 'error' | 'success' } | null>(null);

  const showError = (text: string) => {
    setMessage({ text, type: 'error' });
    setTimeout(() => setMessage(null), 5000);
  };

  const showSuccess = (text: string) => {
    setMessage({ text, type: 'success' });
    setTimeout(() => setMessage(null), 5000);
  };

  const checkAuthStatus = async () => {
    const token = localStorage.getItem('auth_token');
    if (!token) {
      setIsAuthenticated(false);
      setLoginStep('phone');
      return;
    }

    try {
      const response = await fetch(`${API_BASE_URL}/admin/auth/check`, {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json',
        },
      });

      if (response.ok) {
        setIsAuthenticated(true);
      } else {
        throw new Error('توکن شما منقضی شده است.');
      }
    } catch (error) {
      localStorage.removeItem('auth_token');
      localStorage.removeItem('auth_phone');
      setIsAuthenticated(false);
      setLoginStep('phone');
      showError('لطفاً دوباره وارد شوید.');
    }
  };

  useEffect(() => {
    checkAuthStatus();
  }, []);

  useEffect(() => {
    const savedTheme = localStorage.getItem('theme');
    if (savedTheme && (savedTheme === 'light' || savedTheme === 'dark')) {
      setTheme(savedTheme);
    } else {
      localStorage.setItem('theme', 'dark');
    }
  }, []);

  useEffect(() => {
    if (theme === 'dark') {
      document.documentElement.classList.add('dark');
    } else {
      document.documentElement.classList.remove('dark');
    }
    localStorage.setItem('theme', theme);
  }, [theme]);

  const toggleTheme = () => {
    setTheme(prevTheme => (prevTheme === 'dark' ? 'light' : 'dark'));
  };

  const toggleSidebar = () => {
    setSidebarOpen(!isSidebarOpen);
  };

  const handleNavigation = async (page: string, payload?: any) => {
    setSelectedTicketId(null);
    setSelectedShipmentId(null);
    setSelectedTripId(null);
    setSelectedTripType(null);
    setSelectedPurchaseId(null);
    setSelectedPurchaseType(null);
    setSelectedArticleId(null);
    setSelectedAllRequestId(null);
    setSelectedAllRequestType(null);

    if (page === 'admin-ticket-detail' && payload) {
      setSelectedTicketId(payload);
    } else if (page === 'shipment-detail' && payload) {
      setSelectedShipmentId(payload);
    } else if (page === 'trip-detail' && payload) {
      setSelectedTripId(payload.id);
      setSelectedTripType(payload.type);
    } else if (page === 'purchase-detail' && payload) {
      setSelectedPurchaseId(payload.id);
      setSelectedPurchaseType(payload.type);
    } else if (page === 'edit-article' && payload) {
      setSelectedArticleId(payload);
    } else if (page === 'all-request-detail' && payload) {
      setSelectedAllRequestId(payload.id);
      setSelectedAllRequestType(payload.type);
    }

    if (page === 'logout') {
      localStorage.removeItem('auth_token');
      localStorage.removeItem('auth_phone');
      setIsAuthenticated(false);
      setLoginStep('phone');
      setCurrentPage('dashboard');
      return;
    }

    await checkAuthStatus();

    if (isAuthenticated) {
      setCurrentPage(page);
      setSidebarOpen(false);
    }
  };

  const renderContent = () => {
    switch (currentPage) {
      case 'dashboard':
        return <AdminDashboardPage />;
      case 'users':
        return <UserListPage
          showError={showError}
          showSuccess={showSuccess}
          onNavigate={handleNavigation}
        />;

      case 'all-requests':
        return <AllRequestsListPage onNavigate={handleNavigation}  />;

      case 'admin-shipments':
        return <AdminShipmentRequestsPage
          onNavigate={handleNavigation}
          showError={showError}
          showSuccess={showSuccess}
        />;
      case 'admin-trips':
        return <AdminTripRequestsPage
          onNavigate={handleNavigation}
          showError={showError}
          showSuccess={showSuccess}
        />;
      case 'admin-purchases':
        return <AdminPurchaseRequestsPage
          onNavigate={handleNavigation}
          showError={showError}
          showSuccess={showSuccess}
        />;
      case 'admin-tickets':
        return <AdminTicketListPage onNavigate={handleNavigation} />;
      case 'admin-ticket-detail':
        return <AdminTicketDetailPage ticketId={selectedTicketId} onNavigate={handleNavigation} />;

      // ✅ مهم: پاس دادن showError و showSuccess به صفحات مقاله
      case 'articles':
        return <ArticlesListPage onNavigate={handleNavigation} showError={showError} showSuccess={showSuccess} />;
      case 'article-categories':
        return <ArticleCategoriesPage showError={showError} showSuccess={showSuccess} />;
      case 'add-article':
        return <AddOrEditArticlePage onNavigate={handleNavigation} showError={showError} showSuccess={showSuccess} />;
      case 'edit-article':
        return <AddOrEditArticlePage
          onNavigate={handleNavigation}
          articleId={selectedArticleId}
          isEditing={true}
          showError={showError}
          showSuccess={showSuccess}
        />;

      case 'shipment-detail':
        return <ShipmentDetailPage
          shipmentId={selectedShipmentId}
          onNavigate={handleNavigation}
          isOwnerView={true}
          showError={showError}
          showSuccess={showSuccess}
        />;
      case 'trip-detail':
        return <TripDetailPage
          tripId={selectedTripId}
          tripType={selectedTripType}
          onNavigate={handleNavigation}
          showError={showError}
          showSuccess={showSuccess}
        />;
      case 'purchase-detail':
        return <PurchaseDetailPage
          purchaseId={selectedPurchaseId}
          purchaseType={selectedPurchaseType}
          onNavigate={handleNavigation}
          showError={showError}
          showSuccess={showSuccess}
        />;
      case 'all-request-detail':
        return <AllRequestsDetailPage
          requestId={selectedAllRequestId}
          requestType={selectedAllRequestType}
          onNavigate={handleNavigation}
          showError={showError}
          showSuccess={showSuccess}
        />;
      case 'account':
        return <AccountPage />;
      default:
        return <AdminDashboardPage />;
    }
  };

  const handlePhoneSubmit = async (phone: string) => {
    try {
      const response = await fetch(`${API_BASE_URL}/admin/auth/request-otp`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ phone })
      });

      const data = await response.json();

      if (response.ok) {
        setLoginStep('code');
        localStorage.setItem('auth_phone', phone);
        showSuccess('کد تایید ارسال شد');
      } else {
        showError(data.error || 'خطا در ارسال کد');
      }
    } catch (err) {
      showError('اتصال به سرور برقرار نشد');
    }
  };

  const handleCodeSubmit = async (code: string) => {
    const phone = localStorage.getItem('auth_phone');
    if (!phone) return;

    try {
      const response = await fetch(`${API_BASE_URL}/admin/auth/verify-otp`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ phone, otp: code })
      });

      const data = await response.json();

      if (response.ok) {
        localStorage.setItem('auth_token', data.token);
        setIsAuthenticated(true);
        showSuccess('ورود موفقیت‌آمیز بود');
      } else {
        showError(data.error || 'کد نامعتبر است');
      }
    } catch (err) {
      showError('اتصال به سرور برقرار نشد');
    }
  };

  const handleResendCode = async () => {
    const phone = localStorage.getItem('auth_phone');
    if (!phone) return;

    try {
      const response = await fetch(`${API_BASE_URL}/admin/auth/request-otp`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ phone })
      });

      if (response.ok) {
        showSuccess('کد جدید ارسال شد!');
      } else {
        const data = await response.json();
        showError(data.error || 'خطا در ارسال مجدد');
      }
    } catch (err) {
      showError('اتصال به سرور برقرار نشد');
    }
  };

  if (!isAuthenticated) {
    return (
      <>
        {message && (
          <div
            className={`fixed top-4 right-4 px-6 py-3 rounded-lg shadow-lg text-white font-medium z-50 ${message.type === 'success' ? 'bg-green-500' : 'bg-red-500'
              }`}
          >
            {message.text}
          </div>
        )}

        {loginStep === 'phone' && (
          <LoginPage onLogin={handlePhoneSubmit} theme={theme} toggleTheme={toggleTheme} showError={showError} />
        )}
        {loginStep === 'code' && (
          <VerificationPage onVerify={handleCodeSubmit} onResend={handleResendCode} theme={theme} toggleTheme={toggleTheme} />
        )}
      </>
    );
  }

  return (
    <div className="bg-custom-light-bg dark:bg-custom-dark-blue min-h-screen text-gray-800 dark:text-gray-200 font-sans flex relative">
      {message && (
        <div
          className={`fixed top-4 right-4 px-6 py-3 rounded-lg shadow-lg text-white font-medium z-50 ${message.type === 'success' ? 'bg-green-500' : 'bg-red-500'
            }`}
        >
          {message.text}
        </div>
      )}

      <Sidebar isOpen={isSidebarOpen} currentPage={currentPage} onNavigate={handleNavigation} />
      <div className="flex-1 flex flex-col overflow-x-hidden">
        <Header onMenuClick={toggleSidebar} theme={theme} toggleTheme={toggleTheme} onNavigate={handleNavigation} />
        <main className="flex-1 p-4 sm:p-6 lg:p-8 overflow-y-auto">
          {renderContent()}
        </main>
      </div>
      {isSidebarOpen && (
        <div
          onClick={() => setSidebarOpen(false)}
          className="bg-black/50 fixed inset-0 z-30 lg:hidden"
          aria-hidden="true"
        ></div>
      )}
    </div>
  );
};

export const authFetch = async (url: string, options: RequestInit = {}) => {
  const token = localStorage.getItem('auth_token');

  if (!token) {
    localStorage.removeItem('auth_token');
    localStorage.removeItem('auth_phone');
    window.location.reload();
    throw new Error('Not authenticated');
  }

  const response = await fetch(url, {
    ...options,
    headers: {
      'Authorization': `Bearer ${token}`,
      ...options.headers,
    },
  });

  if (response.status === 401) {
    localStorage.removeItem('auth_token');
    localStorage.removeItem('auth_phone');
    window.location.reload();
    throw new Error('Token expired or invalid');
  }

  if (!response.ok) {
    const data = await response.json().catch(() => ({}));
    throw new Error(data.error || 'خطا در انجام عملیات');
  }

  return response;
};

export default App;