// src/components/Sidebar.tsx
import React, { useState, useEffect } from 'react';
import {
    DashboardIcon, UserIcon, LogoutIcon, ChevronDownIcon, SuitcaseIcon, 
    CheckCircleIcon, TicketIcon, FileTextIcon, ClipboardListIcon, XIcon
} from './Icons';

interface SidebarProps {
    isOpen: boolean;
    currentPage: string;
    onNavigate: (page: string) => void;
    onClose?: () => void;
}

interface MenuItem {
    name: string;
    icon: React.FC<{ className?: string }>;
    id: string;
    children?: { name: string; id: string }[];
}

const menuItems: MenuItem[] = [
    { name: 'داشبورد', icon: DashboardIcon, id: 'dashboard' },
    { name: 'مدیریت کاربران', icon: UserIcon, id: 'users' },
    { name: 'لیست درخواست‌ها', icon: ClipboardListIcon, id: 'all-requests' },
    {
        name: 'بررسی درخواست‌ها',
        icon: CheckCircleIcon,
        id: 'requests',
        children: [
            { name: 'مرسولات', id: 'admin-shipments' },
            { name: 'سفرها و همسفرها', id: 'admin-trips' },
            { name: 'خریدهای اینترنتی', id: 'admin-purchases' },
        ],
    },
    { name: 'مدیریت تیکت‌ها', icon: TicketIcon, id: 'admin-tickets' },
    {
        name: 'مقالات',
        icon: FileTextIcon,
        id: 'articles-root',
        children: [
            { name: 'لیست مقالات', id: 'articles' },
            { name: 'دسته‌بندی‌ها', id: 'article-categories' },
        ],
    },
   
    { name: 'خروج', icon: LogoutIcon, id: 'logout' },
];

const Sidebar: React.FC<SidebarProps> = ({ isOpen, currentPage, onNavigate, onClose }) => {
    const [openMenus, setOpenMenus] = useState<string[]>([]);

    useEffect(() => {
        const parentMenu = menuItems.find(item => 
            item.children?.some(child => child.id === currentPage)
        );
        
        if (parentMenu && !openMenus.includes(parentMenu.id)) {
            setOpenMenus(prev => [...prev, parentMenu.id]);
        }
    }, [currentPage]);

    const handleParentClick = (id: string) => {
        setOpenMenus(prev => 
            prev.includes(id) ? prev.filter(menuId => menuId !== id) : [...prev, id]
        );
    };

    const handleChildClick = (id: string) => {
        onNavigate(id);
        if (window.innerWidth < 1024 && onClose) {
            onClose();
        }
    };

    const handleItemClick = (id: string) => {
        onNavigate(id);
        if (window.innerWidth < 1024 && onClose) {
            onClose();
        }
    };

    return (
        <>
            {/* Overlay برای موبایل */}
            {isOpen && (
                <div 
                    className="fixed inset-0 bg-black/50 z-30 lg:hidden"
                    onClick={onClose}
                />
            )}
            
            <aside 
                className={`
                    bg-custom-light-card dark:bg-custom-dark-blue-light 
                    w-64 p-6 flex flex-col h-screen 
                    fixed lg:sticky top-0 right-0 z-40 
                    transition-transform duration-300 ease-in-out 
                    ${isOpen ? 'translate-x-0' : 'translate-x-full'} 
                    lg:translate-x-0
                    overflow-y-auto
                    custom-scrollbar
                `}
                style={{
                    scrollbarWidth: 'thin',
                    scrollbarColor: 'rgb(147 51 234 / 0.3) transparent'
                }}
            >
                {/* Header */}
                <div className="flex items-center justify-between mb-10">
                    <div className="flex items-center">
                        <SuitcaseIcon className="w-10 h-10 text-custom-purple flex-shrink-0" />
                        <span className="text-2xl font-bold text-gray-800 dark:text-white mr-2">پنل ادمین</span>
                    </div>
                    
                    <button 
                        onClick={onClose}
                        className="lg:hidden p-2 rounded-lg hover:bg-gray-100 dark:hover:bg-custom-dark-blue transition-colors"
                        aria-label="بستن منو"
                    >
                        <XIcon className="w-5 h-5 text-gray-600 dark:text-gray-300" />
                    </button>
                </div>

                {/* Navigation */}
                <nav className="flex-grow overflow-y-auto">
                    <ul className="space-y-1">
                        {menuItems.map((item) => {
                            const isParentActive = item.children?.some(child => child.id === currentPage);
                            const isParentOpen = openMenus.includes(item.id);

                            if (item.children) {
                                return (
                                    <li key={item.id}>
                                        <div
                                            className={`
                                                flex items-center justify-between p-3 cursor-pointer rounded-lg 
                                                transition-colors duration-200
                                                ${isParentActive 
                                                    ? 'text-custom-purple bg-custom-purple/5' 
                                                    : 'text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-custom-dark-blue'
                                                }
                                            `}
                                            onClick={() => handleParentClick(item.id)}
                                        >
                                            <div className="flex items-center">
                                                <item.icon className="w-5 h-5 ml-3 flex-shrink-0" />
                                                <span className={`${isParentActive ? 'font-semibold' : ''}`}>
                                                    {item.name}
                                                </span>
                                            </div>
                                            <ChevronDownIcon 
                                                className={`
                                                    w-5 h-5 transition-transform duration-200 flex-shrink-0
                                                    ${isParentOpen ? 'rotate-180' : ''}
                                                `} 
                                            />
                                        </div>
                                        
                                        <div 
                                            className={`
                                                overflow-hidden transition-all duration-300 ease-in-out
                                                ${isParentOpen ? 'max-h-96 opacity-100' : 'max-h-0 opacity-0'}
                                            `}
                                        >
                                            <ul className="mr-6 mt-1 border-r-2 border-gray-200 dark:border-gray-600">
                                                {item.children.map((child) => {
                                                    const isChildActive = currentPage === child.id;
                                                    return (
                                                        <li
                                                            key={child.id}
                                                            className={`
                                                                p-2 pr-4 my-1 cursor-pointer rounded-r-lg 
                                                                transition-colors duration-200 relative
                                                                ${isChildActive 
                                                                    ? 'bg-custom-purple/10 text-custom-purple' 
                                                                    : 'text-gray-600 dark:text-gray-400 hover:bg-gray-100 dark:hover:bg-custom-dark-blue hover:text-gray-800 dark:hover:text-gray-200'
                                                                }
                                                            `}
                                                            onClick={() => handleChildClick(child.id)}
                                                        >
                                                            {isChildActive && (
                                                                <div className="absolute right-0 top-0 h-full w-1 bg-custom-purple rounded-r-full" />
                                                            )}
                                                            <span className={`${isChildActive ? 'font-bold' : ''}`}>
                                                                {child.name}
                                                            </span>
                                                        </li>
                                                    );
                                                })}
                                            </ul>
                                        </div>
                                    </li>
                                );
                            }

                            const isActive = currentPage === item.id;
                            return (
                                <li
                                    key={item.id}
                                    className={`
                                        flex items-center p-3 cursor-pointer rounded-lg 
                                        transition-colors duration-200
                                        ${isActive 
                                            ? 'bg-custom-purple text-white' 
                                            : 'text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-custom-dark-blue'
                                        }
                                    `}
                                    onClick={() => handleItemClick(item.id)}
                                >
                                    <item.icon className="w-5 h-5 ml-3 flex-shrink-0" />
                                    <span className={`flex-grow ${isActive ? 'font-semibold' : ''}`}>
                                        {item.name}
                                    </span>
                                </li>
                            );
                        })}
                    </ul>
                </nav>

                
            </aside>
            
            {/* استایل سفارشی اسکرول */}
            <style>{`
                .custom-scrollbar::-webkit-scrollbar {
                    width: 6px;
                }
                
                .custom-scrollbar::-webkit-scrollbar-track {
                    background: transparent;
                }
                
                .custom-scrollbar::-webkit-scrollbar-thumb {
                    background: rgb(147 51 234 / 0.3);
                    border-radius: 3px;
                    transition: background 0.2s;
                }
                
                .custom-scrollbar::-webkit-scrollbar-thumb:hover {
                    background: rgb(147 51 234 / 0.5);
                }
                
                /* برای حالت دارک */
                .dark .custom-scrollbar::-webkit-scrollbar-thumb {
                    background: rgb(147 51 234 / 0.4);
                }
                
                .dark .custom-scrollbar::-webkit-scrollbar-thumb:hover {
                    background: rgb(147 51 234 / 0.6);
                }
            `}</style>
        </>
    );
};

export default Sidebar;
