import React, { useState } from 'react';
import { SunIcon, MoonIcon, SuitcaseIcon } from '../Icons';
import AnimatedBackground from './AnimatedBackground';

interface LoginPageProps {
    onLogin: (phone: string) => void;
    theme: string;
    toggleTheme: () => void;
    showError: (message: string) => void; // ✅ این prop الان حتمی هست
}

const LoginPage: React.FC<LoginPageProps> = ({ onLogin, theme, toggleTheme, showError }) => {
    const [phoneNumber, setPhoneNumber] = useState('');
    const [localError, setLocalError] = useState<string | null>(null);

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        const cleanPhone = phoneNumber.trim();
        if (cleanPhone.length === 11 && cleanPhone.startsWith('09')) {
            onLogin(cleanPhone);
            setLocalError(null);
        } else {
            const errorMsg = 'لطفاً یک شماره موبایل معتبر (11 رقمی با 09) وارد کنید.';
            setLocalError(errorMsg);
            showError(errorMsg);
        }
    };

    return (
        <div className="relative min-h-screen">
            {localError && (
                <div className="fixed top-4 left-1/2 transform -translate-x-1/2 z-50 bg-red-500 text-white px-6 py-3 rounded-lg shadow-lg flex items-center animate-fade-in">
                    <span>{localError}</span>
                    <button
                        onClick={() => setLocalError(null)}
                        className="mr-3 text-white hover:text-gray-200"
                        aria-label="بستن"
                    >
                        ✕
                    </button>
                </div>
            )}

            <div className={`relative flex items-center justify-center min-h-screen font-sans text-white ${theme === 'dark' ? 'bg-custom-dark-blue' : 'bg-gray-100'} bg-[radial-gradient(#27314e_1px,transparent_1px)] [background-size:16px_16px] overflow-hidden`}>
                <AnimatedBackground />
                <div className="absolute top-6 left-6 z-20">
                    <button onClick={toggleTheme} className="p-2 bg-custom-dark-blue-light rounded-full cursor-pointer" aria-label="Toggle theme">
                        {theme === 'dark' ? (
                            <SunIcon className="h-6 w-6 text-yellow-400" />
                        ) : (
                            <MoonIcon className="h-6 w-6 text-gray-800" />
                        )}
                    </button>
                </div>
                <div className="relative z-10 w-full max-w-sm p-8 space-y-8 bg-custom-dark-blue-light/80 backdrop-blur-sm border border-white/10 rounded-2xl shadow-lg">
                    <div className="text-center">
                        <div className="flex items-center justify-center mb-6">
                            <SuitcaseIcon className="w-12 h-12 text-custom-purple" />
                            <span className="text-4xl font-bold text-white mr-3">اینوببر</span>
                        </div>
                        <h2 className="text-2xl font-bold">ورود | ثبت نام</h2>
                        <p className="mt-2 text-gray-400">لطفا شماره موبایل خود را وارد کنید</p>
                    </div>
                    <form className="space-y-6" onSubmit={handleSubmit}>
                        <input
                            id="phone"
                            name="phone"
                            type="tel"
                            autoComplete="tel"
                            required
                            className="w-full px-4 py-3 text-center tracking-[.2em] bg-custom-dark-blue border border-gray-700 rounded-lg placeholder-gray-500 focus:outline-none focus:ring-2 focus:ring-custom-purple"
                            placeholder="۰۹۱۲*******"
                            value={phoneNumber}
                            onChange={(e) => setPhoneNumber(e.target.value)}
                        />
                        <button
                            type="submit"
                            className="w-full py-3 px-4 bg-custom-purple hover:bg-custom-light-purple rounded-lg font-semibold transition-colors duration-300"
                        >
                            ورود
                        </button>
                    </form>
                    <p className="text-xs text-center text-gray-500">
                        ورود شما به معنای پذیرش قوانین سایت است
                    </p>
                </div>
            </div>
        </div>
    );
};

export default LoginPage;