import React, { useState, useRef, ChangeEvent, KeyboardEvent, ClipboardEvent } from 'react';
import { SunIcon, MoonIcon, SuitcaseIcon } from '../Icons';
import AnimatedBackground from './AnimatedBackground';

interface VerificationPageProps {
    onVerify: (code: string) => void;
    onResend: () => void;
    theme: string;
    toggleTheme: () => void;
}

const VerificationPage: React.FC<VerificationPageProps> = ({ onVerify, onResend, theme, toggleTheme }) => {
    const [otp, setOtp] = useState<string[]>(new Array(6).fill(''));
    const inputsRef = useRef<(HTMLInputElement | null)[]>([]);

    const handleChange = (e: ChangeEvent<HTMLInputElement>, index: number) => {
        const value = e.target.value;
        if (isNaN(Number(value))) return;

        const newOtp = [...otp];
        newOtp[index] = value.substring(value.length - 1);
        setOtp(newOtp);

        if (value && index < 5) {
            inputsRef.current[index + 1]?.focus();
        }
    };

    const handleKeyDown = (e: KeyboardEvent<HTMLInputElement>, index: number) => {
        if (e.key === 'Backspace' && !otp[index] && index > 0) {
            inputsRef.current[index - 1]?.focus();
        }
    };

    const handlePaste = (e: ClipboardEvent<HTMLDivElement>) => {
        e.preventDefault();
        const pasteData = e.clipboardData.getData('text');
        if (pasteData.length === 6 && /^\d+$/.test(pasteData)) {
            const newOtp = pasteData.split('');
            setOtp(newOtp);
            inputsRef.current[5]?.focus();
        }
    };
    
    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        const code = otp.join('');
        if (code.length === 6) {
            onVerify(code);
        } else {
            // چون خطا رو `App` مدیریت می‌کنه، اینجا فقط alert نمی‌دیم
            // ولی برای UX بهتر، می‌تونیم یه مکانیزم داخلی داشته باشیم
            // اما طبق درخواستت: "فقط وصل کن، گرافیکی تغییر نده"
            // پس اینجا کاری نمی‌کنیم — خطا از `handleCodeSubmit` در App میاد
            onVerify(code); // حتی اگر کوتاه باشه، ارسال می‌شه و API خطا می‌ده
        }
    };

    return (
        <div className={`relative flex items-center justify-center min-h-screen font-sans text-white ${theme === 'dark' ? 'bg-custom-dark-blue' : 'bg-gray-100'} bg-[radial-gradient(#27314e_1px,transparent_1px)] [background-size:16px_16px] overflow-hidden`}>
            <AnimatedBackground />
            <div className="absolute top-6 left-6 z-20">
                <button onClick={toggleTheme} className="p-2 bg-custom-dark-blue-light rounded-full cursor-pointer" aria-label="Toggle theme">
                     {theme === 'dark' ? (
                        <SunIcon className="h-6 w-6 text-yellow-400" />
                    ) : (
                        <MoonIcon className="h-6 w-6 text-gray-800" />
                    )}
                </button>
            </div>
             <div className="relative z-10 w-full max-w-md p-8 space-y-8 bg-custom-dark-blue-light/80 backdrop-blur-sm border border-white/10 rounded-2xl shadow-lg">
                <div className="text-center">
                    <div className="flex items-center justify-center mb-6">
                        <SuitcaseIcon className="w-12 h-12 text-custom-purple" />
                        <span className="text-4xl font-bold text-white mr-3">اینوببر</span>
                    </div>
                    <h2 className="text-2xl font-bold">کد تایید را وارد کنید</h2>
                    <p className="mt-2 text-gray-400">کد تایید برای شما ارسال گردید.</p>
                </div>

                <form onSubmit={handleSubmit}>
                    <div className="flex justify-center gap-2 mb-6" dir="ltr" onPaste={handlePaste}>
                        {otp.map((data, index) => (
                            <input
                                key={index}
                                type="text"
                                name="otp"
                                inputMode="numeric"
                                pattern="[0-9]*"
                                maxLength={1}
                                className="w-12 h-14 sm:w-14 sm:h-14 text-center text-2xl font-semibold bg-custom-dark-blue border border-gray-700 rounded-lg focus:outline-none focus:ring-2 focus:ring-custom-purple"
                                value={data}
                                onChange={(e) => handleChange(e, index)}
                                onKeyDown={(e) => handleKeyDown(e, index)}
                                ref={(el) => { inputsRef.current[index] = el; }}
                                autoFocus={index === 0}
                            />
                        ))}
                    </div>
                    <button
                        type="submit"
                        className="w-full py-3 px-4 bg-custom-purple hover:bg-custom-light-purple rounded-lg font-semibold transition-colors duration-300"
                    >
                        ادامه
                    </button>
                </form>
                <div className="text-center">
                    <button onClick={onResend} className="text-sm text-custom-light-purple hover:underline">
                        دریافت مجدد کد
                    </button>
                </div>
            </div>
        </div>
    );
};

export default VerificationPage;