// src/components/pages/AcceptCompanionPage.tsx
import React, { useState, useEffect } from 'react';
import { SearchIcon, MapPinIcon, CheckIcon, XIcon, UserIcon, PlaneIcon, ScaleIcon, CarIcon, CalendarIcon } from '../Icons';
import { API_BASE_URL } from '@/src/config.js';

interface TripRequest {
  id: number;
  type: 'trip' | 'companion';
  status: 'pending' | 'approved' | 'rejected';
  title: string;
  user: { name: string; phone: string;};
  origin: { country: string; city: string; };
  destination: { country: string; city: string; };
  vehicle_type_label?: string;
  vehicle_types_label?: string;
  capacity?: number;
  departure_date_shamsi?: string | null;
  arrival_date_shamsi?: string | null;
  available_from_shamsi?: string | null;
  available_to_shamsi?: string | null;
  created_at_shamsi: string;
  rejection_reason?: string;
}

interface AcceptCompanionPageProps {
  onNavigate: (page: string, payload?: any) => void;
  showError: (message: string) => void;
  showSuccess: (message: string) => void;
}

// Helper function to translate vehicle types
// تابع ترجمه نوع وسیله نقلیه
const translateVehicleType = (vehicle: string): string => {
  const translations: Record<string, string> = {
    'private_car': 'خودروی شخصی',
    'train': 'قطار',
    'bus': 'اتوبوس',
    'airplane': 'هواپیما',
    'ship': 'کشتی',
    'truck': 'کامیون',
    'van': 'ون',
    'motorcycle': 'موتورسیکلت'
  };
  return translations[vehicle] || vehicle;
};

// تابع پردازش و ترجمه وسایل نقلیه
const parseAndTranslateVehicles = (vehicleLabel?: string): string => {
  if (!vehicleLabel) return 'نامشخص';

  // اگر رشته JSON آرایه است (برای همراهان)
  if (vehicleLabel.startsWith('[') && vehicleLabel.endsWith(']')) {
    try {
      const vehicles = JSON.parse(vehicleLabel);
      if (Array.isArray(vehicles)) {
        return vehicles.map(v => translateVehicleType(v)).join('، ');
      }
    } catch (e) {
      console.error('Error parsing vehicle types:', e);
    }
  }

  // برای سفرها که تک‌مورد هستند
  return translateVehicleType(vehicleLabel);
};



const AcceptCompanionPage: React.FC<AcceptCompanionPageProps> = ({ onNavigate, showError, showSuccess }) => {
  const [requests, setRequests] = useState<TripRequest[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState<'all' | 'pending' | 'approved' | 'rejected'>('pending');
  const [typeFilter, setTypeFilter] = useState<'all' | 'trip' | 'companion'>('all');
  const [showRejectDialog, setShowRejectDialog] = useState(false);
  const [selectedId, setSelectedId] = useState<number | null>(null);
  const [rejectReason, setRejectReason] = useState('');
  const [isProcessing, setIsProcessing] = useState(false);

  const fetchRequests = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem('auth_token');
      if (!token) {
        showError('لطفاً دوباره وارد شوید.');
        return;
      }

      const params = new URLSearchParams();
      if (statusFilter !== 'all') params.append('status', statusFilter);
      const queryParams = params.toString() ? `?${params.toString()}` : '';

      const [tripRes, compRes] = await Promise.all([
        fetch(`${API_BASE_URL}/admin/trips${queryParams}`, {
          headers: { 'X-Auth-Token': token, }
        }),
        fetch(`${API_BASE_URL}/admin/trip-companions${queryParams}`, {
          headers: { 'X-Auth-Token': token, }
        })
      ]);

      if (!tripRes.ok || !compRes.ok) {
        const tripErr = await tripRes.json().catch(() => ({}));
        const compErr = await compRes.json().catch(() => ({}));
        throw new Error(tripErr.error || compErr.error || 'خطا در دریافت درخواست‌ها');
      }

      const tripData = await tripRes.json();
      const compData = await compRes.json();

      const trips = (tripData.success ? tripData.data : tripData).map((t: any) => ({
        id: t.id,
        type: 'trip' as const,
        status: t.status || 'pending',
        title: t.title || 'درخواست سفر',
       user: { name: t.user?.name || 'نامشخص' ,phone: t.user?.phone || 'نامشخص' },
        origin: {
          country: t.origin?.country || 'نامشخص',
          city: t.origin?.city || 'نامشخص'
        },
        destination: {
          country: t.destination?.country || 'نامشخص',
          city: t.destination?.city || 'نامشخص'
        },
        vehicle_type_label: t.vehicle_type_label,
        capacity: t.capacity ?? 0,
        departure_date_shamsi: t.departure_date_shamsi,
        arrival_date_shamsi: t.arrival_date_shamsi,
        created_at_shamsi: t.created_at_shamsi || 'نامشخص',
        rejection_reason: t.rejection_reason
      }));

      const companions = (compData.success ? compData.data : compData).map((c: any) => ({
        id: c.id,
        type: 'companion' as const,
        status: c.status || 'pending',
        title: c.title || 'درخواست همسفری',
        user: { name: c.user?.name || 'نامشخص' ,phone: c.user?.phone || 'نامشخص' },
        origin: {
          country: c.origin?.country || 'نامشخص',
          city: c.origin?.city || 'نامشخص'
        },
        destination: {
          country: c.destination?.country || 'نامشخص',
          city: c.destination?.city || 'نامشخص'
        },
        vehicle_types_label: c.vehicle_types_label,
        capacity: c.capacity ?? 0,
        available_from_shamsi: c.available_from_shamsi,
        available_to_shamsi: c.available_to_shamsi,
        created_at_shamsi: c.created_at_shamsi || 'نامشخص',
        rejection_reason: c.rejection_reason
      }));

      setRequests([...trips, ...companions]);
    } catch (error: any) {
      showError(error.message || 'خطا در ارتباط با سرور');
    } finally {
      setLoading(false);
    }
  };

  const handleApprove = async (id: number, type: 'trip' | 'companion') => {
    if (!confirm('آیا از تایید این درخواست اطمینان دارید؟')) return;

    setIsProcessing(true);
    try {
      const token = localStorage.getItem('auth_token');
      if (!token) {
        showError('لطفاً دوباره وارد شوید.');
        return;
      }

      const endpoint = type === 'trip' ? 'trips' : 'trip-companions';
      const res = await fetch(`${API_BASE_URL}/admin/${endpoint}/${id}/approve`, {
        method: 'PUT',
        headers: { 'X-Auth-Token': token, }
      });

      if (!res.ok) {
        const err = await res.json().catch(() => ({}));
        throw new Error(err.message || err.error || 'خطا در تأیید');
      }

      showSuccess('درخواست با موفقیت تأیید شد');
      fetchRequests();
    } catch (err: any) {
      showError(err.message || 'خطا در ارتباط');
    } finally {
      setIsProcessing(false);
    }
  };

  const handleReject = async () => {
    if (!selectedId || !rejectReason.trim()) {
      showError('لطفاً دلیل رد را وارد کنید');
      return;
    }

    setIsProcessing(true);
    try {
      const request = requests.find(r => r.id === selectedId);
      if (!request) {
        showError('درخواست یافت نشد');
        return;
      }

      const token = localStorage.getItem('auth_token');
      const endpoint = request.type === 'trip' ? 'trips' : 'trip-companions';
      const res = await fetch(`${API_BASE_URL}/admin/${endpoint}/${selectedId}/reject`, {
        method: 'PUT',
        headers: {
          'X-Auth-Token': token,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ reason: rejectReason })
      });

      if (!res.ok) {
        const err = await res.json().catch(() => ({}));
        throw new Error(err.message || err.error || 'خطا در رد درخواست');
      }

      showSuccess('درخواست با موفقیت رد شد');
      setShowRejectDialog(false);
      setSelectedId(null);
      setRejectReason('');
      fetchRequests();
    } catch (err: any) {
      showError(err.message || 'خطا در ارتباط');
    } finally {
      setIsProcessing(false);
    }
  };

  const openRejectDialog = (id: number) => {
    setSelectedId(id);
    setShowRejectDialog(true);
  };

  useEffect(() => {
    fetchRequests();
  }, [statusFilter]);

  const filteredRequests = requests.filter(req => {
    if (typeFilter !== 'all' && req.type !== typeFilter) {
      return false;
    }
    if (searchTerm === '') return true;
    return (
      req.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
      req.user.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
      req.origin.city.toLowerCase().includes(searchTerm.toLowerCase()) ||
      req.destination.city.toLowerCase().includes(searchTerm.toLowerCase())
    );
  });

  return (
    <div className="flex flex-col space-y-8">
      {showRejectDialog && (
        <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4">
          <div className="bg-white dark:bg-custom-dark-blue-light rounded-2xl p-6 max-w-md w-full shadow-xl">
            <h3 className="text-xl font-bold mb-4 text-gray-800 dark:text-white">دلیل رد درخواست</h3>
            <textarea
              value={rejectReason}
              onChange={(e) => setRejectReason(e.target.value)}
              placeholder="لطفاً دلیل رد کردن این درخواست را وارد کنید..."
              className="w-full h-32 p-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-gray-50 dark:bg-custom-dark-blue text-gray-800 dark:text-gray-200 resize-none focus:ring-2 focus:ring-custom-purple focus:border-transparent"
            />
            <div className="flex gap-3 mt-4">
              <button
                onClick={() => {
                  setShowRejectDialog(false);
                  setSelectedId(null);
                  setRejectReason('');
                }}
                disabled={isProcessing}
                className="flex-1 py-2 px-4 bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-300 rounded-lg hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors disabled:opacity-50"
              >
                انصراف
              </button>
              <button
                onClick={handleReject}
                disabled={isProcessing || !rejectReason.trim()}
                className="flex-1 py-2 px-4 bg-red-500 text-white rounded-lg hover:bg-red-600 transition-colors disabled:opacity-50"
              >
                {isProcessing ? 'در حال پردازش...' : 'رد کردن'}
              </button>
            </div>
          </div>
        </div>
      )}

      <div>
        <h1 className="text-3xl font-bold text-gray-800 dark:text-white">بررسی درخواست‌های سفر و همسفری</h1>
        <p className="text-gray-500 dark:text-gray-400 mt-1">درخواست‌های ثبت شده توسط کاربران را مدیریت کنید.</p>
      </div>

      <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div className="relative">
            <SearchIcon className="absolute right-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
            <input
              type="text"
              placeholder="جستجو در درخواست‌ها..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="w-full pr-10 pl-4 py-2 bg-gray-50 dark:bg-custom-dark-blue border border-gray-300 dark:border-gray-600 rounded-lg text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-custom-purple focus:border-transparent"
            />
          </div>

          <select
            value={statusFilter}
            onChange={(e) => setStatusFilter(e.target.value as any)}
            className="px-4 py-2 bg-gray-50 dark:bg-custom-dark-blue border border-gray-300 dark:border-gray-600 rounded-lg text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-custom-purple focus:border-transparent"
          >
            <option value="all">همه وضعیت‌ها</option>
            <option value="pending">در انتظار بررسی</option>
            <option value="approved">تایید شده</option>
            <option value="rejected">رد شده</option>
          </select>

          <select
            value={typeFilter}
            onChange={(e) => setTypeFilter(e.target.value as any)}
            className="px-4 py-2 bg-gray-50 dark:bg-custom-dark-blue border border-gray-300 dark:border-gray-600 rounded-lg text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-custom-purple focus:border-transparent"
          >
            <option value="all">همه انواع</option>
            <option value="trip">سفر</option>
            <option value="companion">همسفری</option>
          </select>
        </div>

        <div className="mt-4 text-sm text-gray-500 dark:text-gray-400">
          {filteredRequests.length} درخواست یافت شد
        </div>
      </div>

      {loading ? (
        <div className="flex items-center justify-center py-16">
          <div className="animate-spin rounded-full h-12 w-12 border-4 border-custom-purple border-t-transparent"></div>
        </div>
      ) : filteredRequests.length > 0 ? (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {filteredRequests.map(req => (
            <TripRequestCard
              key={`${req.type}-${req.id}`}
              {...req}
              onNavigate={onNavigate}
              onApprove={handleApprove}
              onReject={openRejectDialog}
              isProcessing={isProcessing}
            />
          ))}
        </div>
      ) : (
        <div className="text-center py-12 bg-custom-light-card dark:bg-custom-dark-blue-light rounded-2xl">
          <PlaneIcon className="w-16 h-16 mx-auto text-gray-300 dark:text-gray-600" />
          <h3 className="mt-4 text-xl font-semibold text-gray-800 dark:text-gray-200">درخواستی یافت نشد</h3>
          <p className="text-gray-500 dark:text-gray-400 mt-2">فیلترهای خود را تغییر دهید</p>
        </div>
      )}
    </div>
  );
};

const TripRequestCard: React.FC<{
  id: number;
  type: 'trip' | 'companion';
  user: { name: string; phone:string;};
  title: string;
  origin: { country: string; city: string; };
  destination: { country: string; city: string; };
  vehicle_type_label?: string;
  vehicle_types_label?: string;
  capacity?: number;
  departure_date_shamsi?: string | null;
  arrival_date_shamsi?: string | null;
  available_from_shamsi?: string | null;
  available_to_shamsi?: string | null;
  status: 'pending' | 'approved' | 'rejected';
  created_at_shamsi: string;
  onNavigate: (page: string, payload?: any) => void;
  onApprove: (id: number, type: 'trip' | 'companion') => void;
  onReject: (id: number) => void;
  isProcessing: boolean;
}> = ({
  id,
  type,
  user,
  title,
  origin,
  destination,
  vehicle_type_label,
  vehicle_types_label,
  capacity,
  departure_date_shamsi,
  arrival_date_shamsi,
  available_from_shamsi,
  available_to_shamsi,
  status,
  created_at_shamsi,
  onNavigate,
  onApprove,
  onReject,
  isProcessing
}) => {
    const isTrip = type === 'trip';
    const vehicleLabel = parseAndTranslateVehicles(
      isTrip ? vehicle_type_label : vehicle_types_label
    );
    const dateInfo = isTrip
      ? departure_date_shamsi || 'نامشخص'
      : available_from_shamsi || 'نامشخص';

    return (
      <div className="bg-custom-light-card dark:bg-custom-dark-blue-light rounded-2xl shadow-sm overflow-hidden flex flex-col group transform hover:-translate-y-1.5 transition-transform duration-300">
        <div className="p-5 flex flex-col flex-grow">
          <div className="flex items-center justify-between mb-3">
            <div className="flex items-center">
              <div className="w-10 h-10 rounded-full bg-gray-300 dark:bg-gray-600 flex items-center justify-center text-white">
                {user.name.charAt(0).toUpperCase()}
              </div>
              <div className="mr-3">
                <h3 className="font-semibold text-gray-800 dark:text-white">{user.name}</h3>
                <p className="text-xs text-gray-400">{user.phone}</p>
              </div>
            </div>
            <div className={`flex items-center gap-2 text-xs font-bold px-2 py-1 rounded-full ${isTrip ? 'bg-custom-orange/10 text-custom-orange' : 'bg-custom-cyan/10 text-custom-cyan'}`}>
              {isTrip ? <PlaneIcon className="w-4 h-4" /> : <UserIcon className="w-4 h-4" />}
              <span>{isTrip ? 'سفر' : 'همسفری'}</span>
            </div>
          </div>

          <div className="mb-3">
            <span className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${status === 'pending' ? 'bg-yellow-100 text-yellow-700 dark:bg-yellow-900/30 dark:text-yellow-400' :
                status === 'approved' ? 'bg-green-100 text-green-700 dark:bg-green-900/30 dark:text-green-400' :
                  'bg-red-100 text-red-700 dark:bg-red-900/30 dark:text-red-400'
              }`}>
              {status === 'pending' ? '⏳ در انتظار' : status === 'approved' ? '✓ تایید شده' : '✗ رد شده'}
            </span>
          </div>
          <h4 className="text-lg font-bold text-gray-800 dark:text-white mb-3 flex-grow">{title}</h4>

          <div className="space-y-3 text-sm text-gray-600 dark:text-gray-300 mb-4 border-t border-gray-200 dark:border-gray-700 pt-4">
            <InfoRow icon={MapPinIcon} label="مبدا" value={`${origin.city}, ${origin.country}`} />
            <InfoRow icon={MapPinIcon} label="مقصد" value={`${destination.city}, ${destination.country}`} />
            {vehicleLabel && (
              <InfoRow icon={CarIcon} label="وسیله نقلیه" value={vehicleLabel} />
            )}
            <InfoRow icon={CalendarIcon} label="تاریخ" value={dateInfo} />
            {capacity != null && (
              <InfoRow icon={ScaleIcon} label="ظرفیت" value={`${capacity} کیلوگرم`} />
            )}
          </div>

          <div className="mt-auto space-y-2">
            <button
              onClick={() => onNavigate('trip-detail', { id, type })}
              className="w-full bg-gray-100 dark:bg-custom-dark-blue hover:bg-gray-200 dark:hover:bg-gray-800 text-gray-700 dark:text-gray-300 font-bold py-2 px-4 rounded-lg transition-colors duration-300"
            >
              مشاهده جزئیات
            </button>
            {status === 'pending' && (
              <div className="grid grid-cols-2 gap-2">
                <button
                  onClick={() => onReject(id)}
                  disabled={isProcessing}
                  className="w-full flex items-center justify-center gap-1 bg-red-500/10 hover:bg-red-500/20 text-red-500 font-bold py-2 px-4 rounded-lg transition-colors duration-300 disabled:opacity-50"
                >
                  <XIcon className="w-4 h-4" />
                  <span>رد</span>
                </button>
                <button
                  onClick={() => onApprove(id, type)}
                  disabled={isProcessing}
                  className="w-full flex items-center justify-center gap-1 bg-green-500/10 hover:bg-green-500/20 text-green-500 font-bold py-2 px-4 rounded-lg transition-colors duration-300 disabled:opacity-50"
                >
                  <CheckIcon className="w-4 h-4" />
                  <span>تایید</span>
                </button>
              </div>
            )}
          </div>
        </div>
      </div>
    );
  };

const InfoRow: React.FC<{ icon: React.FC<{ className?: string }>, label: string, value: string }> = ({ icon: Icon, label, value }) => (
  <div className="flex items-center justify-between">
    <div className="flex items-center text-gray-500 dark:text-gray-400">
      <Icon className="w-4 h-4 ml-2" />
      <span>{label}:</span>
    </div>
    <span className="font-semibold text-gray-800 dark:text-gray-200">{value}</span>
  </div>
);

export default AcceptCompanionPage;
