// src/components/pages/AcceptOnlinePurchasePage.tsx
import React, { useState, useEffect } from 'react';
import { SearchIcon, MapPinIcon, CheckIcon, XIcon, ECommerceIcon, DollarSignIcon, PackageIcon, UserIcon } from '../Icons';
import { API_BASE_URL } from '@/src/config.js';

interface PurchaseRequest {
  id: number;
  type: 'purchase' | 'shopper';
  status: 'pending' | 'approved' | 'rejected';
  title: string;
  user: { name: string; phone:string;};
  purchaseLocation?: { country: string; city: string; };
  deliveryLocation?: { country: string; city: string; };
  itemPrice?: number;
  proposedFee?: number;
  currency?: 'USD' | 'CAD' | 'IRT' | 'EUR';
  serviceTitle?: string;
  categories?: string;
  datePosted: string;
  created_at: string;
}

interface AcceptOnlinePurchasePageProps {
  onNavigate: (page: string, payload?: any) => void;
  showError: (message: string) => void;
  showSuccess: (message: string) => void;
}

// ✅ تعریف دسته‌بندی‌ها برای تبدیل انگلیسی به فارسی
const categoryOptions: { [key: string]: string } = {
  'electronics': 'الکترونیک',
  'fashion': 'مد و پوشاک',
  'books': 'کتاب و رسانه',
  'home': 'خانه و آشپزخانه',
  'beauty': 'آرایشی و بهداشتی',
  'other': 'سایر'
};

const translateCategories = (categoriesInput: string | string[] | null | undefined): string => {
  if (!categoriesInput) return 'عمومی';

  let categoriesArray: string[] = [];

  if (typeof categoriesInput === 'string') {
    try {
      categoriesArray = JSON.parse(categoriesInput);
    } catch {
      categoriesArray = categoriesInput.split(',').map(cat => cat.trim()).filter(Boolean);
    }
  } else if (Array.isArray(categoriesInput)) {
    categoriesArray = categoriesInput;
  }

  if (categoriesArray.length === 0) return 'عمومی';

  const translated = categoriesArray.map(engCat => {
    const lowerCat = engCat.toLowerCase().trim();
    return categoryOptions[lowerCat] || engCat;
  });

  return translated.join('، ');
};

const AcceptOnlinePurchasePage: React.FC<AcceptOnlinePurchasePageProps> = ({ onNavigate, showError, showSuccess }) => {
  const [requests, setRequests] = useState<PurchaseRequest[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState<'all' | 'pending' | 'approved' | 'rejected'>('pending');
  const [typeFilter, setTypeFilter] = useState<'all' | 'purchase' | 'shopper'>('all');
  const [showRejectDialog, setShowRejectDialog] = useState(false);
  const [selectedId, setSelectedId] = useState<number | null>(null);
  const [selectedType, setSelectedType] = useState<'purchase' | 'shopper'>('purchase');
  const [rejectReason, setRejectReason] = useState('');
  const [isProcessing, setIsProcessing] = useState(false);

  const fetchRequests = async () => {
    setLoading(true);
    try {
      const token = localStorage.getItem('auth_token');
      if (!token) {
        showError('لطفاً دوباره وارد شوید.');
        return;
      }

      const params = new URLSearchParams();
      if (statusFilter !== 'all') {
        params.append('status', statusFilter);
      }
      const queryParams = params.toString() ? `?${params.toString()}` : '';

      const [purchaseRes, shopperRes] = await Promise.all([
        fetch(`${API_BASE_URL}/admin/online-purchases${queryParams}`, {
          headers: { 'X-Auth-Token': token, }
        }),
        fetch(`${API_BASE_URL}/admin/online-shoppers${queryParams}`, {
          headers: { 'X-Auth-Token': token, }
        })
      ]);

      if (!purchaseRes.ok || !shopperRes.ok) {
        throw new Error('خطا در دریافت درخواست‌ها');
      }

      const purchases = (await purchaseRes.json()).map((item: any) => ({
        id: item.id,
        type: 'purchase' as const,
        status: item.status || 'pending',
        title: item.title || 'درخواست خرید',
        user: { name: item.user?.name || 'نامشخص' ,phone: item.user?.phone || 'نامشخص' },
        purchaseLocation: {
          country: item.purchase_country || 'نامشخص',
          city: item.purchase_city || 'نامشخص'
        },
        deliveryLocation: {
          country: item.delivery_country || 'نامشخص',
          city: item.delivery_city || 'نامشخص'
        },
        itemPrice: item.item_price || 0,
        proposedFee: item.proposed_fee || 0,
        currency: item.currency || 'IRT',
        datePosted: new Date(item.created_at).toLocaleDateString('fa-IR'),
        created_at: item.created_at
      }));

      const shoppers = (await shopperRes.json()).map((item: any) => ({
        id: item.id,
        type: 'shopper' as const,
        status: item.status || 'pending',
        title: item.service_title || 'ثبت‌نام به عنوان خریدار',
        user: { name: item.user?.name || 'نامشخص' ,phone: item.user?.phone || 'نامشخص' },
        serviceTitle: item.service_title || 'خریدار',
        categories: item.categories || 'عمومی', // ✅ این مقدار به همین شکل نگه داشته میشه
        datePosted: new Date(item.created_at).toLocaleDateString('fa-IR'),
        created_at: item.created_at
      }));

      setRequests([...purchases, ...shoppers]);
    } catch (error: any) {
      showError(error.message || 'خطا در ارتباط با سرور');
    } finally {
      setLoading(false);
    }
  };

  const handleApprove = async (id: number, type: 'purchase' | 'shopper') => {
    if (!confirm('آیا از تایید این درخواست اطمینان دارید؟')) return;

    setIsProcessing(true);
    try {
      const token = localStorage.getItem('auth_token');
      if (!token) {
        showError('لطفاً دوباره وارد شوید.');
        return;
      }

      const endpoint = type === 'purchase' ? 'online-purchases' : 'online-shoppers';
      const res = await fetch(`${API_BASE_URL}/admin/${endpoint}/${id}/approve`, {
        method: 'PUT',
        headers: { 'X-Auth-Token': token, }
      });

      if (!res.ok) {
        const err = await res.json().catch(() => ({}));
        throw new Error(err.message || 'خطا در تأیید');
      }

      showSuccess('درخواست با موفقیت تأیید شد');
      fetchRequests();
    } catch (err: any) {
      showError(err.message || 'خطا در ارتباط');
    } finally {
      setIsProcessing(false);
    }
  };

  const handleReject = async () => {
    if (!selectedId || !rejectReason.trim()) {
      showError('لطفاً دلیل رد را وارد کنید');
      return;
    }

    setIsProcessing(true);
    try {
      const token = localStorage.getItem('auth_token');
      const endpoint = selectedType === 'purchase' ? 'online-purchases' : 'online-shoppers';
      const res = await fetch(`${API_BASE_URL}/admin/${endpoint}/${selectedId}/reject`, {
        method: 'PUT',
        headers: {
          'X-Auth-Token': token,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ reason: rejectReason })
      });

      if (!res.ok) {
        const err = await res.json().catch(() => ({}));
        throw new Error(err.message || 'خطا در رد درخواست');
      }

      showSuccess('درخواست با موفقیت رد شد');
      setShowRejectDialog(false);
      setSelectedId(null);
      setRejectReason('');
      fetchRequests();
    } catch (err: any) {
      showError(err.message || 'خطا در ارتباط');
    } finally {
      setIsProcessing(false);
    }
  };

  const openRejectDialog = (id: number, type: 'purchase' | 'shopper') => {
    setSelectedId(id);
    setSelectedType(type);
    setShowRejectDialog(true);
  };

  useEffect(() => {
    fetchRequests();
  }, [statusFilter]);

  const filteredRequests = requests.filter(req => {
    if (typeFilter !== 'all' && req.type !== typeFilter) return false;
    if (searchTerm === '') return true;
    if (req.title.toLowerCase().includes(searchTerm.toLowerCase())) return true;
    if (req.user.name.toLowerCase().includes(searchTerm.toLowerCase())) return true;
    if (req.type === 'purchase') {
      return (
        req.purchaseLocation?.city.toLowerCase().includes(searchTerm.toLowerCase()) ||
        req.purchaseLocation?.country.toLowerCase().includes(searchTerm.toLowerCase()) ||
        req.deliveryLocation?.city.toLowerCase().includes(searchTerm.toLowerCase()) ||
        req.deliveryLocation?.country.toLowerCase().includes(searchTerm.toLowerCase())
      );
    }
    return false;
  });

  return (
    <div className="flex flex-col space-y-8">
      {showRejectDialog && (
        <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4">
          <div className="bg-white dark:bg-custom-dark-blue-light rounded-2xl p-6 max-w-md w-full shadow-xl">
            <h3 className="text-xl font-bold mb-4 text-gray-800 dark:text-white">دلیل رد درخواست</h3>
            <textarea
              value={rejectReason}
              onChange={(e) => setRejectReason(e.target.value)}
              placeholder="لطفاً دلیل رد کردن این درخواست را وارد کنید..."
              className="w-full h-32 p-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-gray-50 dark:bg-custom-dark-blue text-gray-800 dark:text-gray-200 resize-none focus:ring-2 focus:ring-custom-purple focus:border-transparent"
            />
            <div className="flex gap-3 mt-4">
              <button
                onClick={() => {
                  setShowRejectDialog(false);
                  setSelectedId(null);
                  setRejectReason('');
                }}
                disabled={isProcessing}
                className="flex-1 py-2 px-4 bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-300 rounded-lg hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors disabled:opacity-50"
              >
                انصراف
              </button>
              <button
                onClick={handleReject}
                disabled={isProcessing || !rejectReason.trim()}
                className="flex-1 py-2 px-4 bg-red-500 text-white rounded-lg hover:bg-red-600 transition-colors disabled:opacity-50"
              >
                {isProcessing ? 'در حال پردازش...' : 'رد کردن'}
              </button>
            </div>
          </div>
        </div>
      )}

      <div>
        <h1 className="text-3xl font-bold text-gray-800 dark:text-white">بررسی درخواست‌های خرید اینترنتی</h1>
        <p className="text-gray-500 dark:text-gray-400 mt-1">درخواست‌های خرید و ثبت‌نام خریدار را مدیریت کنید.</p>
      </div>

      <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div className="relative">
            <SearchIcon className="absolute right-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
            <input
              type="text"
              placeholder="جستجو در درخواست‌ها..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="w-full pr-10 pl-4 py-2 bg-gray-50 dark:bg-custom-dark-blue border border-gray-300 dark:border-gray-600 rounded-lg text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-custom-purple focus:border-transparent"
            />
          </div>

          <select
            value={statusFilter}
            onChange={(e) => setStatusFilter(e.target.value as any)}
            className="px-4 py-2 bg-gray-50 dark:bg-custom-dark-blue border border-gray-300 dark:border-gray-600 rounded-lg text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-custom-purple focus:border-transparent"
          >
            <option value="all">همه وضعیت‌ها</option>
            <option value="pending">در انتظار بررسی</option>
            <option value="approved">تایید شده</option>
            <option value="rejected">رد شده</option>
          </select>

          <select
            value={typeFilter}
            onChange={(e) => setTypeFilter(e.target.value as any)}
            className="px-4 py-2 bg-gray-50 dark:bg-custom-dark-blue border border-gray-300 dark:border-gray-600 rounded-lg text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-custom-purple focus:border-transparent"
          >
            <option value="all">همه انواع</option>
            <option value="purchase">درخواست خرید</option>
            <option value="shopper">ثبت‌نام خریدار</option>
          </select>
        </div>

        <div className="mt-4 text-sm text-gray-500 dark:text-gray-400">
          {filteredRequests.length} درخواست یافت شد
        </div>
      </div>

      {loading ? (
        <div className="flex items-center justify-center py-16">
          <div className="animate-spin rounded-full h-12 w-12 border-4 border-custom-purple border-t-transparent"></div>
        </div>
      ) : filteredRequests.length > 0 ? (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {filteredRequests.map(req => (
            <PurchaseRequestCard
              key={`${req.type}-${req.id}`}
              {...req}
              onNavigate={onNavigate}
              onApprove={(id) => handleApprove(id, req.type)}
              onReject={(id) => openRejectDialog(id, req.type)}
              isProcessing={isProcessing}
            />
          ))}
        </div>
      ) : (
        <div className="text-center py-12 bg-custom-light-card dark:bg-custom-dark-blue-light rounded-2xl">
          <ECommerceIcon className="w-16 h-16 mx-auto text-gray-300 dark:text-gray-600" />
          <h3 className="mt-4 text-xl font-semibold text-gray-800 dark:text-gray-200">درخواستی یافت نشد</h3>
          <p className="text-gray-500 dark:text-gray-400 mt-2">فیلترهای خود را تغییر دهید</p>
        </div>
      )}
    </div>
  );
};

const PurchaseRequestCard: React.FC<{
  id: number;
  type: 'purchase' | 'shopper';
  user: { name: string; phone:string;};
  title: string;
  status: 'pending' | 'approved' | 'rejected';
  purchaseLocation?: { country: string; city: string; };
  deliveryLocation?: { country: string; city: string; };
  itemPrice?: number;
  proposedFee?: number;
  currency?: 'USD' | 'CAD' | 'IRT' | 'EUR';
  serviceTitle?: string;
  categories?: string;
  datePosted: string;
  onNavigate: (page: string, payload?: any) => void;
  onApprove: (id: number) => void;
  onReject: (id: number) => void;
  isProcessing: boolean;
}> = ({
  id,
  type,
  user,
  title,
  status,
  purchaseLocation,
  deliveryLocation,
  itemPrice,
  proposedFee,
  currency,
  serviceTitle,
  categories,
  datePosted,
  onNavigate,
  onApprove,
  onReject,
  isProcessing
}) => {
    return (
      <div className="bg-custom-light-card dark:bg-custom-dark-blue-light rounded-2xl shadow-sm overflow-hidden flex flex-col group transform hover:-translate-y-1.5 transition-transform duration-300">
        <div className="p-5 flex flex-col flex-grow">
          <div className="flex items-center justify-between mb-3">
            <div className="flex items-center">
              <div className="w-10 h-10 rounded-full bg-gray-300 dark:bg-gray-600 flex items-center justify-center text-white">
                {user.name.charAt(0).toUpperCase()}
              </div>
              <div className="mr-3">
                <h3 className="font-semibold text-gray-800 dark:text-white">{user.name}</h3>
                <p className="text-xs text-gray-400">{user.phone}</p>
              </div>
            </div>
            <div className={`flex items-center gap-2 text-xs font-bold px-2 py-1 rounded-full ${type === 'purchase' ? 'bg-custom-purple/10 text-custom-purple' : 'bg-custom-cyan/10 text-custom-cyan'
              }`}>
              {type === 'purchase' ? <PackageIcon className="w-4 h-4" /> : <UserIcon className="w-4 h-4" />}
              <span>{type === 'purchase' ? 'خرید' : 'خریدار'}</span>
            </div>
          </div>

          <div className="mb-3">
            <span className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${status === 'pending' ? 'bg-yellow-100 text-yellow-700 dark:bg-yellow-900/30 dark:text-yellow-400' :
                status === 'approved' ? 'bg-green-100 text-green-700 dark:bg-green-900/30 dark:text-green-400' :
                  'bg-red-100 text-red-700 dark:bg-red-900/30 dark:text-red-400'
              }`}>
              {status === 'pending' ? '⏳ در انتظار' : status === 'approved' ? '✓ تایید شده' : '✗ رد شده'}
            </span>
          </div>

          <h4 className="text-lg font-bold text-gray-800 dark:text-white mb-3 flex-grow line-clamp-2">{title}</h4>

          <div className="space-y-3 text-sm text-gray-600 dark:text-gray-300 mb-4 border-t border-gray-200 dark:border-gray-700 pt-4">
            {type === 'purchase' ? (
              <>
                
                <InfoRow icon={MapPinIcon} label="تحویل در" value={`${deliveryLocation?.city}, ${deliveryLocation?.country}`} />
                <InfoRow icon={DollarSignIcon} label="قیمت کالا" value={`${itemPrice?.toLocaleString() || 0} ${currency || 'IRT'}`} />
                <InfoRow icon={DollarSignIcon} label="کارمزد" value={`${proposedFee?.toLocaleString() || 0} ${currency || 'IRT'}`} valueClassName="text-green-600 dark:text-green-400" />
              </>
            ) : (
              <>
                <InfoRow icon={UserIcon} label="خدمت" value={serviceTitle || 'خریدار'} />
                <InfoRow icon={ECommerceIcon} label="دسته‌بندی" value={translateCategories(categories)} />
              </>
            )}
          </div>

          <div className="mt-auto space-y-2">
            <button
              onClick={() => onNavigate('purchase-detail', { id, type })}
              className="w-full bg-gray-100 dark:bg-custom-dark-blue hover:bg-gray-200 dark:hover:bg-gray-800 text-gray-700 dark:text-gray-300 font-bold py-2 px-4 rounded-lg transition-colors duration-300"
            >
              مشاهده جزئیات
            </button>

            {status === 'pending' && (
              <div className="grid grid-cols-2 gap-2">
                <button
                  onClick={() => onReject(id)}
                  disabled={isProcessing}
                  className="w-full flex items-center justify-center gap-1 bg-red-500/10 hover:bg-red-500/20 text-red-500 font-bold py-2 px-4 rounded-lg transition-colors duration-300 disabled:opacity-50"
                >
                  <XIcon className="w-4 h-4" />
                  <span>رد</span>
                </button>
                <button
                  onClick={() => onApprove(id)}
                  disabled={isProcessing}
                  className="w-full flex items-center justify-center gap-1 bg-green-500/10 hover:bg-green-500/20 text-green-500 font-bold py-2 px-4 rounded-lg transition-colors duration-300 disabled:opacity-50"
                >
                  <CheckIcon className="w-4 h-4" />
                  <span>تایید</span>
                </button>
              </div>
            )}
          </div>
        </div>
      </div>
    );
  };

const InfoRow: React.FC<{ icon: React.FC<{ className?: string }>, label: string, value: string, valueClassName?: string }> = ({ icon: Icon, label, value, valueClassName = 'text-gray-800 dark:text-gray-200' }) => (
  <div className="flex items-center justify-between">
    <div className="flex items-center text-gray-500 dark:text-gray-400">
      <Icon className="w-4 h-4 ml-2" />
      <span>{label}:</span>
    </div>
    <span className={`font-semibold ${valueClassName}`}>{value}</span>
  </div>
);

export default AcceptOnlinePurchasePage;
