// src/components/pages/AcceptShipmentPage.tsx
import React, { useState, useEffect } from 'react';
import { SearchIcon, MapPinIcon, CheckIcon, XIcon, PackageIcon, ChevronDownIcon, ScaleIcon, UserIcon, PlaneIcon } from '../Icons';
import { API_BASE_URL } from '@/src/config.js';

interface ShipmentRequest {
    id: number;
    type: 'send' | 'carry';
    status: 'pending' | 'approved' | 'rejected';
    title: string;
    user: { name: string; avatar: string; phone: string; };
    origin: { country: string; city: string; };
    destination: { country: string; city: string; };
    weight?: number;
    capacity?: number;
    fee: number;
    currency: 'USD' | 'CAD' | 'IRT' | 'EUR';
    datePosted: string;
    created_at: string;
}

interface AdminShipmentRequestsPageProps {
    onNavigate: (page: string, payload?: any) => void;
    showError: (message: string) => void;
    showSuccess: (message: string) => void;
}

const AdminShipmentRequestsPage: React.FC<AdminShipmentRequestsPageProps> = ({ onNavigate, showError, showSuccess }) => {
    const [requests, setRequests] = useState<ShipmentRequest[]>([]);
    const [loading, setLoading] = useState(true);
    const [searchTerm, setSearchTerm] = useState('');
    const [statusFilter, setStatusFilter] = useState<'all' | 'pending' | 'approved' | 'rejected'>('pending');
    const [typeFilter, setTypeFilter] = useState<'all' | 'send' | 'carry'>('all');
    const [showRejectDialog, setShowRejectDialog] = useState(false);
    const [selectedShipmentId, setSelectedShipmentId] = useState<number | null>(null);
    const [rejectReason, setRejectReason] = useState('');
    const [isProcessing, setIsProcessing] = useState(false);

    const fetchRequests = async () => {
        setLoading(true);
        try {
            const token = localStorage.getItem('auth_token');
            if (!token) {
                showError('لطفاً دوباره وارد شوید.');
                return;
            }

            const params = new URLSearchParams();
            if (statusFilter !== 'all') params.append('status', statusFilter);
            if (typeFilter !== 'all') params.append('type', typeFilter);
            if (searchTerm.trim()) params.append('search', searchTerm.trim());

            const url = `${API_BASE_URL}/admin/shipments${params.toString() ? `?${params.toString()}` : ''}`;

            const response = await fetch(url, {
                headers: {
                    'X-Auth-Token': token,
                    'Content-Type': 'application/json',
                },
            });

            if (!response.ok) {
                const errorData = await response.json().catch(() => ({}));
                throw new Error(errorData.error || 'خطا در دریافت درخواست‌ها');
            }

            const result = await response.json();

            let shipments = [];
            if (result.success && result.data) {
                shipments = result.data;
            } else if (Array.isArray(result)) {
                shipments = result;
            } else {
                throw new Error('ساختار پاسخ سرور نامعتبر است');
            }

            const mappedRequests = shipments.map((item: any) => ({
                id: item.id,
                type: item.type,
                status: item.status || 'pending',
                title: item.title || 'بدون عنوان',
                user: {
                    name: item.user?.name || 'نامشخص',
                    phone: item.user?.phone || 'نامشخص',
                    avatar: item.user?.avatar || `https://ui-avatars.com/api/?name=${encodeURIComponent(item.user?.name || 'نامشخص')}&background=random`
                },
                origin: {
                    country: item.origin?.country || 'نامشخص',
                    city: item.origin?.city || 'نامشخص'
                },
                destination: {
                    country: item.destination?.country || 'نامشخص',
                    city: item.destination?.city || 'نامشخص'
                },
                weight: item.weight,
                capacity: item.carry_weight_capacity,
                fee: item.fee || 0,
                currency: item.currency || 'IRT',
                datePosted: new Date(item.created_at).toLocaleDateString('fa-IR'),
                created_at: item.created_at
            }));

            setRequests(mappedRequests);
        } catch (error: any) {
            showError(error.message || 'خطا در ارتباط با سرور');
            console.error('Fetch requests error:', error);
        } finally {
            setLoading(false);
        }
    };

    const handleApprove = async (id: number) => {
        if (!confirm('آیا از تایید این درخواست اطمینان دارید؟')) {
            return;
        }

        setIsProcessing(true);
        try {
            const token = localStorage.getItem('auth_token');
            if (!token) {
                showError('لطفاً دوباره وارد شوید.');
                return;
            }

            const response = await fetch(`${API_BASE_URL}/admin/shipments/${id}/approve`, {
                method: 'PUT',
                headers: {
                    'X-Auth-Token': token,
                    'Content-Type': 'application/json',
                },
            });

            if (!response.ok) {
                const errorData = await response.json().catch(() => ({}));
                throw new Error(errorData.message || errorData.error || 'خطا در تأیید درخواست');
            }

            const result = await response.json();
            showSuccess(result.message || 'درخواست با موفقیت تأیید شد');
            fetchRequests();
        } catch (error: any) {
            showError(error.message || 'خطا در ارتباط با سرور');
        } finally {
            setIsProcessing(false);
        }
    };

    const handleReject = async () => {
        if (!selectedShipmentId) return;

        if (!rejectReason.trim()) {
            showError('لطفاً دلیل رد کردن را وارد کنید');
            return;
        }

        setIsProcessing(true);
        try {
            const token = localStorage.getItem('auth_token');
            if (!token) {
                showError('لطفاً دوباره وارد شوید.');
                return;
            }

            const response = await fetch(`${API_BASE_URL}/admin/shipments/${selectedShipmentId}/reject`, {
                method: 'PUT',
                headers: {
                    'X-Auth-Token': token,
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ reason: rejectReason })
            });

            if (!response.ok) {
                const errorData = await response.json().catch(() => ({}));
                throw new Error(errorData.message || errorData.error || 'خطا در رد درخواست');
            }

            const result = await response.json();
            showSuccess(result.message || 'درخواست با موفقیت رد شد');

            setShowRejectDialog(false);
            setSelectedShipmentId(null);
            setRejectReason('');

            fetchRequests();
        } catch (error: any) {
            showError(error.message || 'خطا در ارتباط با سرور');
        } finally {
            setIsProcessing(false);
        }
    };

    const openRejectDialog = (id: number) => {
        setSelectedShipmentId(id);
        setShowRejectDialog(true);
    };

    useEffect(() => {
        fetchRequests();
    }, [statusFilter, typeFilter]);

    const filteredRequests = requests.filter(req =>
        searchTerm === '' ||
        req.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
        req.user.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
        req.origin.city.toLowerCase().includes(searchTerm.toLowerCase()) ||
        req.destination.city.toLowerCase().includes(searchTerm.toLowerCase())
    );

    return (
        <div className="flex flex-col space-y-8">
            {showRejectDialog && (
                <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4">
                    <div className="bg-white dark:bg-custom-dark-blue-light rounded-2xl p-6 max-w-md w-full shadow-xl">
                        <h3 className="text-xl font-bold mb-4 text-gray-800 dark:text-white">دلیل رد درخواست</h3>
                        <textarea
                            value={rejectReason}
                            onChange={(e) => setRejectReason(e.target.value)}
                            placeholder="لطفاً دلیل رد کردن این درخواست را وارد کنید..."
                            className="w-full h-32 p-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-gray-50 dark:bg-custom-dark-blue text-gray-800 dark:text-gray-200 resize-none focus:ring-2 focus:ring-custom-purple focus:border-transparent"
                        />
                        <div className="flex gap-3 mt-4">
                            <button
                                onClick={() => {
                                    setShowRejectDialog(false);
                                    setSelectedShipmentId(null);
                                    setRejectReason('');
                                }}
                                disabled={isProcessing}
                                className="flex-1 py-2 px-4 bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-300 rounded-lg hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors disabled:opacity-50"
                            >
                                انصراف
                            </button>
                            <button
                                onClick={handleReject}
                                disabled={isProcessing || !rejectReason.trim()}
                                className="flex-1 py-2 px-4 bg-red-500 text-white rounded-lg hover:bg-red-600 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
                            >
                                {isProcessing ? 'در حال پردازش...' : 'رد کردن'}
                            </button>
                        </div>
                    </div>
                </div>
            )}

            <div>
                <h1 className="text-3xl font-bold text-gray-800 dark:text-white">بررسی درخواست‌های مرسولات</h1>
                <p className="text-gray-500 dark:text-gray-400 mt-1">درخواست‌های ارسال و حمل مرسوله را تایید یا رد کنید.</p>
            </div>

            <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div className="relative">
                        <SearchIcon className="absolute right-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-400" />
                        <input
                            type="text"
                            placeholder="جستجو در درخواست‌ها..."
                            value={searchTerm}
                            onChange={(e) => setSearchTerm(e.target.value)}
                            className="w-full pr-10 pl-4 py-2 bg-gray-50 dark:bg-custom-dark-blue border border-gray-300 dark:border-gray-600 rounded-lg text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-custom-purple focus:border-transparent"
                        />
                    </div>

                    <select
                        value={statusFilter}
                        onChange={(e) => setStatusFilter(e.target.value as any)}
                        className="px-4 py-2 bg-gray-50 dark:bg-custom-dark-blue border border-gray-300 dark:border-gray-600 rounded-lg text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-custom-purple focus:border-transparent"
                    >
                        <option value="all">همه وضعیت‌ها</option>
                        <option value="pending">در انتظار تایید</option>
                        <option value="approved">تایید شده</option>
                        <option value="rejected">رد شده</option>
                    </select>

                    <select
                        value={typeFilter}
                        onChange={(e) => setTypeFilter(e.target.value as any)}
                        className="px-4 py-2 bg-gray-50 dark:bg-custom-dark-blue border border-gray-300 dark:border-gray-600 rounded-lg text-gray-800 dark:text-gray-200 focus:ring-2 focus:ring-custom-purple focus:border-transparent"
                    >
                        <option value="all">همه انواع</option>
                        <option value="send">ارسال بار</option>
                        <option value="carry">حمل بار</option>
                    </select>
                </div>

                <div className="mt-4 text-sm text-gray-500 dark:text-gray-400">
                    {filteredRequests.length} درخواست یافت شد
                </div>
            </div>

            {loading ? (
                <div className="flex items-center justify-center py-16">
                    <div className="animate-spin rounded-full h-12 w-12 border-4 border-custom-purple border-t-transparent"></div>
                </div>
            ) : filteredRequests.length > 0 ? (
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                    {filteredRequests.map(req => (
                        <RequestCard
                            key={req.id}
                            {...req}
                            onNavigate={onNavigate}
                            onApprove={handleApprove}
                            onReject={openRejectDialog}
                            isProcessing={isProcessing}
                        />
                    ))}
                </div>
            ) : (
                <div className="text-center py-12 bg-custom-light-card dark:bg-custom-dark-blue-light rounded-2xl">
                    <PackageIcon className="w-16 h-16 mx-auto text-gray-300 dark:text-gray-600" />
                    <h3 className="mt-4 text-xl font-semibold text-gray-800 dark:text-gray-200">درخواستی یافت نشد</h3>
                    <p className="text-gray-500 dark:text-gray-400 mt-2">فیلترهای خود را تغییر دهید</p>
                </div>
            )}
        </div>
    );
};

interface RequestCardProps extends ShipmentRequest {
    onNavigate: (page: string, payload?: any) => void;
    onApprove: (id: number) => void;
    onReject: (id: number) => void;
    isProcessing: boolean;
}

const RequestCard: React.FC<RequestCardProps> = (props) => {
    const { id, type, status, title, user, origin, destination, weight, capacity, fee, currency, datePosted, onNavigate, onApprove, onReject, isProcessing } = props;
    const isSendRequest = type === 'send';

    return (
        <div className="bg-custom-light-card dark:bg-custom-dark-blue-light rounded-2xl shadow-sm overflow-hidden flex flex-col group transform hover:-translate-y-1.5 transition-transform duration-300">
            <div className="p-5 flex flex-col flex-grow">
                <div className="flex items-center justify-between mb-3">
                    <div className="flex items-center">
                        <img src={user.avatar} alt={user.name} className="w-10 h-10 rounded-full" />
                        <div className="mr-3">
                            <h3 className="font-semibold text-gray-800 dark:text-white">{user.name}</h3>
                            <p className="text-xs text-gray-400">{user.phone}</p>
                        </div>
                    </div>
                    <div className={`flex items-center gap-2 text-xs font-bold px-2 py-1 rounded-full ${isSendRequest ? 'bg-custom-cyan/10 text-custom-cyan' : 'bg-custom-orange/10 text-custom-orange'}`}>
                        {isSendRequest ? <PackageIcon className="w-4 h-4" /> : <PlaneIcon className="w-4 h-4" />}
                        <span>{isSendRequest ? 'ارسال' : 'حمل'}</span>
                    </div>
                </div>

                <div className="mb-3">
                    <span className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${status === 'pending' ? 'bg-yellow-100 text-yellow-700 dark:bg-yellow-900/30 dark:text-yellow-400' :
                            status === 'approved' ? 'bg-green-100 text-green-700 dark:bg-green-900/30 dark:text-green-400' :
                                'bg-red-100 text-red-700 dark:bg-red-900/30 dark:text-red-400'
                        }`}>
                        {status === 'pending' ? '⏳ در انتظار' : status === 'approved' ? '✓ تایید شده' : '✗ رد شده'}
                    </span>
                </div>

                <h4 className="text-lg font-bold text-gray-800 dark:text-white mb-3 flex-grow line-clamp-2">{title}</h4>

                <div className="space-y-3 text-sm text-gray-600 dark:text-gray-300 mb-4 border-t border-gray-200 dark:border-gray-700 pt-4">
                    <InfoRow icon={MapPinIcon} label="مبدا" value={`${origin.city}, ${origin.country}`} />
                    <InfoRow icon={MapPinIcon} label="مقصد" value={`${destination.city}, ${destination.country}`} />
                    {isSendRequest ?
                        <InfoRow icon={ScaleIcon} label="وزن" value={weight ? `${weight} کیلوگرم` : 'نامشخص'} /> :
                        <InfoRow icon={ScaleIcon} label="ظرفیت" value={capacity ? `${capacity} کیلوگرم` : 'نامشخص'} />
                    }
                </div>

                <div className="mt-auto space-y-2">
                    <button
                        onClick={() => onNavigate('shipment-detail', id.toString())}
                        className="w-full bg-custom-purple hover:bg-purple-600 text-white font-bold py-2 px-4 rounded-lg transition-colors duration-300"
                    >
                        مشاهده جزئیات
                    </button>

                    {status === 'pending' && (
                        <div className="grid grid-cols-2 gap-2">
                            <button
                                onClick={() => onReject(id)}
                                disabled={isProcessing}
                                className="w-full flex items-center justify-center gap-1 bg-red-500/10 hover:bg-red-500/20 text-red-500 font-bold py-2 px-4 rounded-lg transition-colors duration-300 disabled:opacity-50 disabled:cursor-not-allowed"
                            >
                                <XIcon className="w-4 h-4" />
                                <span>رد</span>
                            </button>
                            <button
                                onClick={() => onApprove(id)}
                                disabled={isProcessing}
                                className="w-full flex items-center justify-center gap-1 bg-green-500/10 hover:bg-green-500/20 text-green-500 font-bold py-2 px-4 rounded-lg transition-colors duration-300 disabled:opacity-50 disabled:cursor-not-allowed"
                            >
                                <CheckIcon className="w-4 h-4" />
                                <span>تایید</span>
                            </button>
                        </div>
                    )}
                </div>
            </div>
        </div>
    );
};

const InfoRow: React.FC<{ icon: React.FC<{ className?: string }>, label: string, value: string }> = ({ icon: Icon, label, value }) => (
    <div className="flex items-center justify-between">
        <div className="flex items-center text-gray-500 dark:text-gray-400"> {/* ✅ اصلاح شد: tex → text */}
            <Icon className="w-4 h-4 ml-2" />
            <span>{label}:</span>
        </div>
        <span className="font-semibold text-gray-800 dark:text-gray-200 text-xs">{value}</span>
    </div>
);

export default AdminShipmentRequestsPage;