// src/components/pages/AddOrEditArticlePage.tsx
import React, { useState, useEffect, useRef } from 'react';
import { ArrowRightIcon, TypeIcon, TagIcon, ListIcon, UploadCloudIcon, XIcon, ChevronDownIcon } from '../Icons';
import { API_BASE_URL } from '@/src/config.js';
import { authFetch } from '@/App';

interface AddOrEditArticlePageProps {
    onNavigate: (page: string) => void;
    articleId?: string | null;
    isEditing?: boolean;
    showError: (msg: string) => void;
    showSuccess: (msg: string) => void;
}

interface CategoryOption {
    id: number;
    name: string;
}

// --- کامپوننت InputWithIcon ---
interface InputWithIconProps { 
    id: string; 
    icon: React.FC<{ className?: string }>; 
    label: string; 
    placeholder: string; 
    value: string; 
    onChange: (e: React.ChangeEvent<HTMLInputElement>) => void; 
}

const InputWithIcon: React.FC<InputWithIconProps> = ({ id, icon: Icon, label, placeholder, value, onChange }) => (
    <div>
        <label htmlFor={id} className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">{label}</label>
        <div className="relative">
            <div className="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none">
                <Icon className="w-5 h-5 text-gray-400" />
            </div>
            <input 
                type="text" 
                id={id} 
                value={value} 
                onChange={onChange} 
                className="w-full pr-10 pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition" 
                placeholder={placeholder} 
            />
        </div>
    </div>
);

// --- کامپوننت SelectWithIcon ---
interface SelectWithIconProps { 
    id: string; 
    icon: React.FC<{ className?: string }>; 
    label: string; 
    children: React.ReactNode; 
    value: string; 
    onChange: (e: React.ChangeEvent<HTMLSelectElement>) => void; 
}

const SelectWithIcon: React.FC<SelectWithIconProps> = ({ id, icon: Icon, label, children, value, onChange }) => {
    const [isOpen, setIsOpen] = useState(false);
    const selectorRef = useRef<HTMLDivElement>(null);

    const options = React.Children.toArray(children)
        .filter((child): child is React.ReactElement<React.OptionHTMLAttributes<HTMLOptionElement>> => 
            React.isValidElement(child) && child.type === 'option')
        .map(option => ({
            value: option.props.value?.toString() ?? '',
            label: option.props.children?.toString() || '',
            disabled: option.props.disabled
        }));
    
    const selectedOption = options.find(opt => opt.value === value);

    useEffect(() => {
        const handleClickOutside = (event: MouseEvent) => {
            if (selectorRef.current && !selectorRef.current.contains(event.target as Node)) {
                setIsOpen(false);
            }
        };
        document.addEventListener("mousedown", handleClickOutside);
        return () => document.removeEventListener("mousedown", handleClickOutside);
    }, []);

    const handleSelect = (selectedValue: string) => {
        if (onChange) {
            const syntheticEvent = {
                target: { value: selectedValue },
                currentTarget: { value: selectedValue },
            } as React.ChangeEvent<HTMLSelectElement>;
            onChange(syntheticEvent);
        }
        setIsOpen(false);
    };

    return (
        <div>
            <label htmlFor={id} className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">{label}</label>
            <div className="relative" ref={selectorRef}>
                <div className="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none">
                    <Icon className="w-5 h-5 text-gray-400" />
                </div>
                <button
                    type="button"
                    id={id}
                    onClick={() => setIsOpen(!isOpen)}
                    className="w-full flex items-center justify-between text-right pr-10 pl-10 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition"
                >
                    <span className={selectedOption && selectedOption.value ? "text-gray-800 dark:text-gray-200" : "text-gray-500"}>
                        {selectedOption ? selectedOption.label : 'انتخاب کنید'}
                    </span>
                    <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                        <ChevronDownIcon className={`w-5 h-5 text-gray-400 transition-transform ${isOpen ? 'rotate-180' : ''}`} />
                    </div>
                </button>
                {isOpen && (
                    <ul className="absolute z-20 w-full mt-1 bg-custom-light-card dark:bg-custom-dark-blue-light border border-gray-300 dark:border-gray-600 rounded-lg shadow-lg max-h-60 overflow-y-auto">
                        {options.map(option => (
                            <li
                                key={option.value}
                                onClick={() => !option.disabled && handleSelect(option.value)}
                                className={`px-4 py-2 transition-colors ${
                                    option.disabled 
                                        ? 'text-gray-400 cursor-not-allowed' 
                                        : 'cursor-pointer hover:bg-gray-100 dark:hover:bg-custom-dark-blue text-gray-800 dark:text-gray-200'
                                } ${value === option.value ? 'bg-gray-100 dark:bg-custom-dark-blue' : ''}`}
                            >
                                {option.label}
                            </li>
                        ))}
                    </ul>
                )}
            </div>
        </div>
    );
};

// --- کامپوننت TextareaWithIcon ---
interface TextareaWithIconProps { 
    id: string; 
    icon: React.FC<{ className?: string }>; 
    label: string; 
    placeholder: string; 
    value: string; 
    onChange: (e: React.ChangeEvent<HTMLTextAreaElement>) => void; 
}

const TextareaWithIcon: React.FC<TextareaWithIconProps> = ({ id, icon: Icon, label, placeholder, value, onChange }) => (
    <div>
        <label htmlFor={id} className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">{label}</label>
        <div className="relative">
            <div className="absolute top-3.5 right-0 flex items-center pr-3 pointer-events-none">
                <Icon className="w-5 h-5 text-gray-400" />
            </div>
            <textarea 
                id={id} 
                value={value} 
                onChange={onChange} 
                className="w-full pr-10 pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition min-h-[300px] resize-y" 
                placeholder={placeholder}
            />
        </div>
    </div>
);

// --- صفحه اصلی ---
const AddOrEditArticlePage: React.FC<AddOrEditArticlePageProps> = ({ 
    onNavigate, 
    articleId, 
    isEditing = false,
    showError,
    showSuccess
}) => {
    const [title, setTitle] = useState('');
    const [category, setCategory] = useState('');
    const [status, setStatus] = useState('draft');
    const [content, setContent] = useState('');
    const [imageFile, setImageFile] = useState<File | null>(null);
    const [previewUrl, setPreviewUrl] = useState<string | null>(null);
    const [categories, setCategories] = useState<CategoryOption[]>([]);
    const [loading, setLoading] = useState(false);
    const [existingImageUrl, setExistingImageUrl] = useState<string | null>(null);

    // Fetch categories
    useEffect(() => {
        const fetchCategories = async () => {
            try {
                const res = await authFetch(`${API_BASE_URL}/admin/article-categories`);
                const data = await res.json();
                const cats = Array.isArray(data.data) ? data.data : (Array.isArray(data) ? data : []);
                setCategories(cats.map((c: any) => ({ id: c.id, name: c.name })));
            } catch (err: any) {
                console.error('خطا در بارگذاری دسته‌بندی‌ها:', err);
                showError('خطا در بارگذاری دسته‌بندی‌ها');
            }
        };
        fetchCategories();
    }, []);

    // Fetch article if editing
    useEffect(() => {
        if (isEditing && articleId) {
            const fetchArticle = async () => {
                try {
                    setLoading(true);
                    const res = await authFetch(`${API_BASE_URL}/admin/articles/${articleId}`);
                    
                    if (!res.ok) {
                        throw new Error('خطا در دریافت اطلاعات مقاله');
                    }
                    
                    const response = await res.json();
                    const article = response.data || response;
                    
                    console.log('Article data:', article);
                    
                    setTitle(article.title || '');
                    setCategory(String(article.category_id || ''));
                    setStatus(article.status || 'draft');
                    setContent(article.content || '');
                    
                    if (article.image_url) {
                        const imageUrl = article.image_url.startsWith('http')
                            ? article.image_url
                            : `http://127.0.0.1:8000/${article.image_url.replace(/^\/+/, '')}`;
                        
                        setExistingImageUrl(imageUrl);
                        setPreviewUrl(imageUrl);
                    } else {
                        setExistingImageUrl(null);
                        setPreviewUrl(null);
                    }
                } catch (err: any) {
                    console.error('خطا در بارگذاری مقاله:', err);
                    showError(err.message || 'خطا در بارگذاری اطلاعات مقاله');
                } finally {
                    setLoading(false);
                }
            };
            fetchArticle();
        }
    }, [isEditing, articleId]);

    const handleImageChange = (event: React.ChangeEvent<HTMLInputElement>) => {
        const file = event.target.files?.[0];
        if (file) {
            if (file.size > 2 * 1024 * 1024) {
                showError('حجم فایل نباید بیشتر از 2 مگابایت باشد.');
                return;
            }
            if (!file.type.startsWith('image/')) {
                showError('فقط فایل‌های تصویری مجاز هستند.');
                return;
            }
            setImageFile(file);
            
            // حذف URL قبلی اگر از local file بود
            if (previewUrl && !existingImageUrl) {
                URL.revokeObjectURL(previewUrl);
            }
            
            setPreviewUrl(URL.createObjectURL(file));
        }
    };

    const removeImage = () => {
        if (previewUrl && imageFile) {
            URL.revokeObjectURL(previewUrl);
        }
        setImageFile(null);
        setPreviewUrl(null);
        setExistingImageUrl(null);
    };
    
    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        
        if (!title.trim() || !category || !content.trim()) {
            showError('لطفاً تمام فیلدهای اجباری را پر کنید.');
            return;
        }

        const formData = new FormData();
        formData.append('title', title.trim());
        formData.append('content', content.trim());
        formData.append('status', status);
        formData.append('category_id', category);
        
        if (imageFile) {
            formData.append('image', imageFile);
        }

        if (isEditing) {
            formData.append('_method', 'PUT');
        }

        try {
            setLoading(true);
            const url = isEditing 
                ? `${API_BASE_URL}/admin/articles/${articleId}`
                : `${API_BASE_URL}/admin/articles`;

            const response = await authFetch(url, {
                method: 'POST',
                body: formData,
            });

            if (!response.ok) {
                const errorData = await response.json();
                throw new Error(errorData.message || 'خطا در ذخیره مقاله');
            }

            showSuccess(isEditing ? 'مقاله با موفقیت ویرایش شد.' : 'مقاله با موفقیت ایجاد شد.');
            onNavigate('articles');
        } catch (err: any) {
            console.error('خطا در ذخیره مقاله:', err);
            showError(err.message || 'خطا در ذخیره مقاله');
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className="flex flex-col space-y-8">
            <div className="flex items-center justify-between">
                <div>
                    <h1 className="text-3xl font-bold text-gray-800 dark:text-white">
                        {isEditing ? 'ویرایش مقاله' : 'افزودن مقاله جدید'}
                    </h1>
                    <p className="text-gray-500 dark:text-gray-400 mt-1">
                        {isEditing ? 'اطلاعات مقاله را ویرایش کنید' : 'یک مقاله جدید ایجاد کنید'}
                    </p>
                </div>
                <button 
                    onClick={() => onNavigate('articles')} 
                    className="flex items-center gap-2 py-2 px-4 rounded-lg text-gray-600 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-custom-dark-blue transition-colors"
                >
                    <ArrowRightIcon className="w-5 h-5" />
                    <span>بازگشت به لیست</span>
                </button>
            </div>

            {loading && isEditing ? (
                <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-12 rounded-2xl text-center">
                    <div className="inline-block animate-spin rounded-full h-12 w-12 border-b-2 border-custom-purple"></div>
                    <p className="mt-4 text-gray-600 dark:text-gray-400">در حال بارگذاری اطلاعات مقاله...</p>
                </div>
            ) : (
                <form onSubmit={handleSubmit} className="grid grid-cols-1 lg:grid-cols-3 gap-8">
                    <div className="lg:col-span-2 space-y-8">
                        <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl shadow-sm space-y-6">
                            <InputWithIcon 
                                icon={TypeIcon} 
                                id="title" 
                                label="عنوان مقاله" 
                                placeholder="عنوان مقاله را وارد کنید" 
                                value={title} 
                                onChange={e => setTitle(e.target.value)} 
                            />
                            <TextareaWithIcon 
                                icon={ListIcon} 
                                id="content" 
                                label="متن مقاله" 
                                placeholder="محتوای مقاله را اینجا بنویسید..." 
                                value={content} 
                                onChange={e => setContent(e.target.value)} 
                            />
                        </div>
                    </div>
                    
                    <div className="lg:col-span-1 space-y-6">
                        <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm space-y-6">
                            <SelectWithIcon 
                                icon={TagIcon} 
                                id="category" 
                                label="دسته‌بندی" 
                                value={category} 
                                onChange={e => setCategory(e.target.value)}
                            >
                                <option value="" disabled>انتخاب کنید</option>
                                {categories.map(cat => (
                                    <option key={cat.id} value={cat.id}>{cat.name}</option>
                                ))}
                            </SelectWithIcon>
                            
                            <SelectWithIcon 
                                icon={ChevronDownIcon} 
                                id="status" 
                                label="وضعیت" 
                                value={status} 
                                onChange={e => setStatus(e.target.value)}
                            >
                                <option value="draft">پیش‌نویس</option>
                                <option value="published">منتشر شده</option>
                            </SelectWithIcon>
                        </div>
                        
                        <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
                            <h3 className="text-lg font-semibold mb-4 text-gray-800 dark:text-gray-200">تصویر شاخص</h3>
                            {previewUrl ? (
                                <div className="relative group">
                                    <img 
                                        src={previewUrl} 
                                        alt="Preview" 
                                        className="w-full h-40 object-cover rounded-md" 
                                        onError={(e) => {
                                            console.error('خطا در بارگذاری تصویر');
                                            (e.target as HTMLImageElement).src = 'https://via.placeholder.com/400x200?text=Error+Loading+Image';
                                        }}
                                    />
                                    <button 
                                        type="button" 
                                        onClick={removeImage} 
                                        className="absolute top-2 right-2 bg-red-500 text-white rounded-full p-1.5 opacity-0 group-hover:opacity-100 transition-opacity shadow-lg hover:bg-red-600"
                                        title="حذف تصویر"
                                    >
                                        <XIcon className="w-4 h-4" />
                                    </button>
                                </div>
                            ) : (
                                <label 
                                    htmlFor="article-image-upload" 
                                    className="flex flex-col items-center justify-center w-full h-40 border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-lg cursor-pointer bg-custom-light-bg dark:bg-custom-dark-blue hover:bg-gray-100 dark:hover:bg-gray-800 transition-colors"
                                >
                                    <div className="flex flex-col items-center justify-center">
                                        <UploadCloudIcon className="w-8 h-8 mb-2 text-gray-400" />
                                        <p className="text-xs text-gray-500 dark:text-gray-400">آپلود تصویر</p>
                                        <p className="text-xs text-gray-400 dark:text-gray-500 mt-1">(حداکثر 2MB)</p>
                                    </div>
                                    <input 
                                        id="article-image-upload" 
                                        type="file" 
                                        className="hidden" 
                                        accept="image/*" 
                                        onChange={handleImageChange} 
                                    />
                                </label>
                            )}
                        </div>
                        
                        <button 
                            type="submit" 
                            disabled={loading}
                            className={`w-full font-bold py-3 px-4 rounded-lg transition-colors ${
                                loading 
                                    ? 'bg-gray-400 cursor-not-allowed text-gray-200' 
                                    : 'bg-custom-purple hover:bg-custom-light-purple text-white'
                            }`}
                        >
                            {loading ? (
                                <span className="flex items-center justify-center gap-2">
                                    <span className="inline-block animate-spin rounded-full h-4 w-4 border-b-2 border-white"></span>
                                    در حال پردازش...
                                </span>
                            ) : (
                                isEditing ? 'ذخیره تغییرات' : 'ذخیره مقاله'
                            )}
                        </button>
                    </div>
                </form>
            )}
        </div>
    );
};

export default AddOrEditArticlePage;
