// src/components/pages/AllRequestsListPage.tsx
import React, { useState, useEffect, useRef } from 'react';
import { SearchIcon, ChevronDownIcon, PackageIcon, PlaneIcon, ECommerceIcon, UserIcon } from '../Icons';
import { API_BASE_URL } from '@/src/config.js';

type RequestType = 'shipment' | 'trip' | 'purchase' | 'shopper' | 'ticket';
type RequestStatus = 'pending' | 'approved' | 'rejected';

interface Request {
    id: string;
    type: RequestType;
    request_id: number;
    title: string;
    user: { name: string; avatar: string };
    status: RequestStatus;
    datePosted: string;
    originCountry: string;
    destinationCountry: string;
}

interface FilterDropdownProps {
  options: { value: string; label: string }[];
  value: string;
  onChange: (value: string) => void;
  placeholder: string;
}

const FilterDropdown: React.FC<FilterDropdownProps> = ({ options, value, onChange, placeholder }) => {
  const [isOpen, setIsOpen] = useState(false);
  const dropdownRef = useRef<HTMLDivElement>(null);
  const selectedOption = options.find(opt => opt.value === value);

  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (dropdownRef.current && !dropdownRef.current.contains(event.target as Node)) {
        setIsOpen(false);
      }
    };
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, []);

  return (
    <div className="relative" ref={dropdownRef}>
      <button
        type="button"
        onClick={() => setIsOpen(!isOpen)}
        className="w-full flex items-center justify-between text-right pr-4 pl-10 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition"
      >
        <span className={selectedOption ? "text-gray-800 dark:text-gray-200" : "text-gray-500"}>
          {selectedOption ? selectedOption.label : placeholder}
        </span>
        <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
            <ChevronDownIcon className={`w-5 h-5 text-gray-400 transition-transform duration-200 ${isOpen ? 'rotate-180' : ''}`} />
        </div>
      </button>
      {isOpen && (
        <ul className="absolute z-20 w-full mt-1 bg-custom-light-card dark:bg-custom-dark-blue-light border border-gray-300 dark:border-gray-600 rounded-lg shadow-lg max-h-60 overflow-y-auto custom-scrollbar">
          {options.map(option => (
            <li
              key={option.value}
              onClick={() => {
                onChange(option.value);
                setIsOpen(false);
              }}
              className="px-4 py-2 cursor-pointer hover:bg-gray-100 dark:hover:bg-custom-dark-blue text-gray-800 dark:text-gray-200 transition-colors"
            >
              {option.label}
            </li>
          ))}
        </ul>
      )}
      
      {/* استایل اسکرول */}
      <style>{`
        .custom-scrollbar::-webkit-scrollbar {
          width: 6px;
        }
        .custom-scrollbar::-webkit-scrollbar-track {
          background: transparent;
        }
        .custom-scrollbar::-webkit-scrollbar-thumb {
          background: rgb(147 51 234 / 0.3);
          border-radius: 3px;
        }
        .custom-scrollbar::-webkit-scrollbar-thumb:hover {
          background: rgb(147 51 234 / 0.5);
        }
        .dark .custom-scrollbar::-webkit-scrollbar-thumb {
          background: rgb(147 51 234 / 0.4);
        }
        .dark .custom-scrollbar::-webkit-scrollbar-thumb:hover {
          background: rgb(147 51 234 / 0.6);
        }
      `}</style>
    </div>
  );
};

interface AllRequestsListPageProps {
    onNavigate: (page: string, payload?: any) => void;
    showError: (message: string) => void;
}

const AllRequestsListPage: React.FC<AllRequestsListPageProps> = ({ onNavigate, showError }) => {
    const [searchQuery, setSearchQuery] = useState('');
    const [typeFilter, setTypeFilter] = useState('all');
    const [statusFilter, setStatusFilter] = useState('all');
    const [countryFilter, setCountryFilter] = useState('all');
    const [requests, setRequests] = useState<Request[]>([]);
    const [loading, setLoading] = useState(true);

    const fetchRequests = async () => {
        setLoading(true);
        try {
            const token = localStorage.getItem('auth_token');
            if (!token) {
                showError('لطفاً دوباره وارد شوید.');
                return;
            }

            const response = await fetch(`${API_BASE_URL}/admin/all-requests`, {
                headers: {
                    'X-Auth-Token': token,
                },
            });

            if (!response.ok) {
                const errorData = await response.json().catch(() => ({}));
                throw new Error(errorData.error || 'خطا در دریافت درخواست‌ها');
            }

            const data = await response.json();
            setRequests(data);
        } catch (error: any) {
            console.error('Fetch all requests error:', error);
            showError(error.message || 'خطا در ارتباط با سرور');
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchRequests();
    }, []);

    const countries = Array.from(
        new Set(requests.flatMap(r => [r.originCountry, r.destinationCountry]))
    ).sort();

    const filteredRequests = requests.filter(req => {
        const lowercasedQuery = searchQuery.toLowerCase();
        const matchesSearch = 
            req.title.toLowerCase().includes(lowercasedQuery) || 
            req.user.name.toLowerCase().includes(lowercasedQuery) || 
            req.id.toLowerCase().includes(lowercasedQuery);
        const matchesType = typeFilter === 'all' || req.type === typeFilter;
        const matchesStatus = statusFilter === 'all' || req.status === statusFilter;
        const matchesCountry = countryFilter === 'all' || 
            req.originCountry === countryFilter || 
            req.destinationCountry === countryFilter;
        return matchesSearch && matchesType && matchesStatus && matchesCountry;
    });

    const getStatusChip = (status: RequestStatus) => {
        switch (status) {
            case 'pending': return <span className="px-3 py-1 text-xs font-bold rounded-full bg-yellow-500/10 text-yellow-500">در انتظار</span>;
            case 'approved': return <span className="px-3 py-1 text-xs font-bold rounded-full bg-green-500/10 text-green-500">تایید شده</span>;
            case 'rejected': return <span className="px-3 py-1 text-xs font-bold rounded-full bg-red-500/10 text-red-500">رد شده</span>;
        }
    };
    
    const getTypeChip = (type: RequestType) => {
        switch (type) {
            case 'shipment': return <div className="flex items-center gap-2"><PackageIcon className="w-5 h-5 text-custom-cyan" /><span>مرسوله</span></div>;
            case 'trip': return <div className="flex items-center gap-2"><PlaneIcon className="w-5 h-5 text-custom-orange" /><span>سفر/همسفر</span></div>;
            case 'purchase': case 'shopper': return <div className="flex items-center gap-2"><ECommerceIcon className="w-5 h-5 text-custom-purple" /><span>خرید</span></div>;
            case 'ticket': return <div className="flex items-center gap-2"><UserIcon className="w-5 h-5 text-custom-purple" /><span>تیکت</span></div>;
        }
    };

    const handleViewDetails = (req: Request) => {
        onNavigate('all-request-detail', { id: req.request_id, type: req.type });
    };

    const typeOptions = [
        { value: 'all', label: 'همه انواع' },
        { value: 'shipment', label: 'مرسوله' },
        { value: 'trip', label: 'سفر/همسفر' },
        { value: 'purchase', label: 'خرید' },
       
    ];
    
    const statusOptions = [
        { value: 'all', label: 'همه وضعیت‌ها' },
        { value: 'pending', label: 'در انتظار' },
        { value: 'approved', label: 'تایید شده' },
        { value: 'rejected', label: 'رد شده' },
    ];

    // const countryOptions = [
    //     { value: 'all', label: 'همه کشورها' },
    //     ...countries.map(c => ({ value: c, label: c })),
    // ];

    return (
        <div className="flex flex-col space-y-8">
            <div>
                <h1 className="text-3xl font-bold text-gray-800 dark:text-white">لیست تمام درخواست‌ها</h1>
                <p className="text-gray-500 dark:text-gray-400 mt-1">تمام درخواست‌های ثبت شده توسط کاربران را مشاهده و مدیریت کنید.</p>
            </div>

            {loading ? (
                <div className="flex items-center justify-center py-16">
                    <div className="animate-spin rounded-full h-12 w-12 border-4 border-custom-purple border-t-transparent"></div>
                </div>
            ) : (
                <>
                    <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
                        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
                            <div className="relative sm:col-span-2 lg:col-span-1">
                                <input 
                                    type="text" 
                                    placeholder="جستجو (عنوان، کاربر، شناسه)..." 
                                    className="w-full pr-10 pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition" 
                                    value={searchQuery} 
                                    onChange={(e) => setSearchQuery(e.target.value)}
                                />
                                <SearchIcon className="absolute right-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400"/>
                            </div>
                            <FilterDropdown options={typeOptions} value={typeFilter} onChange={setTypeFilter} placeholder="نوع درخواست" />
                            <FilterDropdown options={statusOptions} value={statusFilter} onChange={setStatusFilter} placeholder="وضعیت درخواست" />
                           
                        </div>
                    </div>

                    <div className="bg-custom-light-card dark:bg-custom-dark-blue-light rounded-2xl shadow-sm overflow-hidden">
                        <div className="overflow-x-auto">
                            <table className="w-full text-sm text-right">
                                <thead className="bg-custom-light-bg dark:bg-custom-dark-blue text-gray-500 dark:text-gray-400">
                                    <tr>
                                        <th className="p-4 font-semibold">شناسه</th>
                                        <th className="p-4 font-semibold">عنوان</th>
                                        <th className="p-4 font-semibold">کاربر</th>
                                        <th className="p-4 font-semibold hidden sm:table-cell">نوع</th>
                                        <th className="p-4 font-semibold hidden md:table-cell">تاریخ ثبت</th>
                                        <th className="p-4 font-semibold">وضعیت</th>
                                        <th className="p-4 font-semibold"></th>
                                    </tr>
                                </thead>
                                <tbody className="divide-y divide-gray-200 dark:divide-gray-700">
                                    {filteredRequests.length > 0 ? (
                                        filteredRequests.map(req => (
                                            <tr key={req.id} className="hover:bg-gray-50 dark:hover:bg-custom-dark-blue">
                                                <td className="p-4 font-mono text-custom-purple">{req.id}</td>
                                                <td className="p-4 font-semibold text-gray-800 dark:text-gray-200">{req.title}</td>
                                                <td className="p-4">
                                                    <div className="flex items-center gap-2">
                                                        <img src={req.user.avatar} alt={req.user.name} className="w-8 h-8 rounded-full"/>
                                                        <span>{req.user.name}</span>
                                                    </div>
                                                </td>
                                                <td className="p-4 hidden sm:table-cell">{getTypeChip(req.type)}</td>
                                                <td className="p-4 hidden md:table-cell">{req.datePosted}</td>
                                                <td className="p-4">{getStatusChip(req.status)}</td>
                                                <td className="p-4">
                                                    <button 
                                                        onClick={() => handleViewDetails(req)} 
                                                        className="text-custom-purple hover:underline font-semibold"
                                                    >
                                                        مشاهده
                                                    </button>
                                                </td>
                                            </tr>
                                        ))
                                    ) : (
                                        <tr>
                                            <td colSpan={7} className="text-center p-8 text-gray-500 dark:text-gray-400">
                                                هیچ درخواستی با این مشخصات یافت نشد.
                                            </td>
                                        </tr>
                                    )}
                                </tbody>
                            </table>
                        </div>
                    </div>
                </>
            )}
        </div>
    );
};

export default AllRequestsListPage;
