// src/components/pages/ArticleCategoriesPage.tsx
import React, { useState, useEffect } from 'react';
import { UploadCloudIcon, XIcon, PencilIcon, Trash2Icon, TagIcon } from '../Icons';
import { API_BASE_URL } from '@/src/config.js';
import { authFetch } from '@/App';

interface Category {
    id: number;
    name: string;
    imageUrl: string;
}

interface ArticleCategoriesPageProps {
    showError: (msg: string) => void;
    showSuccess: (msg: string) => void;
}

const ArticleCategoriesPage: React.FC<ArticleCategoriesPageProps> = ({ showError, showSuccess }) => {
    const [categories, setCategories] = useState<Category[]>([]);
    const [newCategoryName, setNewCategoryName] = useState('');
    const [newCategoryImage, setNewCategoryImage] = useState<File | null>(null);
    const [previewUrl, setPreviewUrl] = useState<string | null>(null);
    const [loading, setLoading] = useState(false);

    const [isEditModalOpen, setIsEditModalOpen] = useState(false);
    const [editingCategory, setEditingCategory] = useState<Category | null>(null);
    const [editCategoryName, setEditCategoryName] = useState('');
    const [editCategoryImage, setEditCategoryImage] = useState<File | null>(null);
    const [editPreviewUrl, setEditPreviewUrl] = useState<string | null>(null);
    const [editLoading, setEditLoading] = useState(false);

    const fetchCategories = async () => {
        try {
            const res = await authFetch(`${API_BASE_URL}/admin/article-categories`);
            const data = await res.json();
            const cats = Array.isArray(data.data) ? data.data : data;
            setCategories(cats.map((c: any) => ({
                id: c.id,
                name: c.name,
                imageUrl: c.image_url
                    ? `http://127.0.0.1:8000/${c.image_url}` // ✅ بدون /storage/
                    : 'https://via.placeholder.com/400x200?text=No+Image',
            })));
        } catch (err: any) {
            console.error(err);
            showError('خطا در بارگذاری دسته‌بندی‌ها');
        }
    };

    useEffect(() => {
        fetchCategories();
    }, []);

    const handleImageChange = (event: React.ChangeEvent<HTMLInputElement>) => {
        const file = event.target.files?.[0];
        if (file) {
            if (file.size > 2 * 1024 * 1024) {
                showError('حجم فایل نباید بیشتر از 2 مگابایت باشد.');
                return;
            }
            if (!file.type.startsWith('image/')) {
                showError('فقط فایل‌های تصویری مجاز هستند.');
                return;
            }
            setNewCategoryImage(file);
            setPreviewUrl(URL.createObjectURL(file));
        }
    };

    const handleEditImageChange = (event: React.ChangeEvent<HTMLInputElement>) => {
        const file = event.target.files?.[0];
        if (file) {
            if (file.size > 2 * 1024 * 1024) {
                showError('حجم فایل نباید بیشتر از 2 مگابایت باشد.');
                return;
            }
            if (!file.type.startsWith('image/')) {
                showError('فقط فایل‌های تصویری مجاز هستند.');
                return;
            }
            setEditCategoryImage(file);
            setEditPreviewUrl(URL.createObjectURL(file));
        }
    };

    const removeImage = () => {
        if (previewUrl) URL.revokeObjectURL(previewUrl);
        setNewCategoryImage(null);
        setPreviewUrl(null);
    };

    const removeEditImage = () => {
        if (editPreviewUrl) URL.revokeObjectURL(editPreviewUrl);
        setEditCategoryImage(null);
        setEditPreviewUrl(null);
    };

    const handleAddCategory = async (e: React.FormEvent) => {
        e.preventDefault();
        if (!newCategoryName.trim() || !newCategoryImage) {
            showError('لطفا نام و تصویر دسته را انتخاب کنید.');
            return;
        }

        const formData = new FormData();
        formData.append('name', newCategoryName);
        formData.append('image', newCategoryImage);

        try {
            setLoading(true);
            await authFetch(`${API_BASE_URL}/admin/article-categories`, {
                method: 'POST',
                body: formData,
            });
            showSuccess('دسته‌بندی جدید با موفقیت اضافه شد.');
            fetchCategories();
            setNewCategoryName('');
            setNewCategoryImage(null);
            setPreviewUrl(null);
        } catch (err: any) {
            console.error(err);
            showError(err.message || 'خطا در ایجاد دسته‌بندی');
        } finally {
            setLoading(false);
        }
    };

    const openEditModal = (category: Category) => {
        setEditingCategory(category);
        setEditCategoryName(category.name);
        setEditPreviewUrl(category.imageUrl);
        setEditCategoryImage(null);
        setIsEditModalOpen(true);
    };

    const closeEditModal = () => {
        setIsEditModalOpen(false);
        setEditingCategory(null);
        setEditCategoryName('');
        setEditCategoryImage(null);
        if (editPreviewUrl) {
            URL.revokeObjectURL(editPreviewUrl);
        }
        setEditPreviewUrl(null);
    };

    const handleEditCategory = async (e: React.FormEvent) => {
        e.preventDefault();
        if (!editingCategory || !editCategoryName.trim()) {
            showError('لطفا نام دسته را وارد کنید.');
            return;
        }

        const formData = new FormData();
        formData.append('name', editCategoryName);
        formData.append('_method', 'PUT');

        if (editCategoryImage) {
            formData.append('image', editCategoryImage);
        }

        try {
            setEditLoading(true);
            await authFetch(`${API_BASE_URL}/admin/article-categories/${editingCategory.id}`, {
                method: 'POST',
                body: formData,
            });
            showSuccess('دسته‌بندی با موفقیت ویرایش شد.');
            fetchCategories();
            closeEditModal();
        } catch (err: any) {
            console.error(err);
            showError(err.message || 'خطا در ویرایش دسته‌بندی');
        } finally {
            setEditLoading(false);
        }
    };

    const handleDelete = async (id: number) => {
        if (!confirm('آیا از حذف این دسته‌بندی اطمینان دارید؟')) return;
        try {
            await authFetch(`${API_BASE_URL}/admin/article-categories/${id}`, {
                method: 'DELETE',
            });
            showSuccess('دسته‌بندی با موفقیت حذف شد.');
            fetchCategories();
        } catch (err: any) {
            showError(err.message || 'خطا در حذف دسته‌بندی');
        }
    };

    return (
        <div className="flex flex-col space-y-8">
            <div>
                <h1 className="text-3xl font-bold text-gray-800 dark:text-white">مدیریت دسته‌بندی مقالات</h1>
                <p className="text-gray-500 dark:text-gray-400 mt-1">دسته‌بندی‌های مقالات را ایجاد و مدیریت کنید.</p>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
                <div className="lg:col-span-2 bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
                    <h3 className="text-xl font-semibold mb-4">لیست دسته‌بندی‌ها</h3>
                    {categories.length === 0 ? (
                        <p className="text-gray-500 dark:text-gray-400 text-center py-4">هیچ دسته‌بندی‌ای وجود ندارد.</p>
                    ) : (
                        <div className="space-y-4">
                            {categories.map(cat => (
                                <div key={cat.id} className="flex items-center justify-between p-3 bg-custom-light-bg dark:bg-custom-dark-blue rounded-lg">
                                    <div className="flex items-center gap-4">
                                        <img
                                            src={cat.imageUrl}
                                            alt={cat.name}
                                            className="w-16 h-12 object-cover rounded-md"
                                        />
                                        <span className="font-semibold text-gray-800 dark:text-gray-200">{cat.name}</span>
                                    </div>
                                    <div className="flex items-center gap-3">
                                        <button
                                            onClick={() => openEditModal(cat)}
                                            className="p-2 text-blue-500 hover:bg-blue-500/10 rounded-full transition-colors"
                                        >
                                            <PencilIcon className="w-5 h-5" />
                                        </button>
                                        <button
                                            onClick={() => handleDelete(cat.id)}
                                            className="p-2 text-red-500 hover:bg-red-500/10 rounded-full transition-colors"
                                        >
                                            <Trash2Icon className="w-5 h-5" />
                                        </button>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>

                <div className="lg:col-span-1 bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
                    <h3 className="text-xl font-semibold mb-4">افزودن دسته جدید</h3>
                    <form onSubmit={handleAddCategory} className="space-y-6">
                        <div>
                            <label htmlFor="category_name" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">عنوان دسته</label>
                            <div className="relative">
                                <div className="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none"><TagIcon className="w-5 h-5 text-gray-400" /></div>
                                <input
                                    type="text"
                                    id="category_name"
                                    value={newCategoryName}
                                    onChange={e => setNewCategoryName(e.target.value)}
                                    className="w-full pr-10 pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition"
                                    placeholder="مثال: اخبار"
                                />
                            </div>
                        </div>
                        <div>
                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">تصویر دسته</label>
                            {previewUrl ? (
                                <div className="relative group">
                                    <img
                                        src={previewUrl}
                                        alt="Preview"
                                        className="w-full h-32 object-cover rounded-md"
                                    />
                                    <button
                                        onClick={removeImage}
                                        type="button"
                                        className="absolute top-2 right-2 bg-red-500 text-white rounded-full p-1 opacity-0 group-hover:opacity-100 transition-opacity"
                                    >
                                        <XIcon className="w-4 h-4" />
                                    </button>
                                </div>
                            ) : (
                                <label htmlFor="category-image-upload" className="flex flex-col items-center justify-center w-full h-32 border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-lg cursor-pointer bg-custom-light-bg dark:bg-custom-dark-blue hover:bg-gray-100 dark:hover:bg-gray-800 transition-colors">
                                    <div className="flex flex-col items-center justify-center">
                                        <UploadCloudIcon className="w-8 h-8 mb-2 text-gray-400" />
                                        <p className="text-xs text-gray-500 dark:text-gray-400">آپلود تصویر (حداکثر 2MB)</p>
                                    </div>
                                    <input
                                        id="category-image-upload"
                                        type="file"
                                        className="hidden"
                                        accept="image/*"
                                        onChange={handleImageChange}
                                    />
                                </label>
                            )}
                        </div>
                        <button
                            type="submit"
                            disabled={loading}
                            className={`w-full font-bold py-2.5 px-4 rounded-lg transition-colors ${
                                loading
                                    ? 'bg-gray-400 cursor-not-allowed'
                                    : 'bg-custom-purple hover:bg-custom-light-purple text-white'
                            }`}
                        >
                            {loading ? 'در حال ارسال...' : 'افزودن'}
                        </button>
                    </form>
                </div>
            </div>

            {/* مدال ویرایش */}
            {isEditModalOpen && (
                <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4" onClick={closeEditModal}>
                    <div className="bg-custom-light-card dark:bg-custom-dark-blue-light rounded-2xl p-6 max-w-md w-full shadow-2xl" onClick={(e) => e.stopPropagation()}>
                        <div className="flex items-center justify-between mb-6">
                            <h2 className="text-2xl font-bold text-gray-800 dark:text-white">ویرایش دسته‌بندی</h2>
                            <button
                                onClick={closeEditModal}
                                className="text-gray-500 hover:text-gray-700 dark:hover:text-gray-300 transition-colors"
                            >
                                <XIcon className="w-6 h-6" />
                            </button>
                        </div>

                        <form onSubmit={handleEditCategory} className="space-y-6">
                            <div>
                                <label htmlFor="edit_category_name" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">عنوان دسته</label>
                                <div className="relative">
                                    <div className="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none">
                                        <TagIcon className="w-5 h-5 text-gray-400" />
                                    </div>
                                    <input
                                        type="text"
                                        id="edit_category_name"
                                        value={editCategoryName}
                                        onChange={e => setEditCategoryName(e.target.value)}
                                        className="w-full pr-10 pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition"
                                        placeholder="نام دسته را وارد کنید"
                                    />
                                </div>
                            </div>

                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">تصویر دسته</label>
                                {editPreviewUrl ? (
                                    <div className="relative group">
                                        <img
                                            src={editPreviewUrl}
                                            alt="Preview"
                                            className="w-full h-40 object-cover rounded-md"
                                        />
                                        {editCategoryImage && (
                                            <button
                                                onClick={removeEditImage}
                                                type="button"
                                                className="absolute top-2 right-2 bg-red-500 text-white rounded-full p-1 opacity-0 group-hover:opacity-100 transition-opacity"
                                            >
                                                <XIcon className="w-4 h-4" />
                                            </button>
                                        )}
                                        <label htmlFor="edit-category-image-upload" className="absolute bottom-2 right-2 bg-custom-purple text-white px-3 py-1 rounded-lg text-sm cursor-pointer hover:bg-custom-light-purple transition-colors">
                                            تغییر تصویر
                                        </label>
                                    </div>
                                ) : (
                                    <label htmlFor="edit-category-image-upload" className="flex flex-col items-center justify-center w-full h-40 border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-lg cursor-pointer bg-custom-light-bg dark:bg-custom-dark-blue hover:bg-gray-100 dark:hover:bg-gray-800 transition-colors">
                                        <div className="flex flex-col items-center justify-center">
                                            <UploadCloudIcon className="w-8 h-8 mb-2 text-gray-400" />
                                            <p className="text-xs text-gray-500 dark:text-gray-400">آپلود تصویر جدید (حداکثر 2MB)</p>
                                        </div>
                                    </label>
                                )}
                                <input
                                    id="edit-category-image-upload"
                                    type="file"
                                    className="hidden"
                                    accept="image/*"
                                    onChange={handleEditImageChange}
                                />
                            </div>

                            <div className="flex gap-3">
                                <button
                                    type="button"
                                    onClick={closeEditModal}
                                    className="flex-1 font-bold py-2.5 px-4 rounded-lg border border-gray-300 dark:border-gray-600 text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-800 transition-colors"
                                >
                                    انصراف
                                </button>
                                <button
                                    type="submit"
                                    disabled={editLoading}
                                    className={`flex-1 font-bold py-2.5 px-4 rounded-lg transition-colors ${
                                        editLoading
                                            ? 'bg-gray-400 cursor-not-allowed'
                                            : 'bg-custom-purple hover:bg-custom-light-purple text-white'
                                    }`}
                                >
                                    {editLoading ? 'در حال ذخیره...' : 'ذخیره تغییرات'}
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            )}
        </div>
    );
};

export default ArticleCategoriesPage;