// src/components/pages/ArticlesListPage.tsx
import React, { useState, useEffect } from 'react';
import { PlusCircleIcon, SearchIcon, PencilIcon, Trash2Icon } from '../Icons';
import { API_BASE_URL } from '@/src/config.js';
import { authFetch } from '@/App';

interface Article {
    id: string;
    title: string;
    category: string;
    publishDate: string;
    status: 'published' | 'draft';
    imageUrl: string;
}

interface ArticlesListPageProps {
    onNavigate: (page: string, payload?: any) => void;
    showError: (msg: string) => void;
    showSuccess: (msg: string) => void;
}

const ArticlesListPage: React.FC<ArticlesListPageProps> = ({ onNavigate, showError, showSuccess }) => {
    const [searchQuery, setSearchQuery] = useState('');
    const [articles, setArticles] = useState<Article[]>([]);
    const [loading, setLoading] = useState(true);

    const fetchArticles = async (query = '') => {
        try {
            setLoading(true);
            const params = new URLSearchParams();
            if (query) params.append('search', query);
            const url = `${API_BASE_URL}/admin/articles?${params.toString()}`;
            
            const res = await authFetch(url);
            const responseData = await res.json();
            
            // دریافت آرایه مقالات از response
            const articlesData = responseData.data || [];
            
            setArticles(articlesData.map((item: any) => {
                // تبدیل تاریخ ISO به فرمت قابل نمایش
                let publishDate = '-';
                if (item.created_at) {
                    try {
                        const date = new Date(item.created_at);
                        publishDate = date.toLocaleDateString('fa-IR');
                    } catch (e) {
                        publishDate = item.created_at.split('T')[0];
                    }
                }

                return {
                    id: String(item.id),
                    title: item.title || 'بدون عنوان',
                    category: item.category?.name || 'دسته‌بندی نامشخص',
                    publishDate: publishDate,
                    status: item.status || 'draft',
                    imageUrl: item.image_url 
                        ? `http://127.0.0.1:8000/${item.image_url.replace(/^\/+/, '')}`
                        : 'https://via.placeholder.com/400x200?text=No+Image',
                };
            }));
        } catch (err: any) {
            console.error('خطا در بارگذاری مقالات:', err);
            showError(err.message || 'خطا در بارگذاری مقالات');
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchArticles();
    }, []);

    useEffect(() => {
        const timer = setTimeout(() => {
            fetchArticles(searchQuery);
        }, 300);
        return () => clearTimeout(timer);
    }, [searchQuery]);

    const handleDelete = async (id: string) => {
        if (!confirm('آیا از حذف این مقاله اطمینان دارید؟')) return;
        
        try {
            await authFetch(`${API_BASE_URL}/admin/articles/${id}`, {
                method: 'DELETE',
            });
            showSuccess('مقاله با موفقیت حذف شد.');
            fetchArticles(searchQuery);
        } catch (err: any) {
            showError(err.message || 'خطا در حذف مقاله');
        }
    };

    const getStatusChip = (status: Article['status']) => {
        switch (status) {
            case 'published':
                return <span className="px-3 py-1 text-xs font-bold rounded-full bg-green-500/10 text-green-500">منتشر شده</span>;
            case 'draft':
                return <span className="px-3 py-1 text-xs font-bold rounded-full bg-gray-500/10 text-gray-500 dark:bg-gray-400/10 dark:text-gray-400">پیش‌نویس</span>;
        }
    };

    return (
        <div className="flex flex-col space-y-8">
            <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
                <div>
                    <h1 className="text-3xl font-bold text-gray-800 dark:text-white">مدیریت مقالات</h1>
                    <p className="text-gray-500 dark:text-gray-400 mt-1">مقالات وب‌سایت را ویرایش یا حذف کنید.</p>
                </div>
                <button 
                    onClick={() => onNavigate('add-article')} 
                    className="flex items-center justify-center gap-2 bg-custom-purple hover:bg-custom-light-purple text-white font-bold py-2.5 px-6 rounded-lg transition-colors duration-300"
                >
                    <PlusCircleIcon className="w-5 h-5"/>
                    <span>افزودن مقاله جدید</span>
                </button>
            </div>

            <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
                <div className="relative max-w-sm">
                    <input 
                        type="text" 
                        placeholder="جستجوی عنوان مقاله..." 
                        className="w-full pr-10 pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition"
                        value={searchQuery}
                        onChange={(e) => setSearchQuery(e.target.value)}
                    />
                    <SearchIcon className="absolute right-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400"/>
                </div>
            </div>

            {loading ? (
                <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl text-center">
                    در حال بارگذاری...
                </div>
            ) : (
                <div className="bg-custom-light-card dark:bg-custom-dark-blue-light rounded-2xl shadow-sm overflow-hidden">
                    <div className="overflow-x-auto">
                        <table className="w-full text-sm text-right">
                            <thead className="bg-custom-light-bg dark:bg-custom-dark-blue text-gray-500 dark:text-gray-400">
                                <tr>
                                    <th className="p-4 font-semibold">تصویر</th>
                                    <th className="p-4 font-semibold">عنوان</th>
                                    <th className="p-4 font-semibold hidden sm:table-cell">دسته‌بندی</th>
                                    <th className="p-4 font-semibold hidden md:table-cell">تاریخ انتشار</th>
                                    <th className="p-4 font-semibold">وضعیت</th>
                                    <th className="p-4 font-semibold">عملیات</th>
                                </tr>
                            </thead>
                            <tbody className="divide-y divide-gray-200 dark:divide-gray-700">
                                {articles.length > 0 ? (
                                    articles.map(article => (
                                        <tr key={article.id} className="hover:bg-gray-50 dark:hover:bg-custom-dark-blue">
                                            <td className="p-4">
                                                <img 
                                                    src={article.imageUrl} 
                                                    alt={article.title} 
                                                    className="w-12 h-12 object-cover rounded-md"
                                                    
                                                />
                                            </td>
                                            <td className="p-4 font-semibold text-gray-800 dark:text-gray-200">{article.title}</td>
                                            <td className="p-4 hidden sm:table-cell text-gray-600 dark:text-gray-300">{article.category}</td>
                                            <td className="p-4 hidden md:table-cell text-gray-600 dark:text-gray-300">{article.publishDate}</td>
                                            <td className="p-4">{getStatusChip(article.status)}</td>
                                            <td className="p-4">
                                                <div className="flex gap-2">
                                                    <button 
                                                        onClick={() => onNavigate('edit-article', article.id)} 
                                                        className="p-2 text-custom-cyan hover:bg-custom-cyan/10 rounded-full transition-colors"
                                                        title="ویرایش"
                                                    >
                                                        <PencilIcon className="w-5 h-5"/>
                                                    </button>
                                                    <button 
                                                        onClick={() => handleDelete(article.id)}
                                                        className="p-2 text-red-500 hover:bg-red-500/10 rounded-full transition-colors"
                                                        title="حذف"
                                                    >
                                                        <Trash2Icon className="w-5 h-5"/>
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                    ))
                                ) : (
                                    <tr>
                                        <td colSpan={6} className="text-center p-8 text-gray-500 dark:text-gray-400">
                                            مقاله‌ای یافت نشد.
                                        </td>
                                    </tr>
                                )}
                            </tbody>
                        </table>
                    </div>
                </div>
            )}
        </div>
    );
};

export default ArticlesListPage;
