// src/components/pages/DashboardPage.tsx
import React, { useEffect, useState } from 'react';
import { UserIcon, PackageIcon, PlaneIcon, TicketIcon, ShoppingCartIcon, CheckCircleIcon, XCircleIcon, ClockIcon } from '../Icons';
import { API_BASE_URL } from '@/src/config.js';

interface Stat {
  title: string;
  value: string;
  icon: React.FC<{ className?: string }>;
  change: string;
  changeType: 'increase' | 'decrease';
}

interface PendingRequest {
  id: string;
  request_id: number;
  type: 'shipment' | 'trip' | 'companion' | 'purchase' | 'shopper';
  type_label: string;
  user: string;
  detail: string;
  created_at: string;
}

interface RecentActivity {
  type: string;
  description: string;
  time: string;
  user: { name: string; avatar: string };
}

const AdminDashboardPage: React.FC = () => {
  const [stats, setStats] = useState<Stat[]>([]);
  const [pendingRequests, setPendingRequests] = useState<PendingRequest[]>([]);
  const [recentActivities, setRecentActivities] = useState<RecentActivity[]>([]);
  const [loading, setLoading] = useState(true);
  const [refreshKey, setRefreshKey] = useState(0);

  const [showRejectDialog, setShowRejectDialog] = useState(false);
  const [selectedRequest, setSelectedRequest] = useState<PendingRequest | null>(null);
  const [rejectReason, setRejectReason] = useState('');
  const [isProcessing, setIsProcessing] = useState(false);

  const handleApprove = async (request: PendingRequest) => {
    if (!confirm('آیا از تأیید این درخواست اطمینان دارید؟')) return;

    const token = localStorage.getItem('auth_token');
    if (!token) return;

    const typeToRoute: Record<string, string> = {
      'shipment': 'shipments',
      'trip': 'trips',
      'companion': 'trip-companions',
      'purchase': 'online-purchases',
      'shopper': 'online-shoppers',
    };

    const route = typeToRoute[request.type];
    if (!route) {
      alert('نوع درخواست ناشناخته است.');
      return;
    }

    setIsProcessing(true);
    try {
      const url = `${API_BASE_URL}/admin/${route}/${request.request_id}/approve`;
      const response = await fetch(url, {
        method: 'PUT',
        headers: {
          'X-Auth-Token': token,
          'Content-Type': 'application/json',
        },
      });

      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.message || 'خطا در تأیید درخواست');
      }

      setRefreshKey(prev => prev + 1);
    } catch (error: any) {
      console.error('Approve failed:', error);
      alert('خطا در تأیید درخواست: ' + (error.message || 'خطای نامشخص'));
    } finally {
      setIsProcessing(false);
    }
  };

  const openRejectDialog = (request: PendingRequest) => {
    setSelectedRequest(request);
    setShowRejectDialog(true);
  };

  const handleReject = async () => {
    if (!selectedRequest) return;
    if (!rejectReason.trim()) {
      alert('لطفاً دلیل رد کردن را وارد کنید.');
      return;
    }

    const token = localStorage.getItem('auth_token');
    if (!token) return;

    const typeToRoute: Record<string, string> = {
      'shipment': 'shipments',
      'trip': 'trips',
      'companion': 'trip-companions',
      'purchase': 'online-purchases',
      'shopper': 'online-shoppers',
    };

    const route = typeToRoute[selectedRequest.type];
    if (!route) {
      alert('نوع درخواست ناشناخته است.');
      return;
    }

    setIsProcessing(true);
    try {
      const url = `${API_BASE_URL}/admin/${route}/${selectedRequest.request_id}/reject`;
      const response = await fetch(url, {
        method: 'PUT',
        headers: {
          'X-Auth-Token': token,
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ reason: rejectReason.trim() }),
      });

      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.message || 'خطا در رد درخواست');
      }

      setShowRejectDialog(false);
      setSelectedRequest(null);
      setRejectReason('');
      setRefreshKey(prev => prev + 1);
    } catch (error: any) {
      console.error('Reject failed:', error);
      alert('خطا در رد درخواست: ' + (error.message || 'خطای نامشخص'));
    } finally {
      setIsProcessing(false);
    }
  };

  useEffect(() => {
    const fetchDashboardData = async () => {
      const token = localStorage.getItem('auth_token');
      if (!token) {
        setLoading(false);
        return;
      }

      try {
        const response = await fetch(`${API_BASE_URL}/admin/dashboard`, {
          headers: {
            'X-Auth-Token': token,
            'Content-Type': 'application/json',
          },
        });

        if (!response.ok) throw new Error('خطا در دریافت داده‌های داشبورد');

        const data = await response.json();

        const { 
          total_users, 
          pending_shipments, 
          pending_trips, 
          pending_companions,
          pending_purchases, 
          pending_shoppers,
          open_tickets 
        } = data.stats;

        const totalPendingTrips = pending_trips + pending_companions;
        const totalPendingPurchases = pending_purchases + pending_shoppers;

        setStats([
          { 
            title: 'کل کاربران', 
            value: new Intl.NumberFormat('fa-IR').format(total_users), 
            icon: UserIcon, 
            change: '+۲.۵٪', 
            changeType: 'increase' 
          },
          { 
            title: 'مرسولات در انتظار', 
            value: new Intl.NumberFormat('fa-IR').format(pending_shipments), 
            icon: PackageIcon, 
            change: `${new Intl.NumberFormat('fa-IR').format(pending_shipments)} مورد`, 
            changeType: 'increase' 
          },
          { 
            title: 'سفرها در انتظار', 
            value: new Intl.NumberFormat('fa-IR').format(totalPendingTrips), 
            icon: PlaneIcon, 
            change: `${new Intl.NumberFormat('fa-IR').format(totalPendingTrips)} مورد`, 
            changeType: 'increase' 
          },
          { 
            title: 'خریدها در انتظار', 
            value: new Intl.NumberFormat('fa-IR').format(totalPendingPurchases), 
            icon: ShoppingCartIcon, 
            change: `${new Intl.NumberFormat('fa-IR').format(totalPendingPurchases)} مورد`, 
            changeType: 'increase' 
          },
          { 
            title: 'تیکت‌های باز', 
            value: new Intl.NumberFormat('fa-IR').format(open_tickets), 
            icon: TicketIcon, 
            change: `${new Intl.NumberFormat('fa-IR').format(open_tickets)} مورد`, 
            changeType: 'increase' 
          },
        ]);

        setPendingRequests(data.pending_requests || []);
        setRecentActivities(data.recent_activities || []);
      } catch (error) {
        console.error('Failed to load dashboard:', error);
      } finally {
        setLoading(false);
      }
    };

    fetchDashboardData();
  }, [refreshKey]);

  if (loading && refreshKey === 0) {
    return (
      <div className="flex items-center justify-center min-h-[400px]">
        <div className="animate-spin rounded-full h-12 w-12 border-4 border-custom-purple border-t-transparent"></div>
      </div>
    );
  }

  return (
    <div className="flex flex-col space-y-8 relative">
      {/* مودال رد درخواست */}
      {showRejectDialog && (
        <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4">
          <div className="bg-white dark:bg-custom-dark-blue-light rounded-2xl p-6 max-w-md w-full shadow-xl">
            <h3 className="text-xl font-bold mb-4 text-gray-800 dark:text-white">دلیل رد درخواست</h3>
            <textarea
              value={rejectReason}
              onChange={(e) => setRejectReason(e.target.value)}
              placeholder="لطفاً دلیل رد کردن این درخواست را وارد کنید..."
              className="w-full h-32 p-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-gray-50 dark:bg-custom-dark-blue text-gray-800 dark:text-gray-200 resize-none focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none"
            />
            <div className="flex gap-3 mt-4">
              <button
                onClick={() => {
                  setShowRejectDialog(false);
                  setSelectedRequest(null);
                  setRejectReason('');
                }}
                disabled={isProcessing}
                className="flex-1 py-2 px-4 bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-300 rounded-lg hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors disabled:opacity-50"
              >
                انصراف
              </button>
              <button
                onClick={handleReject}
                disabled={isProcessing || !rejectReason.trim()}
                className="flex-1 py-2 px-4 bg-red-500 text-white rounded-lg hover:bg-red-600 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
              >
                {isProcessing ? 'در حال پردازش...' : 'رد کردن'}
              </button>
            </div>
          </div>
        </div>
      )}

      {/* هدر */}
      <div>
        <h1 className="text-3xl font-bold text-gray-800 dark:text-white">داشبورد مدیریتی</h1>
        <p className="text-gray-500 dark:text-gray-400 mt-1">خوش آمدید! اینجا خلاصه‌ای از فعالیت‌های اخیر را مشاهده می‌کنید.</p>
      </div>

      {/* کارت‌های آمار */}
      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-5 gap-6">
        {stats.map((stat, index) => (
          <StatCard key={index} {...stat} />
        ))}
      </div>

      {/* جدول و فعالیت‌ها */}
      <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
        {/* جدول درخواست‌های در انتظار */}
        <div className="lg:col-span-2 bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
          <h3 className="text-xl font-semibold mb-4 text-gray-800 dark:text-white">آخرین درخواست‌های در انتظار تایید</h3>
          <div className="overflow-x-auto">
            <table className="w-full text-sm text-right">
              <thead className="text-gray-500 dark:text-gray-400 border-b border-gray-200 dark:border-gray-700">
                <tr>
                  <th className="p-3 font-semibold">شناسه</th>
                  <th className="p-3 font-semibold">نوع</th>
                  <th className="p-3 font-semibold">کاربر</th>
                  <th className="p-3 font-semibold hidden sm:table-cell">جزئیات</th>
                  <th className="p-3 font-semibold">عملیات</th>
                </tr>
              </thead>
              <tbody className="divide-y divide-gray-200 dark:divide-gray-700">
                {pendingRequests.map(req => (
                  <tr key={req.id} className="hover:bg-gray-50 dark:hover:bg-custom-dark-blue transition-colors">
                    <td className="p-3 font-mono text-custom-purple font-semibold">{req.id}</td>
                    <td className="p-3 text-gray-800 dark:text-gray-200">{req.type_label}</td>
                    <td className="p-3 text-gray-800 dark:text-gray-200">{req.user}</td>
                    <td className="p-3 hidden sm:table-cell text-gray-600 dark:text-gray-400">
                      {req.detail}
                    </td>
                    <td className="p-3">
                      <div className="flex items-center gap-2">
                        <button
                          onClick={() => handleApprove(req)}
                          disabled={isProcessing}
                          className="p-1.5 text-green-500 hover:bg-green-500/10 rounded-full transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
                          aria-label="تأیید"
                        >
                          <CheckCircleIcon className="w-5 h-5" />
                        </button>
                        <button
                          onClick={() => openRejectDialog(req)}
                          disabled={isProcessing}
                          className="p-1.5 text-red-500 hover:bg-red-500/10 rounded-full transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
                          aria-label="رد"
                        >
                          <XCircleIcon className="w-5 h-5" />
                        </button>
                      </div>
                    </td>
                  </tr>
                ))}
                {pendingRequests.length === 0 && (
                  <tr>
                    <td colSpan={5} className="p-8 text-center text-gray-500 dark:text-gray-400">
                      درخواستی در انتظار تأیید وجود ندارد.
                    </td>
                  </tr>
                )}
              </tbody>
            </table>
          </div>
        </div>

        {/* فعالیت‌های اخیر */}
        <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
          <h3 className="text-xl font-semibold mb-4 text-gray-800 dark:text-white">فعالیت‌های اخیر</h3>
          <ul className="space-y-4">
            {recentActivities.map((activity, index) => (
              <li key={index} className="flex items-start gap-3">
                <img
                  src={activity.user.avatar}
                  alt={activity.user.name}
                  className="w-10 h-10 rounded-full flex-shrink-0 mt-1 border-2 border-gray-200 dark:border-gray-700"
                />
                <div className="flex-grow">
                  <p className="text-sm font-medium text-gray-800 dark:text-gray-200">{activity.description}</p>
                  <div className="flex items-center text-xs text-gray-400 dark:text-gray-500 mt-1">
                    <ClockIcon className="w-3 h-3 ml-1" />
                    <span>{activity.time}</span>
                  </div>
                </div>
              </li>
            ))}
            {recentActivities.length === 0 && (
              <li className="text-center text-gray-500 dark:text-gray-400 text-sm py-4">
                فعالیتی جدید وجود ندارد.
              </li>
            )}
          </ul>
        </div>
      </div>
    </div>
  );
};

interface StatCardProps {
  title: string;
  value: string;
  icon: React.FC<{ className?: string }>;
  change: string;
  changeType: 'increase' | 'decrease';
}

const StatCard: React.FC<StatCardProps> = ({ title, value, icon: Icon, change }) => {
  return (
    <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm flex items-start justify-between hover:shadow-md transition-shadow">
      <div>
        <p className="text-sm font-medium text-gray-500 dark:text-gray-400">{title}</p>
        <p className="text-3xl font-bold text-gray-800 dark:text-white mt-2">{value}</p>
        <div className="mt-2 flex items-center text-xs text-gray-500 dark:text-gray-400">
          <span>{change}</span>
        </div>
      </div>
      <div className="bg-custom-purple/10 p-3 rounded-full">
        <Icon className="w-6 h-6 text-custom-purple" />
      </div>
    </div>
  );
};

export default AdminDashboardPage;
