import React, { useState, useEffect, useRef } from 'react';
import { SearchIcon, ChevronDownIcon, UserIcon, XCircleIcon, CheckCircleIcon } from '../Icons';
import { API_BASE_URL } from '@/src/config.js';
interface User {
    id: number;
    name: string | null;
    phone: string;
    created_at: string;
    status: 'active' | 'banned';
    phone_verified_at: string | null;
}

interface FilterDropdownProps {
  options: { value: string; label: string }[];
  value: string;
  onChange: (value: string) => void;
  placeholder: string;
}

const FilterDropdown: React.FC<FilterDropdownProps> = ({ options, value, onChange, placeholder }) => {
  const [isOpen, setIsOpen] = useState(false);
  const dropdownRef = useRef<HTMLDivElement>(null);
  const selectedOption = options.find(opt => opt.value === value);

  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (dropdownRef.current && !dropdownRef.current.contains(event.target as Node)) {
        setIsOpen(false);
      }
    };
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, []);

  return (
    <div className="relative" ref={dropdownRef}>
      <button
        type="button"
        onClick={() => setIsOpen(!isOpen)}
        className="w-full flex items-center justify-between text-right pr-4 pl-10 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition"
      >
        <span className={selectedOption ? "text-gray-800 dark:text-gray-200" : "text-gray-500"}>
          {selectedOption ? selectedOption.label : placeholder}
        </span>
        <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
            <ChevronDownIcon className={`w-5 h-5 text-gray-400 transition-transform duration-200 ${isOpen ? 'rotate-180' : ''}`} />
        </div>
      </button>
      {isOpen && (
        <ul className="absolute z-20 w-full mt-1 bg-custom-light-card dark:bg-custom-dark-blue-light border border-gray-300 dark:border-gray-600 rounded-lg shadow-lg max-h-60 overflow-y-auto">
          {options.map(option => (
            <li
              key={option.value}
              onClick={() => {
                onChange(option.value);
                setIsOpen(false);
              }}
              className="px-4 py-2 cursor-pointer hover:bg-gray-100 dark:hover:bg-custom-dark-blue text-gray-800 dark:text-gray-200 transition-colors"
            >
              {option.label}
            </li>
          ))}
        </ul>
      )}
    </div>
  );
};

interface UserListPageProps {
  onNavigate: (page: string, userId?: string) => void;
  showError: (message: string) => void;
  showSuccess: (message: string) => void;
}

const UserListPage: React.FC<UserListPageProps> = ({ onNavigate, showError, showSuccess }) => {
    const [searchQuery, setSearchQuery] = useState('');
    const [statusFilter, setStatusFilter] = useState('all');
    const [users, setUsers] = useState<User[]>([]);
    const [loading, setLoading] = useState(true);
    const [currentPage, setCurrentPage] = useState(1);
    const [totalPages, setTotalPages] = useState(1);

    const fetchUsers = async () => {
        setLoading(true);
        try {
            const token = localStorage.getItem('auth_token');
            if (!token) {
                showError('لطفاً دوباره وارد شوید.');
                return;
            }

            const params = new URLSearchParams({
                page: currentPage.toString(),
                per_page: '15',
            });
            
            if (statusFilter !== 'all') {
                params.append('status', statusFilter);
            }
            
            if (searchQuery.trim()) {
                params.append('search', searchQuery);
            }

            const response = await fetch(`http://127.0.0.1:8000/api/admin/users?${params}`, {
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'application/json',
                },
            });

            if (!response.ok) {
                const errorData = await response.json().catch(() => ({}));
                throw new Error(errorData.error || 'خطا در دریافت لیست کاربران');
            }

            const data = await response.json();
            console.log('API Response Structure:', data); // ← این خط برای دیباگ

            // ✅ پشتیبانی از هر دو ساختار پاسخ
            let usersData: User[] = [];
            let totalPagesData = 1;

            if (Array.isArray(data)) {
                // اگر API مستقیماً آرایه برمی‌گردونه
                usersData = data;
                totalPagesData = 1;
            } else if (data && typeof data === 'object') {
                // اگر ساختار {  [...], meta: {...} } باشه
                usersData = Array.isArray(data.data) ? data.data : [];
                totalPagesData = data.meta?.last_page || 1;
            }

            setUsers(usersData);
            setTotalPages(totalPagesData);
        } catch (error: any) {
            console.error('Fetch users error:', error);
            showError(error.message || 'خطا در ارتباط با سرور');
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchUsers();
    }, [currentPage, statusFilter, searchQuery]);

    const handleBanUser = async (userId: number) => {
        if (!confirm('آیا مطمئن هستید که می‌خواهید این کاربر را مسدود کنید؟')) return;
        
        try {
            const token = localStorage.getItem('auth_token');
            if (!token) {
                showError('لطفاً دوباره وارد شوید.');
                return;
            }

            const response = await fetch(`http://127.0.0.1:8000/api/admin/users/${userId}/ban`, {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'application/json',
                },
            });

            if (!response.ok) {
                const errorData = await response.json().catch(() => ({}));
                throw new Error(errorData.error || 'خطا در مسدود کردن کاربر');
            }

            showSuccess('کاربر با موفقیت مسدود شد');
            fetchUsers();
        } catch (error: any) {
            showError(error.message || 'خطا در ارتباط با سرور');
            console.error('Error banning user:', error);
        }
    };

    const handleUnbanUser = async (userId: number) => {
        if (!confirm('آیا مطمئن هستید که می‌خواهید این کاربر را فعال کنید؟')) return;
        
        try {
            const token = localStorage.getItem('auth_token');
            if (!token) {
                showError('لطفاً دوباره وارد شوید.');
                return;
            }

            const response = await fetch(`http://127.0.0.1:8000/api/admin/users/${userId}/unban`, {
                method: 'POST',
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'application/json',
                },
            });

            if (!response.ok) {
                const errorData = await response.json().catch(() => ({}));
                throw new Error(errorData.error || 'خطا در فعال کردن کاربر');
            }

            showSuccess('کاربر با موفقیت فعال شد');
            fetchUsers();
        } catch (error: any) {
            showError(error.message || 'خطا در ارتباط با سرور');
            console.error('Error unbanning user:', error);
        }
    };
    
    const getStatusChip = (status: User['status']) => {
        switch (status) {
            case 'active':
                return (
                    <span className="inline-flex items-center gap-1.5 px-3 py-1.5 text-xs font-semibold rounded-full bg-green-500/10 text-green-600 dark:text-green-400 border border-green-500/20">
                        <CheckCircleIcon className="w-3.5 h-3.5" />
                        فعال
                    </span>
                );
            case 'banned':
                return (
                    <span className="inline-flex items-center gap-1.5 px-3 py-1.5 text-xs font-semibold rounded-full bg-red-500/10 text-red-600 dark:text-red-400 border border-red-500/20">
                        <XCircleIcon className="w-3.5 h-3.5" />
                        مسدود
                    </span>
                );
        }
    };
    
    const statusOptions = [
        { value: 'all', label: 'همه وضعیت‌ها' },
        { value: 'active', label: 'فعال' },
        { value: 'banned', label: 'مسدود' },
    ];

    const convertToJalali = (gregorianDate: string) => {
        const date = new Date(gregorianDate);
        return new Intl.DateTimeFormat('fa-IR').format(date);
    };

    return (
        <div className="flex flex-col space-y-6">
            <div className="flex items-center justify-between">
                <div>
                    <h1 className="text-3xl font-bold text-gray-800 dark:text-white">مدیریت کاربران</h1>
                    <p className="text-gray-500 dark:text-gray-400 mt-1">لیست کاربران ثبت‌نام شده در سیستم</p>
                </div>
                <div className="bg-custom-purple/10 border border-custom-purple/30 px-4 py-2 rounded-lg">
                    <span className="text-sm text-gray-600 dark:text-gray-400">مجموع کاربران:</span>
                    <span className="text-lg font-bold text-custom-purple mr-2">{users.length}</span>
                </div>
            </div>

            <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-5 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div className="relative">
                        <input 
                            type="text" 
                            placeholder="جستجو نام یا شماره موبایل..." 
                            className="w-full pr-10 pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition text-gray-800 dark:text-gray-200"
                            value={searchQuery}
                            onChange={(e) => setSearchQuery(e.target.value)}
                        />
                        <SearchIcon className="absolute right-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400"/>
                    </div>
                    <FilterDropdown
                        options={statusOptions}
                        value={statusFilter}
                        onChange={setStatusFilter}
                        placeholder="وضعیت کاربر"
                    />
                </div>
            </div>

            <div className="bg-custom-light-card dark:bg-custom-dark-blue-light rounded-xl shadow-sm border border-gray-200 dark:border-gray-700 overflow-hidden">
                {loading ? (
                    <div className="flex items-center justify-center py-16">
                        <div className="animate-spin rounded-full h-12 w-12 border-4 border-custom-purple border-t-transparent"></div>
                    </div>
                ) : (
                    <div className="overflow-x-auto">
                        <table className="w-full text-sm text-right">
                            <thead className="bg-gray-50 dark:bg-custom-dark-blue text-gray-600 dark:text-gray-300 border-b border-gray-200 dark:border-gray-700">
                                <tr>
                                    <th className="p-4 font-semibold text-right">کاربر</th>
                                    <th className="p-4 font-semibold text-right hidden md:table-cell">شماره موبایل</th>
                                    <th className="p-4 font-semibold text-right hidden lg:table-cell">تاریخ ثبت نام</th>
                                    <th className="p-4 font-semibold text-center">وضعیت</th>
                                    <th className="p-4 font-semibold text-center">عملیات</th>
                                </tr>
                            </thead>
                            <tbody className="divide-y divide-gray-200 dark:divide-gray-700">
                                {users.length > 0 ? (
                                    users.map(user => (
                                        <tr key={user.id} className="hover:bg-gray-50 dark:hover:bg-custom-dark-blue/50 transition-colors">
                                            <td className="p-4">
                                                <div className="flex items-center gap-3">
                                                    <div className="w-10 h-10 rounded-full bg-gradient-to-br from-custom-purple to-blue-500 flex items-center justify-center text-white font-bold">
                                                        {user.name ? user.name.charAt(0).toUpperCase() : '?'}
                                                    </div>
                                                    <span className="font-semibold text-gray-800 dark:text-gray-200">{user.name || 'بدون نام'}</span>
                                                </div>
                                            </td>
                                            <td className="p-4 hidden md:table-cell">
                                                <span className="font-mono text-gray-700 dark:text-gray-300 bg-gray-100 dark:bg-custom-dark-blue px-2 py-1 rounded">
                                                    {user.phone}
                                                </span>
                                            </td>
                                            <td className="p-4 hidden lg:table-cell text-gray-600 dark:text-gray-400">
                                                {convertToJalali(user.created_at)}
                                            </td>
                                            <td className="p-4 text-center">
                                                {getStatusChip(user.status)}
                                            </td>
                                            <td className="p-4">
                                                <div className="flex items-center justify-center gap-2">
                                                    <button 
                                                        onClick={() => onNavigate('userDetail', user.id.toString())}
                                                        className="px-3 py-1.5 text-xs font-semibold text-custom-purple hover:bg-custom-purple/10 rounded-lg transition-colors"
                                                    >
                                                        مشاهده
                                                    </button>
                                                    {user.status === 'active' ? (
                                                        <button 
                                                            onClick={() => handleBanUser(user.id)}
                                                            className="px-3 py-1.5 text-xs font-semibold text-red-600 dark:text-red-400 hover:bg-red-500/10 rounded-lg transition-colors"
                                                        >
                                                            مسدود
                                                        </button>
                                                    ) : (
                                                        <button 
                                                            onClick={() => handleUnbanUser(user.id)}
                                                            className="px-3 py-1.5 text-xs font-semibold text-green-600 dark:text-green-400 hover:bg-green-500/10 rounded-lg transition-colors"
                                                        >
                                                            فعال‌سازی
                                                        </button>
                                                    )}
                                                </div>
                                            </td>
                                        </tr>
                                    ))
                                ) : (
                                    <tr>
                                        <td colSpan={5} className="text-center p-12">
                                            <div className="flex flex-col items-center gap-3">
                                                <UserIcon className="w-16 h-16 text-gray-300 dark:text-gray-600" />
                                                <p className="text-gray-500 dark:text-gray-400 font-medium">
                                                    هیچ کاربری یافت نشد
                                                </p>
                                            </div>
                                        </td>
                                    </tr>
                                )}
                            </tbody>
                        </table>
                    </div>
                )}

                {totalPages > 1 && (
                    <div className="flex items-center justify-center gap-2 p-4 border-t border-gray-200 dark:border-gray-700">
                        <button
                            onClick={() => setCurrentPage(prev => Math.max(1, prev - 1))}
                            disabled={currentPage === 1}
                            className="px-4 py-2 text-sm font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-custom-dark-blue border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-custom-dark-blue-light disabled:opacity-50 disabled:cursor-not-allowed transition"
                        >
                            قبلی
                        </button>
                        <span className="text-sm text-gray-600 dark:text-gray-400">
                            صفحه {currentPage} از {totalPages}
                        </span>
                        <button
                            onClick={() => setCurrentPage(prev => Math.min(totalPages, prev + 1))}
                            disabled={currentPage === totalPages}
                            className="px-4 py-2 text-sm font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-custom-dark-blue border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-custom-dark-blue-light disabled:opacity-50 disabled:cursor-not-allowed transition"
                        >
                            بعدی
                        </button>
                    </div>
                )}
            </div>
        </div>
    );
};

export default UserListPage;