import React, { useState, useEffect, useRef } from 'react';
import { SearchIcon, ChevronDownIcon, MapPinIcon, ClockIcon, PlaneIcon, UserIcon, ScaleIcon, CalendarIcon } from '../Icons';
import { API_BASE_URL } from '@/src/config.js';
interface TripRequest {
    id: string;
    type: 'trip' | 'companion';
    title: string;
    requestDate: string;
    status: 'active' | 'pending' | 'completed';
    origin: string;
    destination: string;
    departureDate: string;
    capacity?: number;
}

const mockTripRequests: TripRequest[] = [
    { id: '1', type: 'trip', title: 'سفر از تهران به تورنتو', requestDate: '۳ ساعت پیش', status: 'active', origin: 'تهران، ایران', destination: 'تورنتو، کانادا', departureDate: '۱۴۰۳/۰۵/۱۰', capacity: 7 },
    { id: '2', type: 'companion', title: 'همسفر برای مسیر اصفهان به شیراز', requestDate: '۱ روز پیش', status: 'active', origin: 'اصفهان، ایران', destination: 'شیراز، ایران', departureDate: '۱۴۰۳/۰۵/۱۲' },
    { id: '3', type: 'trip', title: 'سفر با قطار از مشهد به تهران', requestDate: '۲ روز پیش', status: 'pending', origin: 'مشهد، ایران', destination: 'تهران، ایران', departureDate: '۱۴۰۳/۰۵/۱۵', capacity: 15 },
    { id: '4', type: 'companion', title: 'آماده همراهی از آلمان به ایران', requestDate: '۵ روز پیش', status: 'completed', origin: 'برلین، آلمان', destination: 'تهران، ایران', departureDate: '۱۴۰۳/۰۶/۰۱' },
    { id: '5', type: 'trip', title: 'سفر کاری از تبریز به استانبول', requestDate: '۱ هفته پیش', status: 'completed', origin: 'تبریز، ایران', destination: 'استانبول، ترکیه', departureDate: '۱۴۰۳/۰۴/۲۰', capacity: 3 },
];

interface FilterDropdownProps {
  options: { value: string; label: string }[];
  value: string;
  onChange: (value: string) => void;
  placeholder: string;
}

const FilterDropdown: React.FC<FilterDropdownProps> = ({ options, value, onChange, placeholder }) => {
  const [isOpen, setIsOpen] = useState(false);
  const dropdownRef = useRef<HTMLDivElement>(null);
  const selectedOption = options.find(opt => opt.value === value);

  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (dropdownRef.current && !dropdownRef.current.contains(event.target as Node)) {
        setIsOpen(false);
      }
    };
    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, []);

  return (
    <div className="relative" ref={dropdownRef}>
      <button
        type="button"
        onClick={() => setIsOpen(!isOpen)}
        className="w-full flex items-center justify-between text-right pr-4 pl-10 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition"
      >
        <span className={selectedOption ? "text-gray-800 dark:text-gray-200" : "text-gray-500"}>
          {selectedOption ? selectedOption.label : placeholder}
        </span>
        <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
            <ChevronDownIcon className={`w-5 h-5 text-gray-400 transition-transform duration-200 ${isOpen ? 'rotate-180' : ''}`} />
        </div>
      </button>
      {isOpen && (
        <ul className="absolute z-20 w-full mt-1 bg-custom-light-card dark:bg-custom-dark-blue-light border border-gray-300 dark:border-gray-600 rounded-lg shadow-lg max-h-60 overflow-y-auto">
          {options.map(option => (
            <li
              key={option.value}
              onClick={() => {
                onChange(option.value);
                setIsOpen(false);
              }}
              className="px-4 py-2 cursor-pointer hover:bg-gray-100 dark:hover:bg-custom-dark-blue text-gray-800 dark:text-gray-200 transition-colors"
            >
              {option.label}
            </li>
          ))}
        </ul>
      )}
    </div>
  );
};


interface MyTripsPageProps {
    onNavigate: (page: string, payload?: any, options?: { isOwnerView?: boolean }) => void;
}

const MyTripsPage: React.FC<MyTripsPageProps> = ({ onNavigate }) => {
    const [searchQuery, setSearchQuery] = useState('');
    const [statusFilter, setStatusFilter] = useState('all');
    const [typeFilter, setTypeFilter] = useState('all');
    const [filteredRequests, setFilteredRequests] = useState<TripRequest[]>(mockTripRequests);

    useEffect(() => {
        const lowercasedQuery = searchQuery.toLowerCase();
        const result = mockTripRequests.filter(request => {
            const matchesSearch = request.title.toLowerCase().includes(lowercasedQuery);
            const matchesStatus = statusFilter === 'all' || request.status === statusFilter;
            const matchesType = typeFilter === 'all' || request.type === typeFilter;
            return matchesSearch && matchesStatus && matchesType;
        });
        setFilteredRequests(result);
    }, [searchQuery, statusFilter, typeFilter]);

    const statusOptions = [
        { value: 'all', label: 'همه وضعیت‌ها' },
        { value: 'active', label: 'فعال' },
        { value: 'pending', label: 'در حال بررسی' },
        { value: 'completed', label: 'انجام شده' },
    ];
    
    const typeOptions = [
        { value: 'all', label: 'همه انواع' },
        { value: 'trip', label: 'سفرهای من' },
        { value: 'companion', label: 'همسفری‌های من' },
    ];
    
    return (
        <div className="flex flex-col space-y-8">
            <div>
                <h1 className="text-3xl font-bold text-gray-800 dark:text-white">لیست سفرهای من</h1>
                <p className="text-gray-500 dark:text-gray-400 mt-1">سفرها و اعلام آمادگی‌های همسفری خود را مدیریت کنید.</p>
            </div>
            
            <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
                <div className="w-full grid grid-cols-1 sm:grid-cols-3 gap-4">
                    <div className="relative">
                        <input 
                            type="text" 
                            placeholder="جستجو..." 
                            className="w-full pr-10 pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition"
                            value={searchQuery}
                            onChange={(e) => setSearchQuery(e.target.value)}
                        />
                        <SearchIcon className="absolute right-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400"/>
                    </div>
                    <FilterDropdown options={statusOptions} value={statusFilter} onChange={setStatusFilter} placeholder="وضعیت" />
                    <FilterDropdown options={typeOptions} value={typeFilter} onChange={setTypeFilter} placeholder="نوع" />
                </div>
            </div>

            <div className="space-y-6">
                {filteredRequests.length > 0 ? (
                    filteredRequests.map(request => (
                        <TripRequestCard key={request.id} request={request} onNavigate={onNavigate} />
                    ))
                ) : (
                    <div className="text-center py-12 bg-custom-light-card dark:bg-custom-dark-blue-light rounded-2xl">
                        <PlaneIcon className="w-16 h-16 mx-auto text-gray-300 dark:text-gray-600" />
                        <h3 className="mt-4 text-xl font-semibold text-gray-800 dark:text-gray-200">سفری یافت نشد</h3>
                        <p className="mt-1 text-gray-500 dark:text-gray-400">هیچ سفری با این مشخصات یافت نشد.</p>
                    </div>
                )}
            </div>
        </div>
    );
};

interface TripRequestCardProps {
    request: TripRequest;
    onNavigate: (page: string, payload?: any, options?: { isOwnerView?: boolean }) => void;
}

const TripRequestCard: React.FC<TripRequestCardProps> = ({ request, onNavigate }) => {
    const { id, type, title, requestDate, status, origin, destination, departureDate, capacity } = request;

    const getStatusChip = () => {
        switch (status) {
            case 'active': return <span className="px-3 py-1 text-xs font-bold rounded-full bg-green-500/10 text-green-500">فعال</span>;
            case 'pending': return <span className="px-3 py-1 text-xs font-bold rounded-full bg-custom-orange/10 text-custom-orange">در حال بررسی</span>;
            case 'completed': return <span className="px-3 py-1 text-xs font-bold rounded-full bg-gray-500/10 text-gray-500 dark:bg-gray-400/10 dark:text-gray-400">انجام شده</span>;
        }
    };

    const isTrip = type === 'trip';
    
    return (
        <div className="flex flex-col md:flex-row md:items-center gap-6 bg-custom-light-card dark:bg-custom-dark-blue-light p-5 rounded-2xl shadow-sm border border-transparent hover:border-custom-purple/50 hover:shadow-lg transition-all duration-300">
            <div className="flex-shrink-0 w-full md:w-32 h-32 bg-custom-light-bg dark:bg-custom-dark-blue rounded-xl flex items-center justify-center">
                {isTrip ? <PlaneIcon className="w-16 h-16 text-custom-purple/50" /> : <UserIcon className="w-16 h-16 text-custom-purple/50" />}
            </div>

            <div className="flex flex-col flex-grow w-full">
                <div className="flex justify-between items-start mb-3">
                    <div>
                        <div className={`flex items-center gap-2 mb-1 ${isTrip ? 'text-custom-orange' : 'text-custom-cyan'}`}>
                            {isTrip ? <PlaneIcon className="w-5 h-5" /> : <UserIcon className="w-5 h-5" />}
                            <span className="text-sm font-semibold">{isTrip ? 'سفر' : 'همسفری'}</span>
                        </div>
                        <h3 className="text-lg font-bold text-gray-800 dark:text-white">{title}</h3>
                    </div>
                    <div className="flex-shrink-0">
                        {getStatusChip()}
                    </div>
                </div>

                <div className="flex-grow space-y-2 text-sm text-gray-600 dark:text-gray-300 border-t border-gray-200 dark:border-gray-700 pt-3">
                    <InfoRow icon={ClockIcon} label="تاریخ ثبت" value={requestDate} />
                    <InfoRow icon={MapPinIcon} label="مبدأ" value={origin} />
                    <InfoRow icon={MapPinIcon} label="مقصد" value={destination} />
                    <InfoRow icon={CalendarIcon} label="تاریخ حرکت" value={departureDate} />
                    {isTrip && capacity !== undefined && (
                        <InfoRow icon={ScaleIcon} label="ظرفیت" value={`${capacity} کیلوگرم`} />
                    )}
                </div>
                
                <div className="mt-4 flex items-center justify-end">
                    <button 
                        onClick={() => onNavigate('trip-detail', id, { isOwnerView: true })}
                        className="bg-custom-purple hover:bg-custom-light-purple text-white font-bold py-2 px-6 rounded-lg transition-colors duration-300 text-sm">
                        مشاهده جزئیات
                    </button>
                </div>
            </div>
        </div>
    );
};


const InfoRow: React.FC<{icon: React.FC<{className?: string}>, label: string, value?: string, valueClassName?: string}> = ({ icon: Icon, label, value, valueClassName = 'text-gray-800 dark:text-gray-200' }) => (
    <div className="flex items-center justify-between">
        <div className="flex items-center text-gray-500 dark:text-gray-400">
            <Icon className="w-4 h-4 ml-2" />
            <span>{label}:</span>
        </div>
        <span className={`font-semibold ${valueClassName}`}>{value || '-'}</span>
    </div>
);

export default MyTripsPage;