import React, { useState, useCallback, useRef, useEffect } from 'react';
import { TypeIcon, TagIcon, ListIcon, UploadCloudIcon, XIcon, ArrowRightIcon, ChevronDownIcon, PriorityLowIcon, PriorityMediumIcon, PriorityHighIcon, AlertCircleIcon, PaperclipIcon, SendIcon } from '../Icons';
import { API_BASE_URL } from '@/src/config.js';
interface NewTicketPageProps {
    onNavigate: (page: string) => void;
}

interface ImageFile {
  id: number;
  url: string;
  file: File;
}

// Reusable components
interface InputWithIconProps {
    id: string; icon: React.FC<{ className?: string }>; label: string; placeholder: string;
}
const InputWithIcon: React.FC<InputWithIconProps> = ({ id, icon: Icon, label, placeholder }) => (
    <div>
        <label htmlFor={id} className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">{label}</label>
        <div className="relative">
            <div className="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none"><Icon className="w-5 h-5 text-gray-400" /></div>
            <input type="text" id={id} className="w-full pr-10 pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition" placeholder={placeholder} />
        </div>
    </div>
);

interface SelectWithIconProps {
    id: string; icon: React.FC<{ className?: string }>; label: string; children: React.ReactNode;
}
const SelectWithIcon: React.FC<SelectWithIconProps> = ({ id, icon: Icon, label, children }) => {
    const [isOpen, setIsOpen] = useState(false);
    const selectorRef = useRef<HTMLDivElement>(null);
    const [selectedValue, setSelectedValue] = useState('');

    const options = React.Children.toArray(children)
        .filter((child): child is React.ReactElement<React.OptionHTMLAttributes<HTMLOptionElement>> =>
            React.isValidElement(child) && child.type === 'option')
        .map(option => ({
            value: option.props.value?.toString() ?? (option.props.children?.toString() || ''),
            label: option.props.children?.toString() || ''
        }));
    
    const selectedOption = options.find(opt => opt.value === selectedValue) || options[0];

    useEffect(() => {
        const handleClickOutside = (event: MouseEvent) => {
            if (selectorRef.current && !selectorRef.current.contains(event.target as Node)) {
                setIsOpen(false);
            }
        };
        document.addEventListener("mousedown", handleClickOutside);
        return () => document.removeEventListener("mousedown", handleClickOutside);
    }, []);

    const handleSelect = (value: string) => {
        setSelectedValue(value);
        setIsOpen(false);
    };

    return (
        <div>
            <label htmlFor={id} className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">{label}</label>
            <div className="relative" ref={selectorRef}>
                <div className="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none"><Icon className="w-5 h-5 text-gray-400" /></div>
                <button
                    type="button"
                    id={id}
                    onClick={() => setIsOpen(!isOpen)}
                    className="w-full flex items-center justify-between text-right pr-10 pl-10 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition"
                >
                    <span className="text-gray-800 dark:text-gray-200">{selectedOption?.label}</span>
                    <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                        <ChevronDownIcon className={`w-5 h-5 text-gray-400 transition-transform ${isOpen ? 'rotate-180' : ''}`} />
                    </div>
                </button>
                {isOpen && (
                    <ul className="absolute z-20 w-full mt-1 bg-custom-light-card dark:bg-custom-dark-blue-light border border-gray-300 dark:border-gray-600 rounded-lg shadow-lg max-h-60 overflow-y-auto">
                        {options.map(option => (
                            <li
                                key={option.value}
                                onClick={() => handleSelect(option.value)}
                                className="px-4 py-2 cursor-pointer hover:bg-gray-100 dark:hover:bg-custom-dark-blue text-gray-800 dark:text-gray-200 transition-colors"
                            >
                                {option.label}
                            </li>
                        ))}
                    </ul>
                )}
            </div>
        </div>
    );
};

interface TextareaWithIconProps {
    id: string; icon: React.FC<{ className?: string }>; label: string; placeholder: string;
}
const TextareaWithIcon: React.FC<TextareaWithIconProps> = ({ id, icon: Icon, label, placeholder }) => (
    <div>
        <label htmlFor={id} className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">{label}</label>
        <div className="relative">
            <div className="absolute top-3.5 right-0 flex items-center pr-3 pointer-events-none"><Icon className="w-5 h-5 text-gray-400" /></div>
            <textarea id={id} className="w-full pr-10 pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition min-h-[150px] resize-y" placeholder={placeholder}></textarea>
        </div>
    </div>
);


const NewTicketPage: React.FC<NewTicketPageProps> = ({ onNavigate }) => {
    const [attachments, setAttachments] = useState<ImageFile[]>([]);
    
    const handleFileUpload = (event: React.ChangeEvent<HTMLInputElement>) => {
        const files = event.target.files;
        if (files) {
            const newFiles: ImageFile[] = Array.from(files).map((file: File) => ({
                id: Date.now() + Math.random(),
                url: URL.createObjectURL(file), // Note: for non-image files, this won't show a preview.
                file: file
            }));
            setAttachments(prev => [...prev, ...newFiles]);
        }
    };
    
    const removeAttachment = (id: number) => {
        const fileToRemove = attachments.find(f => f.id === id);
        if (fileToRemove) {
            URL.revokeObjectURL(fileToRemove.url);
        }
        setAttachments(prev => prev.filter(img => img.id !== id));
    };

    const onDrop = useCallback((event: React.DragEvent<HTMLLabelElement>) => {
        event.preventDefault();
        event.stopPropagation();
        const files = event.dataTransfer.files;
        if (files && files.length > 0) {
            const newFiles: ImageFile[] = Array.from(files).map((file: File) => ({
                    id: Date.now() + Math.random(),
                    url: URL.createObjectURL(file),
                    file: file
                }));
            setAttachments(prev => [...prev, ...newFiles]);
        }
    }, []);

    const onDragOver = (event: React.DragEvent<HTMLLabelElement>) => {
        event.preventDefault();
        event.stopPropagation();
    };

    return (
        <div className="flex flex-col space-y-8">
             <div className="flex items-center justify-between">
                <div>
                    <h1 className="text-3xl font-bold text-gray-800 dark:text-white">ارسال تیکت جدید</h1>
                    <p className="text-gray-500 dark:text-gray-400 mt-1">مشکل یا پیشنهاد خود را با ما در میان بگذارید.</p>
                </div>
                <button onClick={() => onNavigate('tickets')} className="flex items-center gap-2 py-2 px-4 rounded-lg text-gray-600 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-custom-dark-blue transition-colors">
                    <ArrowRightIcon className="w-5 h-5" />
                    <span>بازگشت به لیست</span>
                </button>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
                <div className="lg:col-span-2">
                     <form className="space-y-8 bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl shadow-sm">
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                           <InputWithIcon icon={TypeIcon} id="title" placeholder="مثال: مشکل در پرداخت" label="عنوان تیکت" />
                           <SelectWithIcon icon={TagIcon} id="department" label="دپارتمان">
                                <option>پشتیبانی فنی</option>
                                <option>مالی</option>
                                <option>پیشنهادات</option>
                           </SelectWithIcon>
                        </div>
                        <SelectWithIcon icon={AlertCircleIcon} id="priority" label="اولویت">
                            <option>کم</option>
                            <option>متوسط</option>
                            <option>زیاد</option>
                        </SelectWithIcon>
                        <TextareaWithIcon icon={ListIcon} id="message" placeholder="لطفا مشکل خود را با جزئیات کامل شرح دهید..." label="متن پیام" />
                        
                        <button type="submit" className="w-full bg-custom-purple hover:bg-custom-light-purple text-white font-bold py-3 px-4 rounded-lg transition-colors duration-300 shadow-lg hover:shadow-xl focus:outline-none focus:ring-4 focus:ring-custom-purple/50 flex items-center justify-center gap-2">
                           <SendIcon className="w-5 h-5" />
                           <span>ارسال تیکت</span>
                        </button>
                    </form>
                </div>
                <div className="lg:col-span-1 space-y-8">
                     <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
                        <h3 className="text-lg font-semibold mb-4">پیوست فایل (اختیاری)</h3>
                        <label 
                            htmlFor="file-upload" 
                            className="flex flex-col items-center justify-center w-full h-48 border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-lg cursor-pointer bg-custom-light-bg dark:bg-custom-dark-blue hover:bg-gray-100 dark:hover:bg-gray-800 transition-colors"
                            onDrop={onDrop}
                            onDragOver={onDragOver}
                        >
                            <div className="flex flex-col items-center justify-center pt-5 pb-6">
                                <UploadCloudIcon className="w-10 h-10 mb-3 text-gray-400" />
                                <p className="mb-2 text-sm text-gray-500 dark:text-gray-400"><span className="font-semibold">برای آپلود کلیک کنید</span> یا بکشید</p>
                                <p className="text-xs text-gray-500 dark:text-gray-400">PNG, JPG or PDF</p>
                            </div>
                            <input id="file-upload" type="file" className="hidden" multiple accept="image/*,application/pdf" onChange={handleFileUpload} />
                        </label>
                    </div>

                    {attachments.length > 0 && (
                         <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
                            <h3 className="text-lg font-semibold mb-4">فایل‌های پیوست شده</h3>
                             <div className="space-y-3">
                                {attachments.map(file => (
                                    <div key={file.id} className="flex items-center justify-between bg-custom-light-bg dark:bg-custom-dark-blue p-2 rounded-lg">
                                        <div className="flex items-center gap-2 overflow-hidden">
                                            <PaperclipIcon className="w-5 h-5 text-gray-500 flex-shrink-0" />
                                            <span className="text-sm truncate">{file.file.name}</span>
                                        </div>
                                        <button onClick={() => removeAttachment(file.id)} className="p-1 text-red-500 hover:bg-red-100 dark:hover:bg-red-900/50 rounded-full">
                                            <XIcon className="w-4 h-4" />
                                        </button>
                                    </div>
                                ))}
                            </div>
                        </div>
                    )}
                </div>
            </div>
        </div>
    );
};

export default NewTicketPage;