// src/components/pages/PurchaseDetailPage.tsx
import React, { useEffect, useState, useRef } from 'react';
import { ArrowRightIcon, ECommerceIcon, MapPinIcon, DollarSignIcon, LinkIcon, CheckIcon, XIcon, ImageIcon } from '../Icons';
import { API_BASE_URL } from '@/src/config.js';
declare const L: any; // برای پشتیبانی از Leaflet

// دیکشنری ترجمه دسته‌بندی‌ها
const categoryOptions: { [key: string]: string } = {
  'electronics': 'الکترونیک',
  'fashion': 'مد و پوشاک',
  'books': 'کتاب و رسانه',
  'home': 'خانه و آشپزخانه',
  'beauty': 'آرایشی و بهداشتی',
  'other': 'سایر'
};

const translateCategories = (categoriesInput: string[] | null | undefined): string => {
  if (!categoriesInput || categoriesInput.length === 0) return 'عمومی';
  
  const translated = categoriesInput.map(engCat => {
    const lowerCat = engCat.toLowerCase().trim();
    return categoryOptions[lowerCat] || engCat;
  });
  
  return translated.join('، ');
};

interface ImageData {
  id: number;
  url: string;
}

interface PurchaseData {
  id: number;
  type: 'purchase' | 'shopper';
  title: string;
  status: 'pending' | 'approved' | 'rejected';
  rejection_reason?: string;
  user: {
    name: string;
  };
  images: ImageData[];
  
  // فقط برای purchase
  link?: string;
  description?: string;
  purchaseLocation?: { country: string; city: string; };
  deliveryLocation?: { 
    country: string; 
    city: string; 
    address?: string;
    lat?: number;
    lng?: number;
  };
  itemPrice?: number;
  proposedFee?: number;
  currency?: 'USD' | 'CAD' | 'IRT' | 'EUR' | 'IQD';
  showPhone?: boolean;
  
  // فقط برای shopper
  location?: { country: string; city: string; };
  serviceTitle?: string;
  categories?: string[];
  supportedSites?: string[];
  feeDescription?: string;
  capabilities?: string;
  
  created_at: string;
  updated_at: string;
}

interface PurchaseDetailPageProps {
  purchaseId: string | null;
  purchaseType: 'purchase' | 'shopper' | null;
  onNavigate: (page: string) => void;
  showError: (message: string) => void;
  showSuccess: (message: string) => void;
}

const PurchaseDetailPage: React.FC<PurchaseDetailPageProps> = ({
  purchaseId,
  purchaseType,
  onNavigate,
  showError,
  showSuccess
}) => {
  const [purchase, setPurchase] = useState<PurchaseData | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [rejectReason, setRejectReason] = useState('');
  const [showRejectModal, setShowRejectModal] = useState(false);
  const [isProcessing, setIsProcessing] = useState(false);
  const [selectedImage, setSelectedImage] = useState<string | null>(null);
  const mapContainerRef = useRef<HTMLDivElement>(null);
  const mapRef = useRef<any>(null);

  const handleApprove = async () => {
    if (!purchaseId || !purchaseType) return;
    
    try {
      const token = localStorage.getItem('auth_token');
      if (!token) {
        showError('لطفاً دوباره وارد شوید.');
        return;
      }

      const endpoint = purchaseType === 'purchase' ? 'online-purchases' : 'online-shoppers';
      const response = await fetch(`${API_BASE_URL}/admin/${endpoint}/${purchaseId}/approve`, {
        method: 'PUT',
        headers: { 'X-Auth-Token': token, },
      });

      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.message || 'خطا در تأیید درخواست');
      }

      showSuccess('درخواست با موفقیت تأیید شد');
      setPurchase(prev => prev ? { ...prev, status: 'approved' } : null);
    } catch (error: any) {
      showError(error.message || 'خطا در ارتباط با سرور');
    }
  };

  const handleReject = async () => {
    if (!purchaseId || !purchaseType || !rejectReason.trim()) {
      showError('لطفاً دلیل رد را وارد کنید');
      return;
    }

    setIsProcessing(true);
    try {
      const token = localStorage.getItem('auth_token');
      if (!token) {
        showError('لطفاً دوباره وارد شوید.');
        return;
      }

      const endpoint = purchaseType === 'purchase' ? 'online-purchases' : 'online-shoppers';
      const response = await fetch(`${API_BASE_URL}/admin/${endpoint}/${purchaseId}/reject`, {
        method: 'PUT',
        headers: {
          'X-Auth-Token': token,
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ reason: rejectReason })
      });

      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.message || 'خطا در رد درخواست');
      }

      showSuccess('درخواست با موفقیت رد شد');
      setPurchase(prev => prev ? { ...prev, status: 'rejected', rejection_reason: rejectReason } : null);
      setShowRejectModal(false);
      setRejectReason('');
    } catch (error: any) {
      showError(error.message || 'خطا در ارتباط با سرور');
    } finally {
      setIsProcessing(false);
    }
  };

  useEffect(() => {
    if (!purchaseId || !purchaseType) {
      setError('اطلاعات درخواست ناقص است');
      setLoading(false);
      return;
    }

    const fetchPurchase = async () => {
      try {
        const token = localStorage.getItem('auth_token');
        if (!token) {
          setError('لطفاً دوباره وارد شوید');
          return;
        }

        const endpoint = purchaseType === 'purchase' ? 'online-purchases' : 'online-shoppers';
        const response = await fetch(`${API_BASE_URL}/admin/${endpoint}/${purchaseId}`, {
          headers: { 'X-Auth-Token': token, }
        });

        if (!response.ok) {
          throw new Error('درخواست یافت نشد');
        }

        const result = await response.json();
        setPurchase(result.data);
      } catch (err: any) {
        setError(err.message || 'خطا در دریافت جزئیات');
      } finally {
        setLoading(false);
      }
    };

    fetchPurchase();
  }, [purchaseId, purchaseType]);

  // 🔹 رندر نقشه با Leaflet (فقط برای deliveryLocation)
  useEffect(() => {
    if (!purchase || purchase.type !== 'purchase' || !mapContainerRef.current || mapRef.current) return;

    const { deliveryLocation } = purchase;
    if (!deliveryLocation || !deliveryLocation.lat || !deliveryLocation.lng) return;

    const lat = parseFloat(deliveryLocation.lat as any);
    const lng = parseFloat(deliveryLocation.lng as any);

    if (isNaN(lat) || isNaN(lng)) return;

    const map = L.map(mapContainerRef.current).setView([lat, lng], 12);
    mapRef.current = map;

    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
      attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors'
    }).addTo(map);

    const marker = L.marker([lat, lng]).addTo(map);
    marker.bindPopup(`<b>تحویل کالا:</b><br/>${deliveryLocation.city}, ${deliveryLocation.country}<br/>${deliveryLocation.address || ''}`);

    return () => {
      if (mapRef.current) {
        mapRef.current.remove();
        mapRef.current = null;
      }
    };
  }, [purchase]);

  if (loading) {
    return (
      <div className="flex items-center justify-center min-h-[400px]">
        <div className="animate-spin rounded-full h-12 w-12 border-4 border-custom-purple border-t-transparent"></div>
      </div>
    );
  }

  if (error || !purchase) {
    return (
      <div className="text-center p-8">
        <h2 className="text-xl font-bold text-red-500">{error || 'درخواست یافت نشد'}</h2>
        <button onClick={() => onNavigate('admin-purchases')} className="mt-4 text-custom-purple hover:underline">
          بازگشت به لیست
        </button>
      </div>
    );
  }

  const getStatusBadge = () => {
    switch (purchase.status) {
      case 'approved':
        return <span className="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-green-100 text-green-800">تأیید شده</span>;
      case 'rejected':
        return <span className="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-red-100 text-red-800">رد شده</span>;
      default:
        return <span className="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">در انتظار بررسی</span>;
    }
  };

  const isPurchase = purchase.type === 'purchase';

  return (
    <div className="flex flex-col space-y-6">
      {/* هدر */}
      <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
        <div>
          <h1 className="text-3xl font-bold text-gray-800 dark:text-white">{purchase.title}</h1>
          <div className="flex items-center gap-2 mt-2">
            <span className="text-sm text-gray-500 dark:text-gray-400">
              {isPurchase ? 'درخواست خرید' : 'ثبت‌نام خریدار'}
            </span>
            {getStatusBadge()}
          </div>
        </div>
        <button
          onClick={() => onNavigate('admin-purchases')}
          className="flex items-center gap-2 py-2 px-4 rounded-lg text-gray-600 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-custom-dark-blue transition-colors self-start sm:self-center"
        >
          <ArrowRightIcon className="w-5 h-5" />
          <span>بازگشت به لیست</span>
        </button>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
        {/* محتوای اصلی */}
        <div className="lg:col-span-2 space-y-8">
          {/* جزئیات درخواست */}
          <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl shadow-sm">
            <h3 className="text-xl font-semibold mb-4 border-b pb-3 border-gray-200 dark:border-gray-700">جزئیات درخواست</h3>
            
            {isPurchase ? (
              <div className="space-y-4">
                <div>
                  <h4 className="font-semibold text-gray-700 dark:text-gray-300 mb-2">عنوان کالا</h4>
                  <p className="text-gray-600 dark:text-gray-400">{purchase.title}</p>
                </div>
                
                {purchase.link && (
                  <div>
                    <h4 className="font-semibold text-gray-700 dark:text-gray-300 mb-2">لینک کالا</h4>
                    <a 
                      href={purchase.link} 
                      target="_blank" 
                      rel="noopener noreferrer"
                      className="text-custom-purple hover:underline break-all"
                    >
                      {purchase.link}
                    </a>
                  </div>
                )}
                
                {purchase.description && (
                  <div>
                    <h4 className="font-semibold text-gray-700 dark:text-gray-300 mb-2">توضیحات</h4>
                    <p className="text-gray-600 dark:text-gray-400 whitespace-pre-wrap">{purchase.description}</p>
                  </div>
                )}
              </div>
            ) : (
              <div className="space-y-4">
                {purchase.serviceTitle && (
                  <div>
                    <h4 className="font-semibold text-gray-700 dark:text-gray-300 mb-2">عنوان خدمات</h4>
                    <p className="text-gray-600 dark:text-gray-400">{purchase.serviceTitle}</p>
                  </div>
                )}
                
                {purchase.feeDescription && (
                  <div>
                    <h4 className="font-semibold text-gray-700 dark:text-gray-300 mb-2">توضیحات کارمزد</h4>
                    <p className="text-gray-600 dark:text-gray-400 whitespace-pre-wrap">{purchase.feeDescription}</p>
                  </div>
                )}
                
                {purchase.capabilities && (
                  <div>
                    <h4 className="font-semibold text-gray-700 dark:text-gray-300 mb-2">توانایی‌ها</h4>
                    <p className="text-gray-600 dark:text-gray-400 whitespace-pre-wrap">{purchase.capabilities}</p>
                  </div>
                )}
              </div>
            )}

            {purchase.status === 'rejected' && purchase.rejection_reason && (
              <div className="mt-6 p-4 bg-red-50 dark:bg-red-900/20 rounded-lg">
                <h4 className="font-semibold text-red-800 dark:text-red-400 mb-2">دلیل رد درخواست:</h4>
                <p className="text-red-700 dark:text-red-300">{purchase.rejection_reason}</p>
              </div>
            )}
          </div>

          {/* تصاویر */}
          {purchase.images && purchase.images.length > 0 && (
            <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl shadow-sm">
              <h3 className="text-xl font-semibold mb-4 border-b pb-3 border-gray-200 dark:border-gray-700 flex items-center gap-2">
                <ImageIcon className="w-6 h-6" />
                تصاویر ({purchase.images.length})
              </h3>
              <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
                {purchase.images.map((img) => (
                  <div 
                    key={img.id} 
                    className="relative group cursor-pointer overflow-hidden rounded-lg aspect-square"
                    onClick={() => setSelectedImage(`http://127.0.0.1:9000/${img.url}`)}
                  >
                    <img
                      src={`http://127.0.0.1:9000/${img.url}`}
                      alt="تصویر"
                      className="w-full h-full object-cover transition-transform duration-300 group-hover:scale-110"
                    />
                    <div className="absolute inset-0 bg-black opacity-0 group-hover:opacity-40 transition-opacity duration-300" />
                  </div>
                ))}
              </div>
            </div>
          )}

          {/* مسیر / اطلاعات */}
          <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl shadow-sm">
            <h3 className="text-xl font-semibold mb-4 border-b pb-3 border-gray-200 dark:border-gray-700">
              {isPurchase ? 'مسیر درخواست' : 'اطلاعات خدمات'}
            </h3>
            {isPurchase ? (
              <div className="space-y-6">
               
                <div className="flex items-start gap-3">
                  <MapPinIcon className="w-5 h-5 text-red-500 mt-1 flex-shrink-0" />
                  <div>
                    <h4 className="font-semibold text-gray-800 dark:text-gray-200">تحویل در</h4>
                    <p className="text-gray-600 dark:text-gray-400">
                      {purchase.deliveryLocation?.city || '—'}, {purchase.deliveryLocation?.country || '—'}
                    </p>
                    {purchase.deliveryLocation?.address && (
                      <p className="text-sm text-gray-500 dark:text-gray-500 mt-1">
                        {purchase.deliveryLocation.address}
                      </p>
                    )}
                  </div>
                </div>

                {/* 🔹 نمایش نقشه فقط برای deliveryLocation */}
                {purchase.deliveryLocation?.lat && purchase.deliveryLocation?.lng && (
                  <div className="mt-4">
                    <h4 className="font-semibold text-gray-800 dark:text-gray-200 mb-2">موقعیت تحویل</h4>
                    <div 
                      ref={mapContainerRef} 
                      className="w-full h-64 bg-gray-200 dark:bg-gray-700 rounded-lg border border-gray-300 dark:border-gray-600"
                    />
                  </div>
                )}
              </div>
            ) : (
              <div className="space-y-4">
                <div className="flex items-start gap-3">
                  <MapPinIcon className="w-5 h-5 text-blue-500 mt-1 flex-shrink-0" />
                  <div>
                    <h4 className="font-semibold text-gray-800 dark:text-gray-200">محل فعالیت</h4>
                    <p className="text-gray-600 dark:text-gray-400">
                      {purchase.location?.city}, {purchase.location?.country}
                    </p>
                  </div>
                </div>
                
                <div className="flex items-start gap-3">
                  <ECommerceIcon className="w-5 h-5 text-purple-500 mt-1 flex-shrink-0" />
                  <div>
                    <h4 className="font-semibold text-gray-800 dark:text-gray-200">دسته‌بندی</h4>
                    <p className="text-gray-600 dark:text-gray-400">{translateCategories(purchase.categories)}</p>
                  </div>
                </div>

                {purchase.supportedSites && purchase.supportedSites.length > 0 && (
                  <div className="flex items-start gap-3">
                    <LinkIcon className="w-5 h-5 text-green-500 mt-1 flex-shrink-0" />
                    <div>
                      <h4 className="font-semibold text-gray-800 dark:text-gray-200">سایت‌های پشتیبانی شده</h4>
                      <div className="space-y-1 mt-1">
                        {purchase.supportedSites.map((site, idx) => (
                          <a 
                            key={idx}
                            href={site.startsWith('http') ? site : `https://${site}`}
                            target="_blank"
                            rel="noopener noreferrer"
                            className="block text-custom-purple hover:underline text-sm break-all"
                          >
                            {site}
                          </a>
                        ))}
                      </div>
                    </div>
                  </div>
                )}
              </div>
            )}
          </div>

          {/* اطلاعات مالی */}
          {isPurchase && (
            <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl shadow-sm">
              <h3 className="text-xl font-semibold mb-4 border-b pb-3 border-gray-200 dark:border-gray-700">اطلاعات مالی</h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <InfoRow icon={DollarSignIcon} label="قیمت کالا" value={`${purchase.itemPrice?.toLocaleString() || 0} ${purchase.currency || 'IRT'}`} />
                <InfoRow icon={DollarSignIcon} label="کارمزد" value={`${purchase.proposedFee?.toLocaleString() || 0} ${purchase.currency || 'IRT'}`} />
              </div>
            </div>
          )}
        </div>

        {/* Sidebar */}
        <div className="lg:col-span-1 space-y-6">
          <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm space-y-4 sticky top-28">
            <div className="text-center">
              <div className="w-24 h-24 rounded-full bg-gradient-to-br from-custom-purple to-custom-light-purple flex items-center justify-center text-white text-2xl mx-auto">
                {purchase.user.name.charAt(0).toUpperCase()}
              </div>
              <h3 className="text-xl font-bold mt-3">{purchase.user.name}</h3>
            </div>

            <div className="border-t border-gray-200 dark:border-gray-700 my-4" />
            
            {isPurchase ? (
              <>
                
                <InfoRow icon={MapPinIcon} label="تحویل در" value={`${purchase.deliveryLocation?.city || '-'}, ${purchase.deliveryLocation?.country || '-'}`} />
                <InfoRow icon={DollarSignIcon} label="قیمت کالا" value={`${purchase.itemPrice?.toLocaleString() || 0} ${purchase.currency || 'IRT'}`} />
                <InfoRow icon={DollarSignIcon} label="کارمزد" value={`${purchase.proposedFee?.toLocaleString() || 0} ${purchase.currency || 'IRT'}`} />
              </>
            ) : (
              <>
                <InfoRow icon={MapPinIcon} label="محل فعالیت" value={`${purchase.location?.city}, ${purchase.location?.country}`} />
                <InfoRow icon={ECommerceIcon} label="دسته‌بندی" value={translateCategories(purchase.categories)} />
              </>
            )}

            <div className="border-t border-gray-200 dark:border-gray-700 my-4" />
            <div className="text-sm text-gray-500 dark:text-gray-400">
              <p>ایجاد شده: {new Date(purchase.created_at).toLocaleDateString('fa-IR')}</p>
              <p>آخرین به‌روزرسانی: {new Date(purchase.updated_at).toLocaleDateString('fa-IR')}</p>
            </div>

            {purchase.status === 'pending' && (
              <div className="grid grid-cols-2 gap-3 pt-4">
                <button
                  onClick={() => setShowRejectModal(true)}
                  className="w-full flex items-center justify-center gap-2 bg-red-500/10 hover:bg-red-500/20 text-red-500 font-bold py-3 px-4 rounded-lg transition-colors duration-300"
                >
                  <XIcon className="w-5 h-5" />
                  <span>رد کردن</span>
                </button>
                <button
                  onClick={handleApprove}
                  className="w-full flex items-center justify-center gap-2 bg-green-500/10 hover:bg-green-500/20 text-green-500 font-bold py-3 px-4 rounded-lg transition-colors duration-300"
                >
                  <CheckIcon className="w-5 h-5" />
                  <span>تایید</span>
                </button>
              </div>
            )}
          </div>
        </div>
      </div>

      {/* مودال رد درخواست */}
      {showRejectModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white dark:bg-custom-dark-blue-light rounded-xl p-6 max-w-md w-full">
            <h3 className="text-xl font-bold mb-4 text-gray-800 dark:text-white">رد درخواست</h3>
            <p className="text-gray-600 dark:text-gray-300 mb-4">لطفاً دلیل رد کردن این درخواست را وارد کنید:</p>
            <textarea
              value={rejectReason}
              onChange={(e) => setRejectReason(e.target.value)}
              className="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-custom-dark-blue text-gray-800 dark:text-white focus:outline-none focus:ring-2 focus:ring-custom-purple"
              rows={4}
              placeholder="دلیل رد درخواست..."
            />
            <div className="flex justify-end gap-3 mt-4">
              <button
                onClick={() => {
                  setShowRejectModal(false);
                  setRejectReason('');
                }}
                className="px-4 py-2 bg-gray-200 dark:bg-gray-700 text-gray-800 dark:text-white rounded-lg hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors"
              >
                انصراف
              </button>
              <button
                onClick={handleReject}
                disabled={isProcessing || !rejectReason.trim()}
                className="px-4 py-2 bg-red-500 text-white rounded-lg hover:bg-red-600 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
              >
                {isProcessing ? 'در حال پردازش...' : 'رد درخواست'}
              </button>
            </div>
          </div>
        </div>
      )}

      {/* مودال نمایش تصویر بزرگ */}
      {selectedImage && (
        <div 
          className="fixed inset-0 bg-black bg-opacity-90 flex items-center justify-center z-50 p-4"
          onClick={() => setSelectedImage(null)}
        >
          <div className="relative max-w-4xl max-h-[90vh]">
            <img
              src={selectedImage}
              alt="تصویر بزرگ"
              className="max-w-full max-h-[90vh] object-contain rounded-lg"
            />
            <button
              onClick={() => setSelectedImage(null)}
              className="absolute top-4 right-4 bg-white rounded-full p-2 hover:bg-gray-200 transition-colors"
            >
              <XIcon className="w-6 h-6 text-gray-800" />
            </button>
          </div>
        </div>
      )}
    </div>
  );
};

const InfoRow: React.FC<{icon: React.FC<{className?: string}>, label: string, value: string}> = ({ icon: Icon, label, value }) => (
  <div className="flex items-center justify-between">
    <div className="flex items-center text-gray-500 dark:text-gray-400">
      <Icon className="w-5 h-5 ml-2" />
      <span className="text-sm">{label}</span>
    </div>
    <span className="font-semibold text-sm text-gray-800 dark:text-gray-200">{value}</span>
  </div>
);

export default PurchaseDetailPage;