import React, { useState, useCallback, useRef, useEffect } from 'react';
import { 
    UploadCloudIcon, TypeIcon, MapPinIcon, DollarSignIcon, PhoneIcon, XIcon, ChevronDownIcon, 
    CrosshairIcon, HomeIcon, ListIcon, ECommerceIcon, UserIcon, CrmIcon, SearchIcon, LinkIcon 
} from '../Icons';
import Modal from '../Modal';
import { API_BASE_URL } from '@/src/config.js';
declare const L: any;

type PurchaseMode = 'request' | 'accept';
type LocationField = 'delivery';
type Currency = 'IRT' | 'USD' | 'CAD' | 'IQD';

interface ImageFile {
  id: number;
  url: string;
  file: File;
}

// --- Helper Components ---
interface InputWithIconProps { id: string; icon?: React.FC<{ className?: string }>; label: string; placeholder: string; type?: string; value?: string; onChange?: (e: React.ChangeEvent<HTMLInputElement>) => void; }
const InputWithIcon: React.FC<InputWithIconProps> = ({ id, icon: Icon, label, placeholder, type = 'text', value, onChange }) => (
    <div><label htmlFor={id} className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">{label}</label><div className="relative">{Icon && (<div className="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none"><Icon className="w-5 h-5 text-gray-400" /></div>)}<input type={type} id={id} className={`w-full ${Icon ? 'pr-10' : 'pr-4'} pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition`} placeholder={placeholder} value={value} onChange={onChange}/></div></div>
);

interface TextareaWithIconProps { id: string; icon: React.FC<{ className?: string }>; label: string; placeholder: string; value?: string; onChange?: (e: React.ChangeEvent<HTMLTextAreaElement>) => void; }
const TextareaWithIcon: React.FC<TextareaWithIconProps> = ({ id, icon: Icon, label, placeholder, value, onChange }) => (
    <div><label htmlFor={id} className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">{label}</label><div className="relative"><div className="absolute top-3.5 right-0 flex items-center pr-3 pointer-events-none"><Icon className="w-5 h-5 text-gray-400" /></div><textarea id={id} className="w-full pr-10 pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition min-h-[100px] resize-y" placeholder={placeholder} value={value} onChange={onChange}/></div></div>
);

interface SelectWithIconProps { id: string; icon: React.FC<{ className?: string }>; label: string; children: React.ReactNode; value?: string; onChange?: (e: React.ChangeEvent<HTMLSelectElement>) => void; }
const SelectWithIcon: React.FC<SelectWithIconProps> = ({ id, icon: Icon, label, children, value, onChange }) => {
    const [isOpen, setIsOpen] = useState(false);
    const selectorRef = useRef<HTMLDivElement>(null);

    const options = React.Children.toArray(children)
        .filter((child): child is React.ReactElement<React.OptionHTMLAttributes<HTMLOptionElement>> => 
            React.isValidElement(child) && child.type === 'option')
        .map(option => ({
            value: option.props.value?.toString() ?? (option.props.children?.toString() || ''),
            label: option.props.children?.toString() || ''
        }));
    
    const selectedOption = options.find(opt => opt.value === value);

    useEffect(() => {
        const handleClickOutside = (event: MouseEvent) => {
            if (selectorRef.current && !selectorRef.current.contains(event.target as Node)) {
                setIsOpen(false);
            }
        };
        document.addEventListener("mousedown", handleClickOutside);
        return () => document.removeEventListener("mousedown", handleClickOutside);
    }, []);

    const handleSelect = (selectedValue: string) => {
        if (onChange) {
            const syntheticEvent = {
                target: { value: selectedValue },
                currentTarget: { value: selectedValue },
            } as React.ChangeEvent<HTMLSelectElement>;
            onChange(syntheticEvent);
        }
        setIsOpen(false);
    };

    return (
        <div>
            <label htmlFor={id} className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">{label}</label>
            <div className="relative" ref={selectorRef}>
                <div className="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none"><Icon className="w-5 h-5 text-gray-400" /></div>
                <button
                    type="button"
                    id={id}
                    onClick={() => setIsOpen(!isOpen)}
                    className="w-full flex items-center justify-between text-right pr-10 pl-10 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition"
                >
                    <span className="text-gray-800 dark:text-gray-200">{selectedOption?.label}</span>
                    <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                        <ChevronDownIcon className={`w-5 h-5 text-gray-400 transition-transform ${isOpen ? 'rotate-180' : ''}`} />
                    </div>
                </button>
                {isOpen && (
                    <ul className="absolute z-20 w-full mt-1 bg-custom-light-card dark:bg-custom-dark-blue-light border border-gray-300 dark:border-gray-600 rounded-lg shadow-lg max-h-60 overflow-y-auto">
                        {options.map(option => (
                            <li
                                key={option.value}
                                onClick={() => handleSelect(option.value)}
                                className="px-4 py-2 cursor-pointer hover:bg-gray-100 dark:hover:bg-custom-dark-blue text-gray-800 dark:text-gray-200 transition-colors"
                            >
                                {option.label}
                            </li>
                        ))}
                    </ul>
                )}
            </div>
        </div>
    );
};
interface InputWithLocationButtonProps { id: string; label: string; placeholder: string; value: string; onButtonClick: () => void; }
const InputWithLocationButton: React.FC<InputWithLocationButtonProps> = ({ id, label, placeholder, value, onButtonClick }) => (
    <div><label htmlFor={id} className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">{label}</label><div className="relative flex items-center"><div className="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none"><MapPinIcon className="w-5 h-5 text-gray-400" /></div><input type="text" id={id} className="w-full pr-10 pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition" placeholder={placeholder} value={value} readOnly onClick={onButtonClick} style={{cursor: 'pointer'}}/><button type="button" onClick={onButtonClick} className="absolute left-2 text-sm bg-custom-purple text-white px-3 py-1 rounded-md hover:bg-custom-light-purple transition-colors">ثبت</button></div></div>
);

interface ToggleSwitchProps { enabled: boolean; setEnabled: (enabled: boolean) => void; }
const ToggleSwitch: React.FC<ToggleSwitchProps> = ({ enabled, setEnabled }) => (
    <button type="button" dir="ltr" className={`${enabled ? 'bg-custom-purple' : 'bg-gray-200 dark:bg-gray-600'} relative inline-flex h-6 w-11 flex-shrink-0 cursor-pointer rounded-full border-2 border-transparent transition-colors duration-200 ease-in-out focus:outline-none focus:ring-2 focus:ring-custom-purple focus:ring-offset-2`} onClick={() => setEnabled(!enabled)}>
        <span aria-hidden="true" className={`${enabled ? 'translate-x-5' : 'translate-x-0'} pointer-events-none inline-block h-5 w-5 transform rounded-full bg-white shadow-lg ring-0 transition duration-200 ease-in-out`}/>
    </button>
);

const ToggleableContact: React.FC<{enabled: boolean, setEnabled: (e: boolean) => void}> = ({enabled, setEnabled}) => (
    <div className="flex items-center justify-between bg-custom-light-bg dark:bg-custom-dark-blue p-4 rounded-lg"><div className="flex items-center"><PhoneIcon className="w-5 h-5 text-gray-500 dark:text-gray-400 ml-3"/><div><p className="font-medium text-gray-800 dark:text-gray-200">نمایش شماره تماس</p><p className="text-xs text-gray-500 dark:text-gray-400">برای هماهنگی شماره شما نمایش داده شود؟</p></div></div><ToggleSwitch enabled={enabled} setEnabled={setEnabled} /></div>
);

const RegisterOnlinePurchasePage: React.FC = () => {
    const [purchaseMode, setPurchaseMode] = useState<PurchaseMode>('request');
    const [images, setImages] = useState<ImageFile[]>([]);
    const [currency, setCurrency] = useState<Currency>('IRT');
    const [showPhoneNumber, setShowPhoneNumber] = useState(false);
    
    const [modalState, setModalState] = useState<{isOpen: boolean; field: LocationField | null}>({isOpen: false, field: null});
    const [deliveryLocation, setDeliveryLocation] = useState<{lat: number, lng: number} | null>(null);
    const [deliveryCountry, setDeliveryCountry] = useState('');
    const [deliveryCity, setDeliveryCity] = useState('');
    const [deliveryAddress, setDeliveryAddress] = useState('');

    const [shopperCountry, setShopperCountry] = useState('');
    const [shopperCity, setShopperCity] = useState('');

    const deliveryLocationText = deliveryLocation ? `${deliveryLocation.lat.toFixed(5)}, ${deliveryLocation.lng.toFixed(5)}` : '';

    const currencies = [
        { value: 'IRT', label: 'تومان' },
        { value: 'USD', label: 'دلار آمریکا' },
        { value: 'CAD', label: 'دلار کانادا' },
        { value: 'IQD', label: 'دینار' },
    ];

    const handleImageUpload = (event: React.ChangeEvent<HTMLInputElement>) => {
        const files = event.target.files;
        if (files) {
            const newImages: ImageFile[] = Array.from(files).map((file: File) => ({
                id: Date.now() + Math.random(),
                url: URL.createObjectURL(file),
                file: file
            }));
            setImages(prev => [...prev, ...newImages]);
        }
    };
    
    const removeImage = (id: number) => {
        setImages(prev => prev.filter(img => img.id !== id));
    };

    const onDrop = useCallback((event: React.DragEvent<HTMLLabelElement>) => {
        event.preventDefault();
        event.stopPropagation();
        const files = event.dataTransfer.files;
        if (files && files.length > 0) {
            const newImages: ImageFile[] = Array.from(files)
                .filter((file: File) => file.type.startsWith('image/'))
                .map((file: File) => ({
                    id: Date.now() + Math.random(),
                    url: URL.createObjectURL(file),
                    file: file
                }));
            setImages(prev => [...prev, ...newImages]);
        }
    }, []);

    const onDragOver = (event: React.DragEvent<HTMLLabelElement>) => {
        event.preventDefault();
        event.stopPropagation();
    };

    const openModal = (field: LocationField) => {
        setModalState({ isOpen: true, field });
    };

    const closeModal = () => {
        setModalState({ isOpen: false, field: null });
    };

    const confirmLocation = (location: {lat: number, lng: number}) => {
        if (modalState.field === 'delivery') {
            setDeliveryLocation(location);
        }
        closeModal();
    };

    return (
        <div className="flex flex-col space-y-8">
             <div>
                <h1 className="text-3xl font-bold text-gray-800 dark:text-white">
                    {purchaseMode === 'request' ? 'ثبت درخواست خرید اینترنتی' : 'اعلام آمادگی برای انجام خرید'}
                </h1>
                <p className="text-gray-500 dark:text-gray-400 mt-1">
                    {purchaseMode === 'request' ? 'اطلاعات کالای مورد نظر خود برای خرید را ثبت کنید.' : 'اطلاعات خود را برای پذیرش سفارشات خرید ثبت کنید.'}
                </p>
            </div>

            <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
                 <h3 className="text-lg font-semibold mb-4 text-center text-gray-800 dark:text-gray-200">نوع فعالیت خود را انتخاب کنید</h3>
                 <div className="max-w-sm mx-auto">
                    <div className="bg-custom-light-bg dark:bg-custom-dark-blue rounded-xl shadow-inner p-1">
                        <div className="relative flex rounded-lg overflow-hidden">
                            <span className={`absolute top-0 bottom-0 right-0 w-1/2 bg-custom-purple rounded-lg shadow-md transition-transform duration-300 ease-in-out ${purchaseMode === 'accept' ? '-translate-x-full' : 'translate-x-0'}`} aria-hidden="true"/>
                            <button type="button" onClick={() => setPurchaseMode('request')} className={`relative z-10 flex w-1/2 items-center justify-center gap-2 rounded-lg py-2.5 text-md font-semibold transition-colors duration-300 ${purchaseMode === 'request' ? 'text-white' : 'text-gray-600 dark:text-gray-300'}`} aria-label="درخواست خرید">
                                <ECommerceIcon className="h-5 w-5" /><span>درخواست خرید</span>
                            </button>
                            <button type="button" onClick={() => setPurchaseMode('accept')} className={`relative z-10 flex w-1/2 items-center justify-center gap-2 rounded-lg py-2.5 text-md font-semibold transition-colors duration-300 ${purchaseMode === 'accept' ? 'text-white' : 'text-gray-600 dark:text-gray-300'}`} aria-label="قبول کردن خرید">
                                <UserIcon className="h-5 w-5" /><span>قبول کردن خرید</span>
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            {purchaseMode === 'request' ? (
                <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
                    <div className="lg:col-span-1 space-y-8">
                         <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
                            <h3 className="text-lg font-semibold mb-4">تصویر کالا</h3>
                            <label htmlFor="file-upload" className="flex flex-col items-center justify-center w-full h-48 border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-lg cursor-pointer bg-custom-light-bg dark:bg-custom-dark-blue hover:bg-gray-100 dark:hover:bg-gray-800 transition-colors" onDrop={onDrop} onDragOver={onDragOver}>
                                <div className="flex flex-col items-center justify-center pt-5 pb-6">
                                    <UploadCloudIcon className="w-10 h-10 mb-3 text-gray-400" />
                                    <p className="mb-2 text-sm text-gray-500 dark:text-gray-400"><span className="font-semibold">برای آپلود کلیک کنید</span> یا بکشید</p>
                                </div>
                                <input id="file-upload" type="file" className="hidden" multiple accept="image/*" onChange={handleImageUpload} />
                            </label>
                        </div>
                        {images.length > 0 && (
                             <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
                                <h3 className="text-lg font-semibold mb-4">پیش‌نمایش</h3>
                                 <div className="grid grid-cols-3 gap-4">
                                    {images.map(image => (<div key={image.id} className="relative group"><img src={image.url} alt="Preview" className="w-full h-24 object-cover rounded-md" /><button onClick={() => removeImage(image.id)} className="absolute top-1 right-1 bg-red-500 text-white rounded-full p-1 opacity-0 group-hover:opacity-100 transition-opacity"><XIcon className="w-4 h-4" /></button></div>))}
                                </div>
                            </div>
                        )}
                    </div>
                    <div className="lg:col-span-2">
                         <form className="space-y-8">
                            <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl shadow-sm">
                                <h3 className="text-xl font-semibold mb-6 border-b border-gray-200 dark:border-gray-700 pb-4">۱. اطلاعات کالا</h3>
                                <div className="mt-6 space-y-6"><InputWithIcon icon={TypeIcon} id="title" placeholder="مثال: گوشی موبایل سامسونگ" label="عنوان کالا" /><InputWithIcon icon={LinkIcon} id="link" placeholder="https://example.com/product" label="لینک خرید" /><TextareaWithIcon icon={ListIcon} id="description" placeholder="جزئیات بیشتر در مورد کالا، رنگ، مدل و..." label="توضیحات (اختیاری)" /></div>
                            </div>
                            <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl shadow-sm">
                                <h3 className="text-xl font-semibold mb-6 border-b border-gray-200 dark:border-gray-700 pb-4">۲. محل تحویل کالا</h3>
                                <div className="mt-6">
                                    <div className="border border-gray-200 dark:border-gray-700 rounded-xl p-4 space-y-4">
                                        <h3 className="font-semibold text-lg text-gray-800 dark:text-gray-200">مقصد</h3>
                                        <InputWithLocationButton onButtonClick={() => openModal('delivery')} value={deliveryLocationText} id="destination" placeholder="موقعیت ثبت نشده" label="انتخاب از روی نقشه" />
                                        <div className="text-center text-sm text-gray-500 dark:text-gray-400">یا</div>
                                        <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                                           <InputWithIcon icon={CrmIcon} id="destination_country" placeholder="مثال: ایران" label="کشور" value={deliveryCountry} onChange={(e) => setDeliveryCountry(e.target.value)} />
                                           <InputWithIcon icon={MapPinIcon} id="destination_city" placeholder="مثال: تهران" label="شهر" value={deliveryCity} onChange={(e) => setDeliveryCity(e.target.value)} />
                                        </div>
                                        <InputWithIcon icon={HomeIcon} id="destination_address" placeholder="آدرس کامل، پلاک و واحد" label="آدرس دستی" value={deliveryAddress} onChange={(e) => setDeliveryAddress(e.target.value)} />
                                    </div>
                                </div>
                            </div>
                            <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl shadow-sm">
                                <h3 className="text-xl font-semibold mb-6 border-b border-gray-200 dark:border-gray-700 pb-4">۳. قیمت‌گذاری و اطلاعات تماس</h3>
                                <div className="mt-6 space-y-6">
                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6"><InputWithIcon icon={DollarSignIcon} id="item_price" placeholder="قیمت کالا" label="قیمت کالا" type="number" /><InputWithIcon icon={DollarSignIcon} id="proposed_fee" placeholder="مبلغ پیشنهادی شما" label="هزینه پیشنهادی انجام" type="number" /></div>
                                     <SelectWithIcon id="currency" icon={DollarSignIcon} label="واحد پولی" value={currency} onChange={(e) => setCurrency(e.target.value as Currency)}>
                                        {currencies.map(({ value, label }) => (<option key={value} value={value}>{label}</option>))}
                                    </SelectWithIcon>
                                    <ToggleableContact enabled={showPhoneNumber} setEnabled={setShowPhoneNumber} />
                                </div>
                            </div>
                            <button type="submit" className="w-full bg-custom-purple hover:bg-custom-light-purple text-white font-bold py-3 px-4 rounded-lg transition-colors duration-300 shadow-lg hover:shadow-xl focus:outline-none focus:ring-4 focus:ring-custom-purple/50">ثبت درخواست خرید</button>
                        </form>
                    </div>
                </div>
            ) : (
                <div className="w-full max-w-4xl mx-auto">
                    <form className="space-y-8">
                         <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl shadow-sm">
                            <h3 className="text-xl font-semibold mb-6 border-b border-gray-200 dark:border-gray-700 pb-4">۱. محل فعالیت شما</h3>
                             <div className="mt-6 grid grid-cols-1 md:grid-cols-2 gap-6"><InputWithIcon icon={CrmIcon} id="shopper_country" placeholder="مثال: آمریکا" label="کشور" value={shopperCountry} onChange={(e) => setShopperCountry(e.target.value)} /><InputWithIcon icon={MapPinIcon} id="shopper_city" placeholder="مثال: نیویورک" label="شهر" value={shopperCity} onChange={(e) => setShopperCity(e.target.value)} /></div>
                        </div>
                        <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl shadow-sm">
                            <h3 className="text-xl font-semibold mb-6 border-b border-gray-200 dark:border-gray-700 pb-4">۲. جزئیات خدمات</h3>
                             <div className="mt-6 space-y-6"><TextareaWithIcon icon={DollarSignIcon} id="fee_description" label="توضیحات کارمزد" placeholder="نحوه محاسبه کارمزد خود را توضیح دهید..." /><TextareaWithIcon icon={ListIcon} id="capabilities" label="توضیحات بیشتر" placeholder="اطلاعات بیشتر در مورد خدماتی که ارائه می‌دهید..." /></div>
                        </div>
                        <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl shadow-sm">
                            <h3 className="text-xl font-semibold mb-6 border-b border-gray-200 dark:border-gray-700 pb-4">۳. اطلاعات تماس</h3>
                             <div className="mt-6"><ToggleableContact enabled={showPhoneNumber} setEnabled={setShowPhoneNumber} /></div>
                        </div>
                        <button type="submit" className="w-full bg-custom-purple hover:bg-custom-light-purple text-white font-bold py-3 px-4 rounded-lg transition-colors duration-300 shadow-lg hover:shadow-xl focus:outline-none focus:ring-4 focus:ring-custom-purple/50">ثبت آمادگی برای خرید</button>
                    </form>
                </div>
            )}

            <LocationModalWrapper 
                isOpen={modalState.isOpen}
                onClose={closeModal}
                onConfirm={confirmLocation}
                field={modalState.field}
            />
        </div>
    );
};


interface LocationModalWrapperProps { isOpen: boolean; onClose: () => void; onConfirm: (location: {lat: number, lng: number}) => void; field: LocationField | null; }
const LocationModalWrapper: React.FC<LocationModalWrapperProps> = ({ isOpen, onClose, onConfirm, field }) => {
    const mapRef = useRef<any>(null); const markerRef = useRef<any>(null); const mapContainerRef = useRef<HTMLDivElement>(null);
    const [selectedLocation, setSelectedLocation] = useState<{lat: number, lng: number} | null>(null);
    const [searchQuery, setSearchQuery] = useState(''); const [isSearching, setIsSearching] = useState(false);

    useEffect(() => {
        if (!isOpen) return;
        const cleanup = () => { if (mapRef.current) { mapRef.current.remove(); mapRef.current = null; } markerRef.current = null; setSelectedLocation(null); setSearchQuery(''); };
        if (mapContainerRef.current && !mapRef.current) {
            const map = L.map(mapContainerRef.current).setView([35.6892, 51.3890], 10);
            mapRef.current = map;
            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', { attribution: '&copy; OpenStreetMap' }).addTo(map);
            map.on('click', (e: any) => { const { lat, lng } = e.latlng; setSelectedLocation({ lat, lng }); if (markerRef.current) { markerRef.current.setLatLng(e.latlng); } else { markerRef.current = L.marker(e.latlng).addTo(mapRef.current); } });
            setTimeout(() => map.invalidateSize(), 100);
        }
        return cleanup;
    }, [isOpen]);

    const handleSearch = async () => { /* ... search logic ... */ };
    const handleUseCurrentLocation = () => { /* ... geolocation logic ... */ };
    const handleConfirm = () => { if (selectedLocation) onConfirm(selectedLocation); };
    const title = 'ثبت موقعیت تحویل';

    const modalFooter = (
        <><button onClick={onClose} className="py-2 px-5 rounded-lg text-gray-700 dark:text-gray-300 font-semibold hover:bg-gray-200 dark:hover:bg-gray-700/50 transition-colors">انصراف</button><button onClick={handleConfirm} disabled={!selectedLocation} className="py-2.5 px-6 rounded-xl bg-gradient-to-br from-custom-purple to-custom-light-purple text-white font-semibold hover:shadow-lg hover:shadow-custom-purple/30 transition-all duration-300 disabled:from-gray-400 disabled:to-gray-500 disabled:shadow-none disabled:cursor-not-allowed transform disabled:transform-none hover:-translate-y-0.5">تایید موقعیت</button></>
    );

    return (
        <Modal isOpen={isOpen} onClose={onClose} title={title} footer={modalFooter}>
            <div className="space-y-5">
                <div className="flex items-center gap-2">
                    <div className="relative flex-grow"><div className="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none"><SearchIcon className="w-5 h-5 text-gray-400" /></div><input type="text" value={searchQuery} onChange={(e) => setSearchQuery(e.target.value)} onKeyDown={(e) => { if (e.key === 'Enter') handleSearch(); }} placeholder="جستجوی آدرس یا مکان..." className="w-full pr-10 pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition"/></div>
                    <button onClick={handleSearch} disabled={isSearching} className="flex-shrink-0 flex items-center justify-center w-28 py-2.5 px-4 rounded-lg bg-custom-purple text-white font-semibold hover:bg-custom-light-purple transition-colors disabled:bg-gray-400 disabled:cursor-wait">{isSearching ? (<div className="w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin"></div>) : (<span>جستجو</span>)}</button>
                </div>
                <button onClick={handleUseCurrentLocation} className="w-full flex items-center justify-center gap-2 py-3 px-4 rounded-xl bg-gradient-to-br from-custom-cyan to-teal-400 text-white font-semibold hover:shadow-lg hover:shadow-custom-cyan/30 transition-all duration-300 transform hover:-translate-y-0.5"><CrosshairIcon className="w-5 h-5" /><span>استفاده از موقعیت فعلی من</span></button>
                <div className="relative my-4"><div className="absolute inset-0 flex items-center" aria-hidden="true"><div className="w-full border-t border-gray-300 dark:border-gray-600" /></div><div className="relative flex justify-center"><span className="bg-custom-light-card dark:bg-custom-dark-blue-light px-3 text-sm text-gray-500 dark:text-gray-400">یا</span></div></div>
                <p className="text-center text-sm text-gray-500 dark:text-gray-400 -mt-2 mb-2">برای انتخاب، روی نقشه کلیک کنید.</p>
                <div ref={mapContainerRef} className="w-full h-80 bg-gray-200 dark:bg-gray-700 rounded-lg cursor-pointer z-0 border border-gray-300 dark:border-gray-600 shadow-inner"></div>
            </div>
        </Modal>
    );
};

export default RegisterOnlinePurchasePage;