import React, { useEffect, useRef, useState } from 'react';
import { ArrowRightIcon, ScaleIcon, TagIcon, MapPinIcon, DollarSignIcon, CheckIcon, XIcon, StarIcon, ClockIcon } from '../Icons';
import { API_BASE_URL } from '@/src/config.js';
declare const L: any;

interface ShipmentImage {
    id: number | null;
    url: string;
    path: string | null;
}

interface ShipmentData {
    id: number;
    type: 'send' | 'carry';
    title: string;
    description: string;
    status: 'pending' | 'approved' | 'rejected';
    created_at: string;
    updated_at: string;
    origin: {
        country: string;
        city: string;
        address: string;
        coordinates: string;
        lat: number;
        lng: number;
    };
    destination: {
        country: string;
        city: string;
        address: string;
        coordinates: string;
        lat: number;
        lng: number;
    };
    weight?: number;
    carry_weight_capacity?: number;
    category?: string;
    fee: number;
    currency: string;
    price_type: string;
    user: {
        id: number;
        name: string;
        avatar: string;
        rating: number;
        email?: string;
        phone?: string;
        show_phone: boolean;
    };
    images: ShipmentImage[];
    // فیلدهای مخصوص carry
    departure_date?: string;
    arrival_date?: string;
    // فیلدهای مخصوص send
    rejection_reason?: string;
}

interface ShipmentDetailPageProps {
    shipmentId: string | null;
    onNavigate: (page: string) => void;
    isOwnerView?: boolean;
    showError?: (message: string) => void;
    showSuccess?: (message: string) => void;
}

// تابع ترجمه دسته‌بندی‌ها از انگلیسی به فارسی
const translateCategory = (category?: string): string => {
    if (!category) return 'نامشخص';

    const categoryTranslations: Record<string, string> = {
        'documents': 'مدارک و اسناد',
        'electronics': 'لوازم الکترونیکی',
        'clothing': 'پوشاک',
        'furniture': 'اثاثیه منزل',
        'miscellaneous': 'متفرقه'
    };

    return categoryTranslations[category] || category;
};

const ShipmentDetailPage: React.FC<ShipmentDetailPageProps> = ({
    shipmentId,
    onNavigate,
    isOwnerView = false,
    showError = (msg) => console.error(msg),
    showSuccess = (msg) => console.log(msg)
}) => {
    const [shipment, setShipment] = useState<ShipmentData | null>(null);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState<string | null>(null);
    const [rejectReason, setRejectReason] = useState('');
    const [showRejectModal, setShowRejectModal] = useState(false);
    const mapContainerRef = useRef<HTMLDivElement>(null);
    const mapRef = useRef<any>(null);

    // تابع تایید درخواست
    const handleApprove = async () => {
        if (!shipment) return;

        try {
            const token = localStorage.getItem('auth_token');
            if (!token) {
                showError('لطفاً دوباره وارد شوید.');
                return;
            }

            const response = await fetch(`${API_BASE_URL}/admin/shipments/${shipment.id}/approve`, {
                method: 'PUT',
                headers: {
                    'X-Auth-Token': token,
                    'Content-Type': 'application/json',
                },
            });

            if (!response.ok) {
                const errorData = await response.json().catch(() => ({}));
                throw new Error(errorData.error || 'خطا در تأیید درخواست');
            }

            showSuccess('درخواست با موفقیت تأیید شد');
            // به‌روزرسانی وضعیت در UI
            setShipment(prev => prev ? { ...prev, status: 'approved' } : null);
        } catch (error: any) {
            showError(error.message || 'خطا در ارتباط با سرور');
        }
    };

    // تابع رد درخواست
    const handleReject = async () => {
        if (!shipment) return;

        if (!rejectReason.trim()) {
            showError('لطفاً دلیل رد درخواست را وارد کنید');
            return;
        }

        try {
            const token = localStorage.getItem('auth_token');
            if (!token) {
                showError('لطفاً دوباره وارد شوید.');
                return;
            }

            const response = await fetch(`${API_BASE_URL}/admin/shipments/${shipment.id}/reject`, {
                method: 'PUT',
                headers: {
                    'X-Auth-Token': token,
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    reason: rejectReason
                })
            });

            if (!response.ok) {
                const errorData = await response.json().catch(() => ({}));
                throw new Error(errorData.error || 'خطا در رد درخواست');
            }

            showSuccess('درخواست با موفقیت رد شد');
            // به‌روزرسانی وضعیت در UI
            setShipment(prev => prev ? { ...prev, status: 'rejected', rejection_reason: rejectReason } : null);
            setShowRejectModal(false);
            setRejectReason('');
        } catch (error: any) {
            showError(error.message || 'خطا در ارتباط با سرور');
        }
    };

    useEffect(() => {
        if (!shipmentId) {
            setError('شناسه مرسوله نامعتبر است');
            setLoading(false);
            return;
        }

        const fetchShipment = async () => {
            try {
                const token = localStorage.getItem('auth_token');
                if (!token) {
                    setError('لطفاً دوباره وارد شوید');
                    return;
                }

                const response = await fetch(`${API_BASE_URL}/admin/shipments/${shipmentId}`, {
                    headers: {
                        'X-Auth-Token': token,
                        'Content-Type': 'application/json',
                    },
                });

                if (!response.ok) {
                    const errorData = await response.json().catch(() => ({}));
                    throw new Error(errorData.error || 'خطا در دریافت جزئیات مرسوله');
                }

                const result = await response.json();
                // اصلاح: دسترسی به داده‌ها از طریق result.data
                setShipment(result.data);
            } catch (err: any) {
                setError(err.message || 'خطا در ارتباط با سرور');
                console.error('Fetch shipment error:', err);
            } finally {
                setLoading(false);
            }
        };

        fetchShipment();
    }, [shipmentId]);

    useEffect(() => {
        if (!shipment || !mapContainerRef.current || mapRef.current) return;

        // اطمینان از مختصات معتبر
        if (isNaN(shipment.origin.lat) || isNaN(shipment.origin.lng) ||
            isNaN(shipment.destination.lat) || isNaN(shipment.destination.lng)) {
            console.error('Invalid coordinates for map');
            return;
        }

        const map = L.map(mapContainerRef.current).setView([shipment.origin.lat, shipment.origin.lng], 4);
        mapRef.current = map;

        L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
            attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors'
        }).addTo(map);

        const originMarker = L.marker([shipment.origin.lat, shipment.origin.lng]).addTo(map);
        originMarker.bindPopup(`<b>مبدأ:</b> ${shipment.origin.city}`);

        const destMarker = L.marker([shipment.destination.lat, shipment.destination.lng]).addTo(map);
        destMarker.bindPopup(`<b>مقصد:</b> ${shipment.destination.city}`);

        const latlngs = [
            [shipment.origin.lat, shipment.origin.lng],
            [shipment.destination.lat, shipment.destination.lng]
        ];
        L.polyline(latlngs, { color: '#8a5cf6', weight: 3, dashArray: '5, 10' }).addTo(map);

        map.fitBounds(latlngs, { padding: [50, 50] });

        return () => {
            if (mapRef.current) {
                mapRef.current.remove();
                mapRef.current = null;
            }
        };
    }, [shipment]);

    if (loading) {
        return (
            <div className="flex items-center justify-center min-h-[400px]">
                <div className="animate-spin rounded-full h-12 w-12 border-4 border-custom-purple border-t-transparent"></div>
            </div>
        );
    }

    if (error) {
        return (
            <div className="text-center p-8">
                <h2 className="text-xl font-bold text-red-500">{error}</h2>
                <button onClick={() => onNavigate('admin-shipments')} className="mt-4 text-custom-purple hover:underline">بازگشت به لیست مرسولات</button>
            </div>
        );
    }

    if (!shipment) {
        return (
            <div className="text-center p-8">
                <h2 className="text-xl font-bold">مرسوله یافت نشد</h2>
                <button onClick={() => onNavigate('admin-shipments')} className="mt-4 text-custom-purple hover:underline">بازگشت به لیست مرسولات</button>
            </div>
        );
    }

    // ایمن‌سازی fee و currency
    const fee = typeof shipment.fee === 'number' ? shipment.fee : 0;
    const currency = shipment.currency || 'IRT';

    // تعیین وضعیت مرسوله
    const getStatusBadge = () => {
        switch (shipment.status) {
            case 'approved':
                return <span className="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-green-100 text-green-800">تأیید شده</span>;
            case 'rejected':
                return <span className="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-red-100 text-red-800">رد شده</span>;
            default:
                return <span className="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">در انتظار بررسی</span>;
        }
    };

    // تعیین عنوان بر اساس نوع مرسوله
    const getShipmentTypeLabel = () => {
        return shipment.type === 'send' ? 'ارسال مرسوله' : 'همراهی در سفر';
    };

    return (
        <div className="flex flex-col space-y-6">
            <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
                <div>
                    <h1 className="text-3xl font-bold text-gray-800 dark:text-white">{shipment.title}</h1>
                    <div className="flex items-center gap-2 mt-2">
                        <span className="text-sm text-gray-500 dark:text-gray-400">{getShipmentTypeLabel()}</span>
                        {getStatusBadge()}
                    </div>
                </div>
                <button onClick={() => onNavigate('admin-shipments')} className="flex items-center gap-2 py-2 px-4 rounded-lg text-gray-600 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-custom-dark-blue transition-colors self-start sm:self-center">
                    <ArrowRightIcon className="w-5 h-5" />
                    <span>بازگشت به لیست</span>
                </button>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
                <div className="lg:col-span-2 space-y-8">
                    {/* گالری تصاویر - فقط برای نوع send نمایش داده می‌شود */}
                    {shipment.type === 'send' && (
                        <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
                            {shipment.images && shipment.images.length > 0 ? (
                                <>
                                    <img src={shipment.images[0].url} alt={shipment.title} className="w-full h-auto max-h-[400px] object-cover rounded-xl" />

                                    {/* نمایش سایر تصاویر اگر وجود داشته باشند */}
                                    {shipment.images.length > 1 && (
                                        <div className="grid grid-cols-4 gap-2 mt-4">
                                            {shipment.images.slice(1, 5).map((img, index) => (
                                                <img key={index} src={img.url} alt={`${shipment.title} ${index + 2}`} className="w-full h-20 object-cover rounded-lg cursor-pointer hover:opacity-80" />
                                            ))}
                                        </div>
                                    )}
                                </>
                            ) : (
                                <div className="w-full h-[400px] bg-gray-200 dark:bg-gray-700 rounded-xl flex items-center justify-center">
                                    <span className="text-gray-500 dark:text-gray-400">تصویری موجود نیست</span>
                                </div>
                            )}
                        </div>
                    )}

                    {/* توضیحات مرسوله */}
                    <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl shadow-sm">
                        <h3 className="text-xl font-semibold mb-4 border-b pb-3 border-gray-200 dark:border-gray-700">توضیحات مرسوله</h3>
                        <p className="text-gray-600 dark:text-gray-300 leading-relaxed">{shipment.description}</p>

                        {/* نمایش دلیل رد شدن اگر وجود داشته باشد */}
                        {shipment.status === 'rejected' && shipment.rejection_reason && (
                            <div className="mt-4 p-4 bg-red-50 dark:bg-red-900/20 rounded-lg">
                                <h4 className="font-semibold text-red-800 dark:text-red-400 mb-2">دلیل رد درخواست:</h4>
                                <p className="text-red-700 dark:text-red-300">{shipment.rejection_reason}</p>
                            </div>
                        )}
                    </div>

                    {/* مسیر مرسوله */}
                    <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl shadow-sm">
                        <h3 className="text-xl font-semibold mb-4 border-b pb-3 border-gray-200 dark:border-gray-700">مسیر مرسوله</h3>
                        <div ref={mapContainerRef} className="w-full h-80 bg-gray-200 dark:bg-gray-700 rounded-lg z-0 border border-gray-300 dark:border-gray-600 shadow-inner"></div>

                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mt-6">
                            <div className="flex items-start gap-3">
                                <MapPinIcon className="w-5 h-5 text-green-500 mt-1 flex-shrink-0" />
                                <div>
                                    <h4 className="font-semibold text-gray-800 dark:text-gray-200">مبدأ</h4>
                                    <p className="text-gray-600 dark:text-gray-400">{shipment.origin.city}, {shipment.origin.country}</p>
                                    <p className="text-sm text-gray-500 dark:text-gray-500">{shipment.origin.address}</p>
                                </div>
                            </div>
                            <div className="flex items-start gap-3">
                                <MapPinIcon className="w-5 h-5 text-red-500 mt-1 flex-shrink-0" />
                                <div>
                                    <h4 className="font-semibold text-gray-800 dark:text-gray-200">مقصد</h4>
                                    <p className="text-gray-600 dark:text-gray-400">{shipment.destination.city}, {shipment.destination.country}</p>
                                    <p className="text-sm text-gray-500 dark:text-gray-500">{shipment.destination.address}</p>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* اطلاعات اضافی بر اساس نوع مرسوله */}
                    {shipment.type === 'send' && (
                        <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl shadow-sm">
                            <h3 className="text-xl font-semibold mb-4 border-b pb-3 border-gray-200 dark:border-gray-700">جزئیات مرسوله</h3>
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <InfoRow icon={TagIcon} label="دسته‌بندی" value={translateCategory(shipment.category)} />
                                <InfoRow icon={ScaleIcon} label="وزن" value={`${shipment.weight || 0} کیلوگرم`} />
                            </div>
                        </div>
                    )}

                    {shipment.type === 'carry' && (
                        <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl shadow-sm">
                            <h3 className="text-xl font-semibold mb-4 border-b pb-3 border-gray-200 dark:border-gray-700">جزئیات سفر</h3>
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <InfoRow icon={ScaleIcon} label="ظرفیت حمل" value={`${shipment.carry_weight_capacity || 0} کیلوگرم`} />
                                <InfoRow icon={ClockIcon} label="تاریخ حرکت" value={shipment.departure_date || 'نامشخص'} />
                                <InfoRow icon={ClockIcon} label="تاریخ رسیدن" value={shipment.arrival_date || 'نامشخص'} />
                            </div>
                        </div>
                    )}
                </div>

                <div className="lg:col-span-1 space-y-6">
                    <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm space-y-4 sticky top-28">
                        <UserInfoCard user={shipment.user} />
                        <div className="border-t border-gray-200 dark:border-gray-700 my-4" />

                        {shipment.type === 'send' && (
                            <InfoRow icon={TagIcon} label="دسته‌بندی" value={translateCategory(shipment.category)} />
                        )}

                        <InfoRow
                            icon={ScaleIcon}
                            label="وزن/ظرفیت"
                            value={shipment.weight ? `${shipment.weight} کیلوگرم` : `${shipment.carry_weight_capacity || 0} کیلوگرم`}
                        />

                        <InfoRow icon={MapPinIcon} label="مبدأ" value={`${shipment.origin.city}, ${shipment.origin.country}`} />
                        <InfoRow icon={MapPinIcon} label="مقصد" value={`${shipment.destination.city}, ${shipment.destination.country}`} />

                        <InfoRow
                            icon={DollarSignIcon}
                            label="هزینه حمل"
                            value={shipment.price_type === 'negotiable' ? 'توافقی' : `${fee.toLocaleString()} ${currency}`}
                            valueClassName="text-green-500 font-bold"
                        />

                        <div className="border-t border-gray-200 dark:border-gray-700 my-4" />

                        <div className="text-sm text-gray-500 dark:text-gray-400">
                            <p>ایجاد شده: {new Date(shipment.created_at).toLocaleDateString('fa-IR')}</p>
                            <p>آخرین به‌روزرسانی: {new Date(shipment.updated_at).toLocaleDateString('fa-IR')}</p>
                        </div>

                        {isOwnerView && shipment.status === 'pending' && (
                            <div className="grid grid-cols-2 gap-3 pt-4">
                                <button
                                    onClick={() => setShowRejectModal(true)}
                                    className="w-full flex items-center justify-center gap-2 bg-red-500/10 hover:bg-red-500/20 text-red-500 font-bold py-3 px-4 rounded-lg transition-colors duration-300"
                                >
                                    <XIcon className="w-5 h-5" />
                                    <span>رد کردن</span>
                                </button>
                                <button
                                    onClick={handleApprove}
                                    className="w-full flex items-center justify-center gap-2 bg-green-500/10 hover:bg-green-500/20 text-green-500 font-bold py-3 px-4 rounded-lg transition-colors duration-300"
                                >
                                    <CheckIcon className="w-5 h-5" />
                                    <span>تایید</span>
                                </button>
                            </div>
                        )}
                    </div>
                </div>
            </div>

            {/* مودال رد درخواست */}
            {showRejectModal && (
                <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
                    <div className="bg-white dark:bg-custom-dark-blue-light rounded-xl p-6 max-w-md w-full">
                        <h3 className="text-xl font-bold mb-4 text-gray-800 dark:text-white">رد درخواست</h3>
                        <p className="text-gray-600 dark:text-gray-300 mb-4">لطفاً دلیل رد کردن این درخواست را وارد کنید:</p>
                        <textarea
                            value={rejectReason}
                            onChange={(e) => setRejectReason(e.target.value)}
                            className="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-custom-dark-blue text-gray-800 dark:text-white focus:outline-none focus:ring-2 focus:ring-custom-purple"
                            rows={4}
                            placeholder="دلیل رد درخواست..."
                        />
                        <div className="flex justify-end gap-3 mt-4">
                            <button
                                onClick={() => {
                                    setShowRejectModal(false);
                                    setRejectReason('');
                                }}
                                className="px-4 py-2 bg-gray-200 dark:bg-gray-700 text-gray-800 dark:text-white rounded-lg hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors"
                            >
                                انصراف
                            </button>
                            <button
                                onClick={handleReject}
                                className="px-4 py-2 bg-red-500 text-white rounded-lg hover:bg-red-600 transition-colors"
                            >
                                رد درخواست
                            </button>
                        </div>
                    </div>
                </div>
            )}
        </div>
    );
};

const UserInfoCard: React.FC<{
    user: {
        id: number;
        name: string;
        avatar: string;
        rating: number;
        email?: string;
        phone?: string;
        show_phone: boolean;
    }
}> = ({ user }) => (
    <div className="text-center">
        <img src={user.avatar} alt={user.name} className="w-24 h-24 rounded-full mx-auto border-4 border-custom-purple/20" />
        <h3 className="text-xl font-bold mt-3">{user.name}</h3>
        <div className="flex items-center justify-center text-amber-500 mt-1">
            <StarIcon className="w-5 h-5" fill="currentColor" />
            <span className="mr-1 font-bold text-lg">{user.rating}</span>
        </div>
        {user.email && (
            <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">{user.email}</p>
        )}
        {user.phone && user.show_phone && (
            <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">{user.phone}</p>
        )}
    </div>
);

const InfoRow: React.FC<{ icon: React.FC<{ className?: string }>, label: string, value: string, valueClassName?: string }> = ({ icon: Icon, label, value, valueClassName = 'text-gray-800 dark:text-gray-200' }) => (
    <div className="flex items-center justify-between">
        <div className="flex items-center text-gray-500 dark:text-gray-400">
            <Icon className="w-5 h-5 ml-2" />
            <span className="text-sm">{label}</span>
        </div>
        <span className={`font-semibold text-sm ${valueClassName}`}>{value}</span>
    </div>
);

export default ShipmentDetailPage;