import React, { useState, useEffect, useRef } from 'react';
import { ArrowRightIcon, ListIcon, SendIcon, TagIcon, UserIcon, HeadphonesIcon, ClockIcon, CheckCircleIcon, ChevronDownIcon } from '../Icons';
import { API_BASE_URL } from '@/src/config.js';
import { authFetch } from '../../App';

interface Message {
    id: number;
    message: string;
    is_admin: boolean;
    created_at: string;
    admin?: {
        name: string;
    };
}

interface Ticket {
    id: number;
    ticket_number: string;
    title: string;
    message?: string; // ✅ اضافه شد
    department: string;
    priority: string;
    status: 'open' | 'in_progress' | 'waiting_for_user' | 'resolved' | 'closed';
    user: {
        id: number;
        name: string;
    };
    created_at: string;
    updated_at: string;
    replies: Message[];
}

interface CustomSelectProps {
    options: { value: string; label: string }[];
    value: string;
    onChange: (value: string) => void;
}

const CustomSelect: React.FC<CustomSelectProps> = ({ options, value, onChange }) => {
    const [isOpen, setIsOpen] = useState(false);
    const dropdownRef = useRef<HTMLDivElement>(null);
    const selectedOption = options.find(opt => opt.value === value);

    useEffect(() => {
        const handleClickOutside = (event: MouseEvent) => {
            if (dropdownRef.current && !dropdownRef.current.contains(event.target as Node)) {
                setIsOpen(false);
            }
        };
        document.addEventListener("mousedown", handleClickOutside);
        return () => document.removeEventListener("mousedown", handleClickOutside);
    }, []);

    return (
        <div className="relative" ref={dropdownRef}>
            <button
                type="button"
                onClick={() => setIsOpen(!isOpen)}
                className="w-full flex items-center justify-between text-right pr-4 pl-10 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition"
            >
                <span className="text-gray-800 dark:text-gray-200">
                    {selectedOption ? selectedOption.label : 'انتخاب کنید'}
                </span>
                <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                    <ChevronDownIcon className={`w-5 h-5 text-gray-400 transition-transform duration-200 ${isOpen ? 'rotate-180' : ''}`} />
                </div>
            </button>
            {isOpen && (
                <ul className="absolute z-20 w-full mt-1 bg-custom-light-card dark:bg-custom-dark-blue-light border border-gray-300 dark:border-gray-600 rounded-lg shadow-lg max-h-60 overflow-y-auto">
                    {options.map(option => (
                        <li
                            key={option.value}
                            onClick={() => {
                                onChange(option.value);
                                setIsOpen(false);
                            }}
                            className="px-4 py-2 cursor-pointer hover:bg-gray-100 dark:hover:bg-custom-dark-blue text-gray-800 dark:text-gray-200 transition-colors"
                        >
                            {option.label}
                        </li>
                    ))}
                </ul>
            )}
        </div>
    );
};

interface AdminTicketDetailPageProps {
    ticketId: string | null;
    onNavigate: (page: string) => void;
}

const AdminTicketDetailPage: React.FC<AdminTicketDetailPageProps> = ({ ticketId, onNavigate }) => {
    const [ticket, setTicket] = useState<Ticket | null>(null);
    const [allMessages, setAllMessages] = useState<Message[]>([]); // ✅✅ اضافه شد
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState<string | null>(null);
    const [currentStatus, setCurrentStatus] = useState('');
    const [replyText, setReplyText] = useState('');
    const [submitting, setSubmitting] = useState(false);

    // ✅ تابع fetch
    const fetchTicketDetail = async () => {
        try {
            setLoading(true);
            setError(null);

            const response = await authFetch(`${API_BASE_URL}/admin/tickets/${ticketId}`);
            const result = await response.json();

            if (result.success && result.data) {
                setTicket(result.data);
                setCurrentStatus(result.data.status);
            } else {
                setError('خطا در دریافت اطلاعات تیکت');
            }
        } catch (err: any) {
            setError(err.message || 'خطا در ارتباط با سرور');
        } finally {
            setLoading(false);
        }
    };

    // ✅ اولین بار که صفحه لود میشه
    useEffect(() => {
        if (!ticketId) {
            setError('شناسه تیکت نامعتبر است');
            setLoading(false);
            return;
        }
        fetchTicketDetail();
    }, [ticketId]);

    // ✅ هروقت ticket تغییر کرد، پیام‌ها رو ترکیب کن
    useEffect(() => {
        if (ticket) {
            const messages: Message[] = [];

            // اضافه کردن پیام اولیه
            if (ticket.message) {
                messages.push({
                    id: 0,
                    message: ticket.message,
                    is_admin: false,
                    created_at: ticket.created_at,
                });
            }

            // اضافه کردن بقیه پاسخ‌ها
            if (ticket.replies && ticket.replies.length > 0) {
                messages.push(...ticket.replies);
            }

            setAllMessages(messages);
        }
    }, [ticket]);

    // ✅ ارسال پاسخ
    const handleReplySubmit = async (e: React.FormEvent) => {
        e.preventDefault();

        if (!replyText.trim()) {
            return;
        }

        try {
            setSubmitting(true);

            const response = await authFetch(`${API_BASE_URL}/admin/tickets/${ticketId}/reply`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ message: replyText })
            });

            const result = await response.json();

            if (result.success) {
                setReplyText('');
                await fetchTicketDetail(); // ✅ رفرش کردن تیکت
            }
        } catch (err: any) {
            alert(err.message || 'خطا در ارسال پاسخ');
        } finally {
            setSubmitting(false);
        }
    };

    // ✅ تغییر وضعیت
    const handleStatusChange = async (newStatus: string) => {
        try {
            const response = await authFetch(`${API_BASE_URL}/admin/tickets/${ticketId}/status`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({ status: newStatus })
            });

            const result = await response.json();

            if (result.success) {
                setCurrentStatus(newStatus);
                await fetchTicketDetail(); // ✅ رفرش کردن تیکت
            }
        } catch (err: any) {
            alert(err.message || 'خطا در تغییر وضعیت');
        }
    };

    const getStatusChip = (status: string) => {
        const statusMap: Record<string, { label: string; class: string }> = {
            open: { label: 'باز', class: 'bg-blue-500/10 text-blue-500' },
            in_progress: { label: 'در حال بررسی', class: 'bg-custom-orange/10 text-custom-orange' },
            waiting_for_user: { label: 'در انتظار پاسخ کاربر', class: 'bg-yellow-500/10 text-yellow-500' },
            resolved: { label: 'حل شده', class: 'bg-green-500/10 text-green-500' },
            closed: { label: 'بسته شده', class: 'bg-gray-500/10 text-gray-500' },
        };
        const statusInfo = statusMap[status] || statusMap.open;
        return <span className={`px-3 py-1 text-xs font-bold rounded-full ${statusInfo.class}`}>{statusInfo.label}</span>;
    };

    const getPriorityChip = (priority: string) => {
        const priorityMap: Record<string, { label: string; class: string }> = {
            low: { label: 'کم', class: 'bg-gray-500/10 text-gray-500' },
            normal: { label: 'متوسط', class: 'bg-blue-500/10 text-blue-500' },
            high: { label: 'بالا', class: 'bg-orange-500/10 text-orange-500' },
            urgent: { label: 'فوری', class: 'bg-red-500/10 text-red-500' },
        };
        const priorityInfo = priorityMap[priority] || priorityMap.normal;
        return <span className={`px-3 py-1 text-xs font-bold rounded-full ${priorityInfo.class}`}>{priorityInfo.label}</span>;
    };

    const formatDate = (dateString: string) => {
        const date = new Date(dateString);
        const now = new Date();
        const diff = now.getTime() - date.getTime();
        const hours = Math.floor(diff / (1000 * 60 * 60));
        const days = Math.floor(diff / (1000 * 60 * 60 * 24));

        if (hours < 1) return 'چند دقیقه پیش';
        if (hours < 24) return `${hours} ساعت پیش`;
        if (days < 7) return `${days} روز پیش`;

        return new Intl.DateTimeFormat('fa-IR', {
            year: 'numeric',
            month: 'long',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        }).format(date);
    };

    if (loading) {
        return (
            <div className="flex items-center justify-center min-h-[400px]">
                <div className="text-center">
                    <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-custom-purple mx-auto mb-4"></div>
                    <p className="text-gray-600 dark:text-gray-400">در حال بارگذاری...</p>
                </div>
            </div>
        );
    }

    if (error || !ticket) {
        return (
            <div className="text-center p-8">
                <h2 className="text-xl font-bold text-red-500 mb-4">{error || 'تیکت یافت نشد'}</h2>
                <button
                    onClick={() => onNavigate('admin-tickets')}
                    className="mt-4 px-6 py-2 bg-custom-purple hover:bg-custom-light-purple text-white rounded-lg"
                >
                    بازگشت به لیست تیکت‌ها
                </button>
            </div>
        );
    }

    const statusOptions = [
        { value: 'open', label: 'باز' },
        { value: 'in_progress', label: 'در حال بررسی' },
        { value: 'waiting_for_user', label: 'در انتظار پاسخ کاربر' },
        { value: 'resolved', label: 'حل شده' },
        { value: 'closed', label: 'بستن تیکت' },
    ];

    return (
        <div className="flex flex-col space-y-6">
            <div className="flex items-center justify-between">
                <h1 className="text-2xl font-bold text-gray-800 dark:text-white">{ticket.title}</h1>
                <button
                    onClick={() => onNavigate('admin-tickets')}
                    className="flex items-center gap-2 py-2 px-4 rounded-lg text-gray-600 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-custom-dark-blue"
                >
                    <ArrowRightIcon className="w-5 h-5" />
                    <span>بازگشت</span>
                </button>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                <div className="lg:col-span-2 flex flex-col space-y-6">
                    {/* ✅✅ اینجا از allMessages استفاده میکنیم */}
                    <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm space-y-6">
                        {allMessages.length > 0 ? (
                            allMessages.map((msg, index) => (
                                <div key={msg.id || index} className={`flex items-start gap-4 ${msg.is_admin ? 'flex-row-reverse' : ''}`}>
                                    <div className={`flex-shrink-0 w-10 h-10 rounded-full flex items-center justify-center ${msg.is_admin ? 'bg-custom-cyan' : 'bg-custom-purple'}`}>
                                        {msg.is_admin ? (
                                            <HeadphonesIcon className="w-6 h-6 text-white" />
                                        ) : (
                                            <UserIcon className="w-6 h-6 text-white" />
                                        )}
                                    </div>
                                    <div className={`w-full max-w-xl p-4 rounded-xl ${msg.is_admin ? 'bg-custom-cyan/10 dark:bg-custom-cyan/20 rounded-tr-none' : 'bg-gray-100 dark:bg-custom-dark-blue rounded-tl-none'}`}>
                                        {msg.is_admin && msg.admin && (
                                            <p className="text-xs font-semibold text-custom-cyan mb-2">
                                                {msg.admin.name}
                                            </p>
                                        )}
                                        <p className="text-gray-800 dark:text-gray-200 whitespace-pre-wrap">{msg.message}</p>
                                        <p className="text-xs text-gray-400 dark:text-gray-500 mt-2 text-left">
                                            {formatDate(msg.created_at)}
                                        </p>
                                    </div>
                                </div>
                            ))
                        ) : (
                            <p className="text-center text-gray-500 dark:text-gray-400">هنوز پیامی ثبت نشده است</p>
                        )}
                    </div>

                    {currentStatus !== 'closed' && (
                        <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
                            <h3 className="text-lg font-semibold mb-4 text-gray-800 dark:text-gray-200">ارسال پاسخ</h3>
                            <form onSubmit={handleReplySubmit}>
                                <textarea
                                    value={replyText}
                                    onChange={(e) => setReplyText(e.target.value)}
                                    placeholder="پاسخ خود را اینجا بنویسید..."
                                    className="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple outline-none min-h-[120px] resize-none"
                                    disabled={submitting}
                                />
                                <div className="mt-4 flex items-center justify-between">
                                    <button
                                        type="submit"
                                        disabled={submitting || !replyText.trim()}
                                        className="bg-custom-purple hover:bg-custom-light-purple text-white font-bold py-2.5 px-6 rounded-lg flex items-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed"
                                    >
                                        <SendIcon className="w-5 h-5" />
                                        <span>{submitting ? 'در حال ارسال...' : 'ارسال'}</span>
                                    </button>
                                </div>
                            </form>
                        </div>
                    )}
                </div>

                <div className="lg:col-span-1">
                    <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm space-y-4 sticky top-28">
                        <h3 className="text-lg font-semibold mb-2 pb-3 border-b border-gray-200 dark:border-gray-700">اطلاعات تیکت</h3>
                        <InfoRow icon={UserIcon} label="کاربر" value={ticket.user.name} />
                        <InfoRow icon={ListIcon} label="شماره تیکت" value={ticket.ticket_number} />
                        <InfoRow icon={TagIcon} label="دپارتمان" value={ticket.department} />
                        <div className="flex items-center justify-between">
                            <div className="flex items-center gap-2 text-gray-500 dark:text-gray-400">
                                <TagIcon className="w-5 h-5" />
                                <span>اولویت</span>
                            </div>
                            {getPriorityChip(ticket.priority)}
                        </div>
                        <InfoRow icon={ClockIcon} label="آخرین بروزرسانی" value={formatDate(ticket.updated_at)} />
                        <div className="flex items-center justify-between">
                            <div className="flex items-center gap-2 text-gray-500 dark:text-gray-400">
                                <CheckCircleIcon className="w-5 h-5" />
                                <span>وضعیت</span>
                            </div>
                            {getStatusChip(currentStatus)}
                        </div>
                        {currentStatus !== 'closed' && (
                            <div className="pt-4 border-t border-gray-200 dark:border-gray-700">
                                <label className="block text-sm font-medium mb-2">تغییر وضعیت</label>
                                <CustomSelect
                                    options={statusOptions}
                                    value={currentStatus}
                                    onChange={handleStatusChange}
                                />
                            </div>
                        )}
                    </div>
                </div>
            </div>
        </div>
    );
};

const InfoRow: React.FC<{ icon: React.FC<{ className?: string }>, label: string, value: string }> = ({ icon: Icon, label, value }) => (
    <div className="flex items-center justify-between">
        <div className="flex items-center gap-2 text-gray-500 dark:text-gray-400"><Icon className="w-5 h-5" /><span>{label}</span></div>
        <span className="font-semibold text-gray-800 dark:text-gray-200">{value}</span>
    </div>
);

export default AdminTicketDetailPage;
