import React, { useState, useEffect, useRef } from 'react';
import { PlusCircleIcon, SearchIcon, ChevronDownIcon, UserIcon } from '../Icons';
import { API_BASE_URL } from '@/src/config.js';

interface User {
    id: number;
    name: string;
    email: string;
}

interface Ticket {
    id: number;
    ticket_number: string;
    title: string;
    message: string;
    department: string;
    priority: string;
    status: string;
    user: User;
    replies_count: number;
    unread_admin_replies?: number;
    created_at: string;
    updated_at: string;
}

interface PaginationMeta {
    current_page: number;
    last_page: number;
    per_page: number;
    total: number;
}

interface ApiResponse {
    success: boolean;
    data: Ticket[];  // ✅ مستقیم آرایه تیکت‌ها
    meta: PaginationMeta;  // ✅ meta هم در همین سطح
}

interface FilterDropdownProps {
    options: { value: string; label: string }[];
    value: string;
    onChange: (value: string) => void;
    placeholder: string;
}

const FilterDropdown: React.FC<FilterDropdownProps> = ({ options, value, onChange, placeholder }) => {
    const [isOpen, setIsOpen] = useState(false);
    const dropdownRef = useRef<HTMLDivElement>(null);
    const selectedOption = options.find(opt => opt.value === value);

    useEffect(() => {
        const handleClickOutside = (event: MouseEvent) => {
            if (dropdownRef.current && !dropdownRef.current.contains(event.target as Node)) {
                setIsOpen(false);
            }
        };
        document.addEventListener("mousedown", handleClickOutside);
        return () => document.removeEventListener("mousedown", handleClickOutside);
    }, []);

    return (
        <div className="relative" ref={dropdownRef}>
            <button
                type="button"
                onClick={() => setIsOpen(!isOpen)}
                className="w-full flex items-center justify-between text-right pr-4 pl-10 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition"
            >
                <span className={selectedOption ? "text-gray-800 dark:text-gray-200" : "text-gray-500"}>
                    {selectedOption ? selectedOption.label : placeholder}
                </span>
                <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                    <ChevronDownIcon className={`w-5 h-5 text-gray-400 transition-transform duration-200 ${isOpen ? 'rotate-180' : ''}`} />
                </div>
            </button>
            {isOpen && (
                <ul className="absolute z-20 w-full mt-1 bg-custom-light-card dark:bg-custom-dark-blue-light border border-gray-300 dark:border-gray-600 rounded-lg shadow-lg max-h-60 overflow-y-auto">
                    {options.map(option => (
                        <li
                            key={option.value}
                            onClick={() => {
                                onChange(option.value);
                                setIsOpen(false);
                            }}
                            className="px-4 py-2 cursor-pointer hover:bg-gray-100 dark:hover:bg-custom-dark-blue text-gray-800 dark:text-gray-200 transition-colors"
                        >
                            {option.label}
                        </li>
                    ))}
                </ul>
            )}
        </div>
    );
};

interface AdminTicketListPageProps {
    onNavigate: (page: string, payload?: any) => void;
}

const AdminTicketListPage: React.FC<AdminTicketListPageProps> = ({ onNavigate }) => {
    const [tickets, setTickets] = useState<Ticket[]>([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState<string | null>(null);

    const [searchQuery, setSearchQuery] = useState('');
    const [statusFilter, setStatusFilter] = useState('all');
    const [departmentFilter, setDepartmentFilter] = useState('all');
    const [priorityFilter, setPriorityFilter] = useState('all');

    const [currentPage, setCurrentPage] = useState(1);
    const [pagination, setPagination] = useState<PaginationMeta | null>(null);

    // تابع دریافت تیکت‌ها از API
    const fetchTickets = async () => {
        setLoading(true);
        setError(null);

        try {
            const token = localStorage.getItem('auth_token');
            if (!token) {
                throw new Error('توکن احراز هویت یافت نشد');
            }

            // ساخت query parameters
            const params = new URLSearchParams();

            if (statusFilter !== 'all') params.append('status', statusFilter);
            if (departmentFilter !== 'all') params.append('department', departmentFilter);
            if (priorityFilter !== 'all') params.append('priority', priorityFilter);
            if (searchQuery.trim()) params.append('search', searchQuery.trim());

            params.append('page', currentPage.toString());
            params.append('per_page', '15');
            params.append('sort_by', 'created_at');
            params.append('sort_order', 'desc');

            const response = await fetch(
                `${API_BASE_URL}/admin/tickets?${params.toString()}`,
                {
                    method: 'GET',
                    headers: {
                        'X-Auth-Token': token,
                        'Content-Type': 'application/json',
                        'Accept': 'application/json',
                    },
                }
            );

            if (!response.ok) {
                throw new Error('خطا در دریافت تیکت‌ها');
            }

            const result: ApiResponse = await response.json();

            if (result.success) {
                setTickets(result.data);  // ✅ نه result.data.data
                setPagination(result.meta);  // ✅ نه result.data.meta
            } else {
                throw new Error('پاسخ نامعتبر از سرور');
            }

        } catch (err: any) {
            setError(err.message || 'خطای ناشناخته');
            console.error('Error fetching tickets:', err);
        } finally {
            setLoading(false);
        }
    };

    // بارگذاری تیکت‌ها هنگام تغییر فیلترها
    useEffect(() => {
        fetchTickets();
    }, [searchQuery, statusFilter, departmentFilter, priorityFilter, currentPage]);

    // تبدیل وضعیت به فارسی
    const getStatusChip = (status: string) => {
        switch (status) {
            case 'open':
                return <span className="px-3 py-1 text-xs font-bold rounded-full bg-custom-orange/10 text-custom-orange">باز</span>;
            case 'in_progress':
                return <span className="px-3 py-1 text-xs font-bold rounded-full bg-blue-500/10 text-blue-500">در حال بررسی</span>;
            case 'resolved':
                return <span className="px-3 py-1 text-xs font-bold rounded-full bg-green-500/10 text-green-500">حل شده</span>;
            case 'closed':
                return <span className="px-3 py-1 text-xs font-bold rounded-full bg-gray-500/10 text-gray-500 dark:bg-gray-400/10 dark:text-gray-400">بسته شده</span>;
            default:
                return <span className="px-3 py-1 text-xs font-bold rounded-full bg-gray-500/10 text-gray-500">نامشخص</span>;
        }
    };

    // تبدیل دپارتمان به فارسی
    const getDepartmentLabel = (department: string) => {
        const departments: { [key: string]: string } = {
            'technical': 'پشتیبانی فنی',
            'financial': 'مالی',
            'suggestions': 'پیشنهادات',
        };
        return departments[department] || department;
    };

    // تبدیل اولویت به فارسی
    const getPriorityBadge = (priority: string) => {
        switch (priority) {
            case 'high':
                return <span className="text-red-500 font-bold">بالا</span>;
            case 'medium':
                return <span className="text-yellow-500 font-bold">متوسط</span>;
            case 'low':
                return <span className="text-green-500 font-bold">پایین</span>;
            default:
                return <span>{priority}</span>;
        }
    };

    // تبدیل تاریخ
    const formatDate = (dateString: string) => {
        const date = new Date(dateString);
        const now = new Date();
        const diffInMs = now.getTime() - date.getTime();
        const diffInHours = Math.floor(diffInMs / (1000 * 60 * 60));
        const diffInDays = Math.floor(diffInHours / 24);

        if (diffInHours < 1) return 'چند دقیقه پیش';
        if (diffInHours < 24) return `${diffInHours} ساعت پیش`;
        if (diffInDays === 1) return 'دیروز';
        if (diffInDays < 7) return `${diffInDays} روز پیش`;

        return new Intl.DateTimeFormat('fa-IR').format(date);
    };

    const statusOptions = [
        { value: 'all', label: 'همه وضعیت‌ها' },
        { value: 'open', label: 'باز' },
        { value: 'in_progress', label: 'در حال بررسی' },
        { value: 'resolved', label: 'حل شده' },
        { value: 'closed', label: 'بسته شده' },
    ];

    const departmentOptions = [
        { value: 'all', label: 'همه دپارتمان‌ها' },
        { value: 'technical', label: 'پشتیبانی فنی' },
        { value: 'financial', label: 'مالی' },
        { value: 'suggestions', label: 'پیشنهادات' },
    ];

    const priorityOptions = [
        { value: 'all', label: 'همه اولویت‌ها' },
        { value: 'high', label: 'بالا' },
        { value: 'medium', label: 'متوسط' },
        { value: 'low', label: 'پایین' },
    ];

    return (
        <div className="flex flex-col space-y-8">
            <div>
                <h1 className="text-3xl font-bold text-gray-800 dark:text-white">مدیریت تیکت‌ها</h1>
                <p className="text-gray-500 dark:text-gray-400 mt-1">تیکت‌های پشتیبانی کاربران را مدیریت کنید.</p>
            </div>

            <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
                <div className="grid grid-cols-1 sm:grid-cols-4 gap-4">
                    <div className="relative">
                        <input
                            type="text"
                            placeholder="جستجو در عنوان، شماره یا کاربر..."
                            className="w-full pr-10 pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition"
                            value={searchQuery}
                            onChange={(e) => {
                                setSearchQuery(e.target.value);
                                setCurrentPage(1);
                            }}
                        />
                        <SearchIcon className="absolute right-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
                    </div>
                    <FilterDropdown
                        options={statusOptions}
                        value={statusFilter}
                        onChange={(val) => {
                            setStatusFilter(val);
                            setCurrentPage(1);
                        }}
                        placeholder="وضعیت"
                    />
                    <FilterDropdown
                        options={departmentOptions}
                        value={departmentFilter}
                        onChange={(val) => {
                            setDepartmentFilter(val);
                            setCurrentPage(1);
                        }}
                        placeholder="دپارتمان"
                    />
                    <FilterDropdown
                        options={priorityOptions}
                        value={priorityFilter}
                        onChange={(val) => {
                            setPriorityFilter(val);
                            setCurrentPage(1);
                        }}
                        placeholder="اولویت"
                    />
                </div>
            </div>

            <div className="bg-custom-light-card dark:bg-custom-dark-blue-light rounded-2xl shadow-sm overflow-hidden">
                {loading ? (
                    <div className="text-center p-8">
                        <div className="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-custom-purple"></div>
                        <p className="mt-4 text-gray-500 dark:text-gray-400">در حال بارگذاری...</p>
                    </div>
                ) : error ? (
                    <div className="text-center p-8">
                        <p className="text-red-500">{error}</p>
                        <button
                            onClick={fetchTickets}
                            className="mt-4 px-4 py-2 bg-custom-purple text-white rounded-lg hover:bg-custom-purple/90"
                        >
                            تلاش مجدد
                        </button>
                    </div>
                ) : (
                    <>
                        <div className="overflow-x-auto">
                            <table className="w-full text-sm text-right">
                                <thead className="bg-custom-light-bg dark:bg-custom-dark-blue text-gray-500 dark:text-gray-400">
                                    <tr>
                                        <th className="p-4 font-semibold">شماره تیکت</th>
                                        <th className="p-4 font-semibold">عنوان</th>
                                        <th className="p-4 font-semibold">کاربر</th>
                                        <th className="p-4 font-semibold hidden lg:table-cell">دپارتمان</th>
                                        <th className="p-4 font-semibold hidden xl:table-cell">اولویت</th>
                                        <th className="p-4 font-semibold hidden sm:table-cell">پاسخ‌ها</th>
                                        <th className="p-4 font-semibold hidden md:table-cell">آخرین بروزرسانی</th>
                                        <th className="p-4 font-semibold">وضعیت</th>
                                        <th className="p-4 font-semibold"></th>
                                    </tr>
                                </thead>
                                <tbody className="divide-y divide-gray-200 dark:divide-gray-700">
                                    {tickets.length > 0 ? (
                                        tickets.map(ticket => (
                                            <tr key={ticket.id} className="hover:bg-gray-50 dark:hover:bg-custom-dark-blue">
                                                <td className="p-4 font-mono text-custom-purple font-semibold">
                                                    #{ticket.ticket_number}
                                                </td>
                                                <td className="p-4 font-semibold text-gray-800 dark:text-gray-200 max-w-xs truncate">
                                                    {ticket.title}
                                                </td>
                                                <td className="p-4">
                                                    <div className="flex items-center gap-2">
                                                        <div className="w-8 h-8 rounded-full bg-custom-purple/10 flex items-center justify-center">
                                                            <UserIcon className="w-4 h-4 text-custom-purple" />
                                                        </div>
                                                        <span className="text-sm">{ticket.user.name}</span>
                                                    </div>
                                                </td>
                                                <td className="p-4 hidden lg:table-cell text-gray-600 dark:text-gray-300">
                                                    {getDepartmentLabel(ticket.department)}
                                                </td>
                                                <td className="p-4 hidden xl:table-cell">
                                                    {getPriorityBadge(ticket.priority)}
                                                </td>
                                                <td className="p-4 hidden sm:table-cell">
                                                    <span className="px-2 py-1 bg-gray-100 dark:bg-gray-700 rounded-full text-xs font-semibold">
                                                        {ticket.replies_count}
                                                    </span>
                                                    {ticket.unread_admin_replies && ticket.unread_admin_replies > 0 && (
                                                        <span className="mr-1 px-2 py-1 bg-red-500 text-white rounded-full text-xs font-semibold">
                                                            {ticket.unread_admin_replies} جدید
                                                        </span>
                                                    )}
                                                </td>
                                                <td className="p-4 hidden md:table-cell text-gray-500 dark:text-gray-400 text-xs">
                                                    {formatDate(ticket.updated_at)}
                                                </td>
                                                <td className="p-4">{getStatusChip(ticket.status)}</td>
                                                <td className="p-4">
                                                    <button
                                                        onClick={() => onNavigate('admin-ticket-detail', ticket.id)}
                                                        className="text-custom-purple hover:underline font-semibold"
                                                    >
                                                        مشاهده
                                                    </button>
                                                </td>
                                            </tr>
                                        ))
                                    ) : (
                                        <tr>
                                            <td colSpan={9} className="text-center p-8 text-gray-500 dark:text-gray-400">
                                                هیچ تیکتی با این مشخصات یافت نشد.
                                            </td>
                                        </tr>
                                    )}
                                </tbody>
                            </table>
                        </div>

                        {pagination && pagination.last_page > 1 && (
                            <div className="flex items-center justify-between p-4 border-t border-gray-200 dark:border-gray-700">
                                <div className="text-sm text-gray-500 dark:text-gray-400">
                                    نمایش {((pagination.current_page - 1) * pagination.per_page) + 1} تا{' '}
                                    {Math.min(pagination.current_page * pagination.per_page, pagination.total)} از{' '}
                                    {pagination.total} تیکت
                                </div>
                                <div className="flex gap-2">
                                    <button
                                        onClick={() => setCurrentPage(prev => Math.max(1, prev - 1))}
                                        disabled={currentPage === 1}
                                        className="px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-custom-dark-blue disabled:opacity-50 disabled:cursor-not-allowed"
                                    >
                                        قبلی
                                    </button>
                                    <span className="px-4 py-2">
                                        صفحه {pagination.current_page} از {pagination.last_page}
                                    </span>
                                    <button
                                        onClick={() => setCurrentPage(prev => Math.min(pagination.last_page, prev + 1))}
                                        disabled={currentPage === pagination.last_page}
                                        className="px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-custom-dark-blue disabled:opacity-50 disabled:cursor-not-allowed"
                                    >
                                        بعدی
                                    </button>
                                </div>
                            </div>
                        )}
                    </>
                )}
            </div>
        </div>
    );
};

export default AdminTicketListPage;
