// src/components/pages/TripDetailPage.tsx
import React, { useEffect, useState } from 'react';
import { ArrowRightIcon, MapPinIcon, CalendarIcon, ScaleIcon, CheckIcon, XIcon, CarIcon } from '../Icons';
import { API_BASE_URL } from '@/src/config.js';
import moment from 'moment-jalaali';

interface TripData {
  id: number;
  type: 'trip' | 'companion';
  title: string;
  status: 'pending' | 'approved' | 'rejected';
  rejection_reason?: string;
  user: {
    name: string;
  };
  origin: {
    country: string;
    city: string;
    address?: string;
  };
  destination: {
    country: string;
    city: string;
    address?: string;
  };
  vehicle_type_label?: string;
  vehicle_types_label?: string;
  capacity?: number;
  departure_date_shamsi?: string | null;
  arrival_date_shamsi?: string | null;
  available_from_shamsi?: string | null;
  available_to_shamsi?: string | null;
  created_at: string;
  updated_at: string;
}

interface TripDetailPageProps {
  tripId: string | null;
  tripType: 'trip' | 'companion' | null;
  onNavigate: (page: string) => void;
  showError: (message: string) => void;
  showSuccess: (message: string) => void;
}

// تابع تبدیل تاریخ میلادی به شمسی
const toJalaali = (date: string): string => {
  return moment(date).format('jYYYY/jMM/jDD HH:mm');
};

// تابع ترجمه نوع وسیله نقلیه
const translateVehicleType = (vehicle: string): string => {
  const translations: Record<string, string> = {
    'private_car': 'خودروی شخصی',
    'train': 'قطار',
    'bus': 'اتوبوس',
    'airplane': 'هواپیما',
    'ship': 'کشتی',
    'truck': 'کامیون',
    'van': 'ون',
    'motorcycle': 'موتورسیکلت'
  };
  return translations[vehicle] || vehicle;
};

// تابع پردازش و ترجمه وسایل نقلیه
const parseAndTranslateVehicles = (vehicleLabel?: string): string => {
  if (!vehicleLabel) return 'نامشخص';

  // اگر رشته JSON آرایه است (برای همراهان)
  if (vehicleLabel.startsWith('[') && vehicleLabel.endsWith(']')) {
    try {
      const vehicles = JSON.parse(vehicleLabel);
      if (Array.isArray(vehicles)) {
        return vehicles.map(v => translateVehicleType(v)).join('، ');
      }
    } catch (e) {
      console.error('Error parsing vehicle types:', e);
    }
  }

  // برای سفرها که تک‌مورد هستند
  return translateVehicleType(vehicleLabel);
};

const TripDetailPage: React.FC<TripDetailPageProps> = ({
  tripId,
  tripType,
  onNavigate,
  showError,
  showSuccess
}) => {
  const [trip, setTrip] = useState<TripData | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [rejectReason, setRejectReason] = useState('');
  const [showRejectModal, setShowRejectModal] = useState(false);

  const handleApprove = async () => {
    if (!tripId || !tripType) return;
    
    try {
      const token = localStorage.getItem('auth_token');
      if (!token) {
        showError('لطفاً دوباره وارد شوید.');
        return;
      }

      const endpoint = tripType === 'trip' ? 'trips' : 'trip-companions';
      const response = await fetch(`${API_BASE_URL}/admin/${endpoint}/${tripId}/approve`, {
        method: 'PUT',
        headers: { 'X-Auth-Token': token, },
      });

      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.message || errorData.error || 'خطا در تأیید درخواست');
      }

      showSuccess('درخواست با موفقیت تأیید شد');
      setTrip(prev => prev ? { ...prev, status: 'approved' } : null);
    } catch (error: any) {
      showError(error.message || 'خطا در ارتباط با سرور');
    }
  };

  const handleReject = async () => {
    if (!tripId || !tripType) {
      showError('اطلاعات درخواست نامعتبر است');
      return;
    }
    
    if (!rejectReason.trim()) {
      showError('لطفاً دلیل رد درخواست را وارد کنید');
      return;
    }

    try {
      const token = localStorage.getItem('auth_token');
      if (!token) {
        showError('لطفاً دوباره وارد شوید.');
        return;
      }

      const endpoint = tripType === 'trip' ? 'trips' : 'trip-companions';
      const response = await fetch(`${API_BASE_URL}/admin/${endpoint}/${tripId}/reject`, {
        method: 'PUT',
        headers: {
          'X-Auth-Token': token,
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ reason: rejectReason })
      });

      if (!response.ok) {
        const errorData = await response.json().catch(() => ({}));
        throw new Error(errorData.message || errorData.error || 'خطا در رد درخواست');
      }

      showSuccess('درخواست با موفقیت رد شد');
      setTrip(prev => prev ? { ...prev, status: 'rejected', rejection_reason: rejectReason } : null);
      setShowRejectModal(false);
      setRejectReason('');
    } catch (error: any) {
      showError(error.message || 'خطا در ارتباط با سرور');
    }
  };

  useEffect(() => {
    if (!tripId || !tripType) {
      setError('اطلاعات درخواست ناقص است');
      setLoading(false);
      return;
    }

    const fetchTrip = async () => {
      try {
        const token = localStorage.getItem('auth_token');
        if (!token) {
          setError('لطفاً دوباره وارد شوید');
          return;
        }

        const endpoint = tripType === 'trip' ? 'trips' : 'trip-companions';
        const response = await fetch(`${API_BASE_URL}/admin/${endpoint}/${tripId}`, {
          headers: { 'X-Auth-Token': token, }
        });

        if (!response.ok) {
          throw new Error('درخواست یافت نشد');
        }

        const result = await response.json();
        setTrip(result.data);
      } catch (err: any) {
        setError(err.message || 'خطا در دریافت جزئیات');
      } finally {
        setLoading(false);
      }
    };

    fetchTrip();
  }, [tripId, tripType]);

  if (loading) {
    return (
      <div className="flex items-center justify-center min-h-[400px]">
        <div className="animate-spin rounded-full h-12 w-12 border-4 border-custom-purple border-t-transparent"></div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="text-center p-8">
        <h2 className="text-xl font-bold text-red-500">{error}</h2>
        <button onClick={() => onNavigate('admin-trips')} className="mt-4 text-custom-purple hover:underline">
          بازگشت به لیست
        </button>
      </div>
    );
  }

  if (!trip) {
    return (
      <div className="text-center p-8">
        <h2 className="text-xl font-bold">درخواست یافت نشد</h2>
        <button onClick={() => onNavigate('admin-trips')} className="mt-4 text-custom-purple hover:underline">
          بازگشت به لیست
        </button>
      </div>
    );
  }

  const getStatusBadge = () => {
    switch (trip.status) {
      case 'approved':
        return <span className="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-green-100 text-green-800">تأیید شده</span>;
      case 'rejected':
        return <span className="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-red-100 text-red-800">رد شده</span>;
      default:
        return <span className="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">در انتظار بررسی</span>;
    }
  };

  const isTrip = trip.type === 'trip';
  const vehicleLabel = parseAndTranslateVehicles(
    isTrip ? trip.vehicle_type_label : trip.vehicle_types_label
  );
  const dateInfo = isTrip 
    ? trip.departure_date_shamsi || 'نامشخص'
    : trip.available_from_shamsi || 'نامشخص';

  return (
    <div className="flex flex-col space-y-6">
      <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
        <div>
          <h1 className="text-3xl font-bold text-gray-800 dark:text-white">{trip.title}</h1>
          <div className="flex items-center gap-2 mt-2">
            <span className="text-sm text-gray-500 dark:text-gray-400">
              {isTrip ? 'سفر با خودرو/هواپیما' : 'درخواست همسفری'}
            </span>
            {getStatusBadge()}
          </div>
        </div>
        <button
          onClick={() => onNavigate('admin-trips')}
          className="flex items-center gap-2 py-2 px-4 rounded-lg text-gray-600 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-custom-dark-blue transition-colors self-start sm:self-center"
        >
          <ArrowRightIcon className="w-5 h-5" />
          <span>بازگشت به لیست</span>
        </button>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
        <div className="lg:col-span-2 space-y-8">
          <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl shadow-sm">
            <h3 className="text-xl font-semibold mb-4 border-b pb-3 border-gray-200 dark:border-gray-700">جزئیات درخواست</h3>
            <p className="text-gray-600 dark:text-gray-300">{trip.title}</p>

            {trip.status === 'rejected' && trip.rejection_reason && (
              <div className="mt-4 p-4 bg-red-50 dark:bg-red-900/20 rounded-lg">
                <h4 className="font-semibold text-red-800 dark:text-red-400 mb-2">دلیل رد درخواست:</h4>
                <p className="text-red-700 dark:text-red-300">{trip.rejection_reason}</p>
              </div>
            )}
          </div>

          <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl shadow-sm">
            <h3 className="text-xl font-semibold mb-4 border-b pb-3 border-gray-200 dark:border-gray-700">مسیر سفر</h3>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="flex items-start gap-3">
                <MapPinIcon className="w-5 h-5 text-green-500 mt-1 flex-shrink-0" />
                <div>
                  <h4 className="font-semibold text-gray-800 dark:text-gray-200">مبدأ</h4>
                  <p className="text-gray-600 dark:text-gray-400">{trip.origin.city}, {trip.origin.country}</p>
                  {trip.origin.address && (
                    <p className="text-sm text-gray-500 dark:text-gray-500">{trip.origin.address}</p>
                  )}
                </div>
              </div>
              <div className="flex items-start gap-3">
                <MapPinIcon className="w-5 h-5 text-red-500 mt-1 flex-shrink-0" />
                <div>
                  <h4 className="font-semibold text-gray-800 dark:text-gray-200">مقصد</h4>
                  <p className="text-gray-600 dark:text-gray-400">{trip.destination.city}, {trip.destination.country}</p>
                  {trip.destination.address && (
                    <p className="text-sm text-gray-500 dark:text-gray-500">{trip.destination.address}</p>
                  )}
                </div>
              </div>
            </div>
          </div>

          {(vehicleLabel || trip.capacity != null || 
            (isTrip ? trip.arrival_date_shamsi : trip.available_to_shamsi)) && (
            <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl shadow-sm">
              <h3 className="text-xl font-semibold mb-4 border-b pb-3 border-gray-200 dark:border-gray-700">
                {isTrip ? 'جزئیات سفر' : 'جزئیات همسفری'}
              </h3>
              <div className="space-y-3">
                {vehicleLabel && (
                  <InfoRow icon={CarIcon} label="وسیله نقلیه" value={vehicleLabel} />
                )}
                <InfoRow 
                  icon={CalendarIcon} 
                  label={isTrip ? "تاریخ حرکت" : "تاریخ در دسترس"} 
                  value={dateInfo} 
                />
                {!isTrip && trip.available_to_shamsi && (
                  <InfoRow 
                    icon={CalendarIcon} 
                    label="تاریخ پایان دسترسی" 
                    value={trip.available_to_shamsi} 
                  />
                )}
                {isTrip && trip.arrival_date_shamsi && (
                  <InfoRow 
                    icon={CalendarIcon} 
                    label="تاریخ رسیدن" 
                    value={trip.arrival_date_shamsi} 
                  />
                )}
                {trip.capacity != null && (
                  <InfoRow 
                    icon={ScaleIcon} 
                    label="ظرفیت حمل" 
                    value={`${trip.capacity} کیلوگرم`} 
                  />
                )}
              </div>
            </div>
          )}
        </div>

        <div className="lg:col-span-1 space-y-6">
          <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm space-y-4 sticky top-28">
            <div className="text-center">
              <div className="w-24 h-24 rounded-full bg-gray-300 dark:bg-gray-600 flex items-center justify-center text-white text-2xl mx-auto">
                {trip.user.name.charAt(0).toUpperCase()}
              </div>
              <h3 className="text-xl font-bold mt-3">{trip.user.name}</h3>
            </div>

            <div className="border-t border-gray-200 dark:border-gray-700 my-4" />
            <InfoRow icon={MapPinIcon} label="مبدأ" value={`${trip.origin.city}, ${trip.origin.country}`} />
            <InfoRow icon={MapPinIcon} label="مقصد" value={`${trip.destination.city}, ${trip.destination.country}`} />
            <InfoRow 
              icon={CalendarIcon} 
              label={isTrip ? "تاریخ حرکت" : "تاریخ در دسترس"} 
              value={dateInfo} 
            />
            {trip.capacity != null && (
              <InfoRow 
                icon={ScaleIcon} 
                label="ظرفیت" 
                value={`${trip.capacity} کیلوگرم`} 
              />
            )}

            <div className="border-t border-gray-200 dark:border-gray-700 my-4" />
            <div className="text-sm text-gray-500 dark:text-gray-400 space-y-1">
              <p>ایجاد شده: {toJalaali(trip.created_at)}</p>
              <p>آخرین به‌روزرسانی: {toJalaali(trip.updated_at)}</p>
            </div>

            {trip.status === 'pending' && (
              <div className="grid grid-cols-2 gap-3 pt-4">
                <button
                  onClick={() => setShowRejectModal(true)}
                  className="w-full flex items-center justify-center gap-2 bg-red-500/10 hover:bg-red-500/20 text-red-500 font-bold py-3 px-4 rounded-lg transition-colors duration-300"
                >
                  <XIcon className="w-5 h-5" />
                  <span>رد کردن</span>
                </button>
                <button
                  onClick={handleApprove}
                  className="w-full flex items-center justify-center gap-2 bg-green-500/10 hover:bg-green-500/20 text-green-500 font-bold py-3 px-4 rounded-lg transition-colors duration-300"
                >
                  <CheckIcon className="w-5 h-5" />
                  <span>تایید</span>
                </button>
              </div>
            )}
          </div>
        </div>
      </div>

      {showRejectModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white dark:bg-custom-dark-blue-light rounded-xl p-6 max-w-md w-full">
            <h3 className="text-xl font-bold mb-4 text-gray-800 dark:text-white">رد درخواست</h3>
            <p className="text-gray-600 dark:text-gray-300 mb-4">لطفاً دلیل رد کردن این درخواست را وارد کنید:</p>
            <textarea
              value={rejectReason}
              onChange={(e) => setRejectReason(e.target.value)}
              className="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-custom-dark-blue text-gray-800 dark:text-white focus:outline-none focus:ring-2 focus:ring-custom-purple"
              rows={4}
              placeholder="دلیل رد درخواست..."
            />
            <div className="flex justify-end gap-3 mt-4">
              <button
                onClick={() => {
                  setShowRejectModal(false);
                  setRejectReason('');
                }}
                className="px-4 py-2 bg-gray-200 dark:bg-gray-700 text-gray-800 dark:text-white rounded-lg hover:bg-gray-300 dark:hover:bg-gray-600 transition-colors"
              >
                انصراف
              </button>
              <button
                onClick={handleReject}
                className="px-4 py-2 bg-red-500 text-white rounded-lg hover:bg-red-600 transition-colors"
              >
                رد درخواست
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

const InfoRow: React.FC<{icon: React.FC<{className?: string}>, label: string, value: string}> = ({ icon: Icon, label, value }) => (
  <div className="flex items-center justify-between">
    <div className="flex items-center text-gray-500 dark:text-gray-400">
      <Icon className="w-5 h-5 ml-2" />
      <span className="text-sm">{label}</span>
    </div>
    <span className="font-semibold text-sm text-gray-800 dark:text-gray-200">{value}</span>
  </div>
);

export default TripDetailPage;
