import React, { useState } from 'react';
import { SunIcon, MoonIcon, SuitcaseIcon } from '../Icons';
import AnimatedBackground from './AnimatedBackground';

interface LoginPageProps {
    onLogin: (phone: string) => void;
    theme: string;
    toggleTheme: () => void;
    showError: (message: string) => void;
}

const LoginPage: React.FC<LoginPageProps> = ({ onLogin, theme, toggleTheme, showError }) => {
    const [phoneNumber, setPhoneNumber] = useState('');
    const [localError, setLocalError] = useState<string | null>(null);

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        const cleanPhone = phoneNumber.trim();
        if (cleanPhone.length === 11 && cleanPhone.startsWith('09')) {
            onLogin(cleanPhone);
            setLocalError(null);
        } else {
            const errorMsg = 'لطفاً یک شماره موبایل معتبر (11 رقمی با 09) وارد کنید.';
            setLocalError(errorMsg);
            showError(errorMsg);
        }
    };

    const isDark = theme === 'dark';

    return (
        <div className="relative min-h-screen">
            {/* نمایش خطا در بالای صفحه */}
            {localError && (
                <div className="fixed top-4 left-1/2 transform -translate-x-1/2 z-50 bg-red-500 text-white px-6 py-3 rounded-lg shadow-lg flex items-center animate-fade-in">
                    <span>{localError}</span>
                    <button
                        onClick={() => setLocalError(null)}
                        className="mr-3 text-white hover:text-gray-200"
                        aria-label="بستن"
                    >
                        ✕
                    </button>
                </div>
            )}

            <div className={`relative flex items-center justify-center min-h-screen font-sans overflow-hidden
                ${isDark 
                    ? 'bg-custom-dark-blue text-white bg-[radial-gradient(#27314e_1px,transparent_1px)]' 
                    : 'bg-gradient-to-br from-gray-100 via-purple-50 to-indigo-100 text-gray-800 bg-[radial-gradient(#c7d2fe_1px,transparent_1px)]'
                } [background-size:16px_16px]`}
            >
                <AnimatedBackground />
                
                {/* دکمه تغییر تم */}
                <div className="absolute top-6 left-6 z-20">
                    <button 
                        onClick={toggleTheme} 
                        className={`p-2 rounded-full cursor-pointer shadow-md transition-colors
                            ${isDark 
                                ? 'bg-custom-dark-blue-light' 
                                : 'bg-white border border-gray-200 hover:bg-gray-50'
                            }`}
                        aria-label="Toggle theme"
                    >
                        {isDark ? (
                            <SunIcon className="h-6 w-6 text-yellow-400" />
                        ) : (
                            <MoonIcon className="h-6 w-6 text-indigo-600" />
                        )}
                    </button>
                </div>

                {/* کارت اصلی */}
                <div className={`relative z-10 w-full max-w-sm p-8 space-y-8 rounded-2xl shadow-xl backdrop-blur-sm
                    ${isDark 
                        ? 'bg-custom-dark-blue-light/80 border border-white/10' 
                        : 'bg-white/90 border border-gray-200 shadow-purple-100/50'
                    }`}
                >
                    <div className="text-center">
                        <div className="flex items-center justify-center mb-6">
                            <SuitcaseIcon className={`w-12 h-12 ${isDark ? 'text-custom-purple' : 'text-indigo-600'}`} />
                            <span className={`text-4xl font-bold mr-3 ${isDark ? 'text-white' : 'text-gray-800'}`}>
                                اینوببر
                            </span>
                        </div>
                        <h2 className={`text-2xl font-bold ${isDark ? 'text-white' : 'text-gray-800'}`}>
                            ورود | ثبت نام
                        </h2>
                        <p className={`mt-2 ${isDark ? 'text-gray-400' : 'text-gray-500'}`}>
                            لطفا شماره موبایل خود را وارد کنید
                        </p>
                    </div>

                    <form className="space-y-6" onSubmit={handleSubmit}>
                        <input
                            id="phone"
                            name="phone"
                            type="tel"
                            autoComplete="tel"
                            required
                            className={`w-full px-4 py-3 text-center tracking-[.2em] rounded-lg focus:outline-none focus:ring-2 transition-colors
                                ${isDark 
                                    ? 'bg-custom-dark-blue border border-gray-700 placeholder-gray-500 text-white focus:ring-custom-purple' 
                                    : 'bg-gray-50 border border-gray-300 placeholder-gray-400 text-gray-800 focus:ring-indigo-500 focus:border-indigo-500'
                                }`}
                            placeholder="۰۹۱۲*******"
                            value={phoneNumber}
                            onChange={(e) => setPhoneNumber(e.target.value)}
                        />
                        <button
                            type="submit"
                            className={`w-full py-3 px-4 rounded-lg font-semibold transition-colors duration-300 text-white
                                ${isDark 
                                    ? 'bg-custom-purple hover:bg-custom-light-purple' 
                                    : 'bg-indigo-600 hover:bg-indigo-700 shadow-lg shadow-indigo-200'
                                }`}
                        >
                            ورود
                        </button>
                    </form>

                    <p className={`text-xs text-center ${isDark ? 'text-gray-500' : 'text-gray-400'}`}>
                        ورود شما به معنای پذیرش قوانین سایت است
                    </p>
                </div>
            </div>
        </div>
    );
};

export default LoginPage;
