import React, { useState, useCallback } from 'react';
import { TypeIcon, TagIcon, ListIcon, UploadCloudIcon, XIcon, ArrowRightIcon, ChevronDownIcon, AlertCircleIcon, PaperclipIcon, SendIcon } from '../Icons';
import { API_BASE_URL } from '@/src/config.js';
interface NewTicketPageProps {
    onNavigate: (page: string) => void;
}

interface ImageFile {
  id: number;
  url: string;
  file: File;
}

const NewTicketPage: React.FC<NewTicketPageProps> = ({ onNavigate }) => {
    const [attachments, setAttachments] = useState<ImageFile[]>([]);
    const [loading, setLoading] = useState(false);
    const [error, setError] = useState<string | null>(null);
    const [formData, setFormData] = useState({
        title: '',
        department: 'technical',
        priority: 'medium',
        message: ''
    });

    const handleFileUpload = (event: React.ChangeEvent<HTMLInputElement>) => {
        const files = event.target.files;
        if (files) {
            // بررسی تعداد فایل‌ها
            if (attachments.length + files.length > 5) {
                setError('حداکثر ۵ فایل می‌توانید آپلود کنید');
                return;
            }

            const newFiles: ImageFile[] = Array.from(files).map((file: File) => ({
                id: Date.now() + Math.random(),
                url: URL.createObjectURL(file),
                file: file
            }));
            setAttachments(prev => [...prev, ...newFiles]);
            setError(null);
        }
    };
    
    const removeAttachment = (id: number) => {
        const fileToRemove = attachments.find(f => f.id === id);
        if (fileToRemove) {
            URL.revokeObjectURL(fileToRemove.url);
        }
        setAttachments(prev => prev.filter(img => img.id !== id));
    };

    const onDrop = useCallback((event: React.DragEvent<HTMLLabelElement>) => {
        event.preventDefault();
        event.stopPropagation();
        const files = event.dataTransfer.files;
        if (files && files.length > 0) {
            // بررسی تعداد فایل‌ها
            if (attachments.length + files.length > 5) {
                setError('حداکثر ۵ فایل می‌توانید آپلود کنید');
                return;
            }

            const newFiles: ImageFile[] = Array.from(files).map((file: File) => ({
                id: Date.now() + Math.random(),
                url: URL.createObjectURL(file),
                file: file
            }));
            setAttachments(prev => [...prev, ...newFiles]);
            setError(null);
        }
    }, [attachments.length]);

    const onDragOver = (event: React.DragEvent<HTMLLabelElement>) => {
        event.preventDefault();
        event.stopPropagation();
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        setLoading(true);
        setError(null);

        // اعتبارسنجی فیلدهای اجباری
        if (!formData.title.trim() || !formData.message.trim()) {
            setError('عنوان و متن پیام الزامی هستند');
            setLoading(false);
            return;
        }

        try {
            const token = localStorage.getItem('auth_token');
            if (!token) {
                throw new Error('لطفاً ابتدا وارد شوید');
            }

            // ایجاد FormData برای ارسال فایل‌ها
            const submitData = new FormData();
            submitData.append('title', formData.title);
            submitData.append('department', formData.department);
            submitData.append('priority', formData.priority);
            submitData.append('message', formData.message);

            // اضافه کردن فایل‌های پیوست
            attachments.forEach(file => {
                submitData.append('attachments[]', file.file);
            });

            const response = await fetch(`${API_BASE_URL}/tickets`, {
                method: 'POST',
                headers: {
                    'X-Auth-Token': token,
                    // Note:不要设置 Content-Type，FormData 会自动设置正确的 boundary
                },
                body: submitData
            });

            const result = await response.json();

            if (!response.ok) {
                throw new Error(result.message || result.errors ? Object.values(result.errors).flat().join(', ') : 'خطا در ارسال تیکت');
            }

            // موفقیت آمیز بود - رفتن به صفحه لیست تیکت‌ها
            onNavigate('tickets');

        } catch (err: any) {
            setError(err.message || 'خطای ناشناخته رخ داد');
        } finally {
            setLoading(false);
        }
    };

    const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement | HTMLTextAreaElement>) => {
        const { name, value } = e.target;
        setFormData(prev => ({
            ...prev,
            [name]: value
        }));
    };

    return (
        <div className="flex flex-col space-y-8">
            {/* نمایش خطا */}
            {error && (
                <div className="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded-lg">
                    <strong>خطا:</strong> {error}
                </div>
            )}

            <div className="flex items-center justify-between">
                <div>
                    <h1 className="text-3xl font-bold text-gray-800 dark:text-white">ارسال تیکت جدید</h1>
                    <p className="text-gray-500 dark:text-gray-400 mt-1">مشکل یا پیشنهاد خود را با ما در میان بگذارید.</p>
                </div>
                <button 
                    onClick={() => onNavigate('tickets')} 
                    className="flex items-center gap-2 py-2 px-4 rounded-lg text-gray-600 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-custom-dark-blue transition-colors"
                >
                    <ArrowRightIcon className="w-5 h-5" />
                    <span>بازگشت به لیست</span>
                </button>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
                <div className="lg:col-span-2">
                     <form onSubmit={handleSubmit} className="space-y-8 bg-custom-light-card dark:bg-custom-dark-blue-light p-8 rounded-2xl shadow-sm">
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                           <InputWithIcon 
                                icon={TypeIcon} 
                                id="title" 
                                name="title"
                                placeholder="مثال: مشکل در پرداخت" 
                                label="عنوان تیکت" 
                                value={formData.title}
                                onChange={handleInputChange}
                                required
                            />
                           <SelectWithIcon 
                                icon={TagIcon} 
                                id="department" 
                                name="department"
                                label="دپارتمان"
                                value={formData.department}
                                onChange={handleInputChange}
                                required
                            >
                                <option value="technical">پشتیبانی فنی</option>
                                <option value="financial">مالی</option>
                                <option value="suggestions">پیشنهادات</option>
                           </SelectWithIcon>
                        </div>
                        <SelectWithIcon 
                            icon={AlertCircleIcon} 
                            id="priority" 
                            name="priority"
                            label="اولویت"
                            value={formData.priority}
                            onChange={handleInputChange}
                            required
                        >
                            <option value="low">کم</option>
                            <option value="medium">متوسط</option>
                            <option value="high">زیاد</option>
                        </SelectWithIcon>
                        <TextareaWithIcon 
                            icon={ListIcon} 
                            id="message" 
                            name="message"
                            placeholder="لطفا مشکل خود را با جزئیات کامل شرح دهید..." 
                            label="متن پیام" 
                            value={formData.message}
                            onChange={handleInputChange}
                            required
                        />
                        
                        <button 
                            type="submit" 
                            disabled={loading}
                            className="w-full bg-custom-purple hover:bg-custom-light-purple text-white font-bold py-3 px-4 rounded-lg transition-colors duration-300 shadow-lg hover:shadow-xl focus:outline-none focus:ring-4 focus:ring-custom-purple/50 flex items-center justify-center gap-2 disabled:bg-gray-400 disabled:cursor-not-allowed"
                        >
                           {loading ? (
                                <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white"></div>
                           ) : (
                                <>
                                    <SendIcon className="w-5 h-5" />
                                    <span>ارسال تیکت</span>
                                </>
                           )}
                        </button>
                    </form>
                </div>
                <div className="lg:col-span-1 space-y-8">
                     <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
                        <h3 className="text-lg font-semibold mb-4">پیوست فایل (اختیاری)</h3>
                        <label 
                            htmlFor="file-upload" 
                            className="flex flex-col items-center justify-center w-full h-48 border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-lg cursor-pointer bg-custom-light-bg dark:bg-custom-dark-blue hover:bg-gray-100 dark:hover:bg-gray-800 transition-colors"
                            onDrop={onDrop}
                            onDragOver={onDragOver}
                        >
                            <div className="flex flex-col items-center justify-center pt-5 pb-6">
                                <UploadCloudIcon className="w-10 h-10 mb-3 text-gray-400" />
                                <p className="mb-2 text-sm text-gray-500 dark:text-gray-400">
                                    <span className="font-semibold">برای آپلود کلیک کنید</span> یا بکشید
                                </p>
                                <p className="text-xs text-gray-500 dark:text-gray-400">
                                    PNG, JPG, PDF, DOC, DOCX (حداکثر ۱۰MB)
                                </p>
                                <p className="text-xs text-gray-400 dark:text-gray-500 mt-1">
                                    حداکثر ۵ فایل
                                </p>
                            </div>
                            <input 
                                id="file-upload" 
                                type="file" 
                                className="hidden" 
                                multiple 
                                accept="image/*,application/pdf,application/msword,application/vnd.openxmlformats-officedocument.wordprocessingml.document" 
                                onChange={handleFileUpload} 
                            />
                        </label>
                    </div>

                    {attachments.length > 0 && (
                         <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
                            <h3 className="text-lg font-semibold mb-4">
                                فایل‌های پیوست شده ({attachments.length}/5)
                            </h3>
                             <div className="space-y-3">
                                {attachments.map(file => (
                                    <div key={file.id} className="flex items-center justify-between bg-custom-light-bg dark:bg-custom-dark-blue p-3 rounded-lg border border-gray-200 dark:border-gray-600">
                                        <div className="flex items-center gap-3 overflow-hidden flex-1">
                                            <PaperclipIcon className="w-4 h-4 text-gray-500 flex-shrink-0" />
                                            <div className="flex flex-col min-w-0 flex-1">
                                                <span className="text-sm font-medium text-gray-800 dark:text-gray-200 truncate">
                                                    {file.file.name}
                                                </span>
                                                <span className="text-xs text-gray-500 dark:text-gray-400">
                                                    {(file.file.size / (1024 * 1024)).toFixed(2)} MB
                                                </span>
                                            </div>
                                        </div>
                                        <button 
                                            type="button"
                                            onClick={() => removeAttachment(file.id)} 
                                            className="p-1 text-red-500 hover:bg-red-100 dark:hover:bg-red-900/50 rounded-full flex-shrink-0 ml-2"
                                        >
                                            <XIcon className="w-4 h-4" />
                                        </button>
                                    </div>
                                ))}
                            </div>
                        </div>
                    )}
                </div>
            </div>
        </div>
    );
};

// کامپوننت‌های کمکی با اضافه شدن propsهای جدید
interface InputWithIconProps {
    id: string; 
    name: string;
    icon: React.FC<{ className?: string }>; 
    label: string; 
    placeholder: string;
    value: string;
    onChange: (e: React.ChangeEvent<HTMLInputElement>) => void;
    required?: boolean;
}

const InputWithIcon: React.FC<InputWithIconProps> = ({ 
    id, 
    name, 
    icon: Icon, 
    label, 
    placeholder, 
    value, 
    onChange, 
    required 
}) => (
    <div>
        <label htmlFor={id} className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
            {label}
            {required && <span className="text-red-500 mr-1">*</span>}
        </label>
        <div className="relative">
            <div className="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none">
                <Icon className="w-5 h-5 text-gray-400" />
            </div>
            <input 
                type="text" 
                id={id} 
                name={name}
                className="w-full pr-10 pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition" 
                placeholder={placeholder} 
                value={value}
                onChange={onChange}
                required={required}
            />
        </div>
    </div>
);

interface SelectWithIconProps {
    id: string; 
    name: string;
    icon: React.FC<{ className?: string }>; 
    label: string; 
    children: React.ReactNode;
    value: string;
    onChange: (e: React.ChangeEvent<HTMLSelectElement>) => void;
    required?: boolean;
}

const SelectWithIcon: React.FC<SelectWithIconProps> = ({ 
    id, 
    name, 
    icon: Icon, 
    label, 
    children, 
    value, 
    onChange, 
    required 
}) => (
    <div>
        <label htmlFor={id} className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
            {label}
            {required && <span className="text-red-500 mr-1">*</span>}
        </label>
        <div className="relative">
            <div className="absolute inset-y-0 right-0 flex items-center pr-3 pointer-events-none">
                <Icon className="w-5 h-5 text-gray-400" />
            </div>
            <select 
                id={id} 
                name={name}
                className="w-full appearance-none pr-10 pl-10 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition"
                value={value}
                onChange={onChange}
                required={required}
            >
                {children}
            </select>
            <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                <ChevronDownIcon className="w-5 h-5 text-gray-400" />
            </div>
        </div>
    </div>
);

interface TextareaWithIconProps {
    id: string; 
    name: string;
    icon: React.FC<{ className?: string }>; 
    label: string; 
    placeholder: string;
    value: string;
    onChange: (e: React.ChangeEvent<HTMLTextAreaElement>) => void;
    required?: boolean;
}

const TextareaWithIcon: React.FC<TextareaWithIconProps> = ({ 
    id, 
    name, 
    icon: Icon, 
    label, 
    placeholder, 
    value, 
    onChange, 
    required 
}) => (
    <div>
        <label htmlFor={id} className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
            {label}
            {required && <span className="text-red-500 mr-1">*</span>}
        </label>
        <div className="relative">
            <div className="absolute top-3.5 right-0 flex items-center pr-3 pointer-events-none">
                <Icon className="w-5 h-5 text-gray-400" />
            </div>
            <textarea 
                id={id} 
                name={name}
                className="w-full pr-10 pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition min-h-[150px] resize-y" 
                placeholder={placeholder}
                value={value}
                onChange={onChange}
                required={required}
            />
        </div>
    </div>
);

export default NewTicketPage;