import React, { useState, useEffect, useRef } from 'react';
import { ArrowRightIcon, ListIcon, PaperclipIcon, SendIcon, TagIcon, UserIcon, HeadphonesIcon, ClockIcon, AlertCircleIcon, XIcon } from '../Icons';
import { authFetch } from '../../App';
import { API_BASE_URL } from '@/src/config.js';

interface Attachment {
    id: number;
    file_name: string;
    file_size: number;
    url: string;
    mime_type?: string;
}

interface Message {
    id: number;
    sender: 'user' | 'support';
    text: string;
    timestamp: string;
    created_at?: string;
    attachments?: Attachment[];
}

interface Ticket {
    id: string;
    subject: string;
    department: string;
    lastUpdate: string;
    status: 'answered' | 'pending' | 'closed';
    messages: Message[];
    priority?: string;
    created_at?: string;
}

interface TicketDetailPageProps {
    ticketId: string | null;
    onNavigate: (page: string) => void;
}

const TicketDetailPage: React.FC<TicketDetailPageProps> = ({ ticketId, onNavigate }) => {
    const [ticket, setTicket] = useState<Ticket | null>(null);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState<string | null>(null);
    const [replyText, setReplyText] = useState('');
    const [sending, setSending] = useState(false);

    // 📎 استیت‌های مربوط به فایل
    const [selectedFiles, setSelectedFiles] = useState<File[]>([]);
    const fileInputRef = useRef<HTMLInputElement>(null);

    // 🔄 تابع تبدیل تاریخ
    const formatDate = (dateString: string): string => {
        const date = new Date(dateString);
        const now = new Date();
        const diffMs = now.getTime() - date.getTime();
        const diffMins = Math.floor(diffMs / 60000);
        const diffHours = Math.floor(diffMs / 3600000);
        const diffDays = Math.floor(diffMs / 86400000);

        if (diffMins < 1) return 'الان';
        if (diffMins < 60) return `${diffMins} دقیقه پیش`;
        if (diffHours < 24) return `${diffHours} ساعت پیش`;
        if (diffDays < 7) return `${diffDays} روز پیش`;

        return new Intl.DateTimeFormat('fa-IR', {
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        }).format(date);
    };

    // 📎 تابع فرمت حجم فایل
    const formatFileSize = (bytes: number): string => {
        if (bytes === 0) return '0 بایت';
        const k = 1024;
        const sizes = ['بایت', 'کیلوبایت', 'مگابایت'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
    };

    // 🔄 تابع تبدیل داده‌های API
    const transformTicketData = (apiTicket: any): Ticket => {
        const departmentMap: Record<string, string> = {
            'technical': 'پشتیبانی فنی',
            'financial': 'مالی',
            'sales': 'فروش',
            'suggestions': 'پیشنهادات',
            'general': 'عمومی'
        };

        const statusMap: Record<string, 'answered' | 'pending' | 'closed'> = {
            'open': 'pending',
            'pending': 'pending',
            'in_progress': 'pending',
            'answered': 'answered',
            'closed': 'closed'
        };

        return {
            id: String(apiTicket.id),
            subject: apiTicket.subject || apiTicket.title,
            department: departmentMap[apiTicket.department] || apiTicket.department,
            lastUpdate: formatDate(apiTicket.updated_at || apiTicket.created_at),
            status: statusMap[apiTicket.status] || 'pending',
            priority: apiTicket.priority || 'normal',
            created_at: apiTicket.created_at,
            messages: (apiTicket.replies || []).map((reply: any) => ({
                id: reply.id,
                sender: reply.is_admin ? 'support' : 'user',
                text: reply.message || reply.content,
                timestamp: formatDate(reply.created_at),
                created_at: reply.created_at,
                attachments: reply.attachments || []
            }))
        };
    };

    // 🔄 تابع ارسال پاسخ
    const handleReplySubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        setSending(true);

        // بررسی حداقل متن یا فایل
        if (!replyText.trim() && selectedFiles.length === 0) {
            showError('لطفاً متن پاسخ را وارد کنید یا فایلی را پیوست کنید.');
            setSending(false);
            return;
        }

        // ایجاد FormData برای ارسال فایل و متن
        const formData = new FormData();

        // اضافه کردن متن پاسخ
        if (replyText.trim()) {
            formData.append('message', replyText.trim());
        }

        // اضافه کردن فایل‌های پیوست
        selectedFiles.forEach(file => {
            formData.append('attachments[]', file);
        });

        try {
            console.log('📤 Sending reply to ticket:', ticketId);

            // ⭐️ تغییر کلیدی: استفاده از fetch اصلی به جای authFetch
            const response = await fetch(`${API_BASE_URL}/tickets/${ticketId}/reply`, {
                method: 'POST',
                headers: {
                    // ⭐️ مهم: هدر Content-Type را تنظیم نکنید تا مرورگر آن را به صورت خودکار روی multipart/form-data تنظیم کند
                    'X-Auth-Token': localStorage.getItem('auth_token') || ''
                },
                body: formData // ارسال FormData به عنوان بدنه درخواست
            });

            const result = await response.json();
            console.log('✅ Reply API Response:', result);

            if (result.success) {
                // موفقیت‌آمیز بودن
                const newMessage: Message = {
                    id: Date.now(), // یک id موقت
                    sender: 'user',
                    text: replyText.trim(),
                    timestamp: 'الان',
                    created_at: new Date().toISOString(),
                    attachments: result.data.attachments || [] // اگر بک‌اند اطلاعات فایل‌های آپلود شده را برمی‌گرداند
                };

                setTicket(prev => prev ? {
                    ...prev,
                    messages: [...prev.messages, newMessage],
                    lastUpdate: formatDate(new Date().toISOString())
                } : null);

                // 2. فرم را خالی کن
                setReplyText('');
                setSelectedFiles([]);

                // 3. نمایش پیام موفقیت
                showSuccess('پاسخ شما با موفقیت ارسال شد.');

            } else {
                // خطا از سمت سرور
                throw new Error(result.message || 'خطا در ارسال پاسخ');
            }
        } catch (err: any) {
            console.error('❌ Error sending reply:', err);
            showError(err.message || 'خطا در ارسال پاسخ. لطفاً دوباره تلاش کنید.');
        } finally {
            setSending(false);
        }
    };

    // 📡 دریافت اطلاعات تیکت
    const fetchTicketData = async () => {
        try {
            setLoading(true);
            setError(null);

            console.log('🎫 Fetching ticket:', ticketId);
            const response = await authFetch(`${API_BASE_URL}/tickets/${ticketId}`);
            const result = await response.json();

            console.log('✅ API Response:', result);

            if (result.success && result.data) {
                const transformedTicket = transformTicketData(result.data);
                console.log('✅ Transformed ticket:', transformedTicket);
                setTicket(transformedTicket);
            } else {
                throw new Error(result.message || 'تیکت یافت نشد');
            }
        } catch (err: any) {
            console.error('❌ Error fetching ticket:', err);

            if (err.message === 'Token expired or invalid') {
                setError('session_expired');
            } else {
                setError(err.message || 'خطا در دریافت اطلاعات تیکت');
            }
        } finally {
            setLoading(false);
        }
    };

    // 📎 انتخاب فایل
    const handleFileSelect = (e: React.ChangeEvent<HTMLInputElement>) => {
        const files = Array.from(e.target.files || []);

        // بررسی تعداد فایل
        if (selectedFiles.length + files.length > 5) {
            alert('حداکثر 5 فایل می‌توانید انتخاب کنید');
            return;
        }

        // بررسی حجم فایل‌ها
        const invalidFiles = files.filter(file => file.size > 10 * 1024 * 1024);
        if (invalidFiles.length > 0) {
            alert('حجم هر فایل نباید بیشتر از 10 مگابایت باشد');
            return;
        }

        // بررسی فرمت فایل
        const validFormats = ['image/jpeg', 'image/jpg', 'image/png', 'application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'];
        const invalidFormats = files.filter(file => !validFormats.includes(file.type));
        if (invalidFormats.length > 0) {
            alert('فرمت فایل باید از نوع: jpg, jpeg, png, pdf, doc, docx باشد');
            return;
        }

        setSelectedFiles(prev => [...prev, ...files]);
    };

    // 📎 حذف فایل از لیست
    const handleRemoveFile = (index: number) => {
        setSelectedFiles(prev => prev.filter((_, i) => i !== index));
    };



    useEffect(() => {
        console.log('🎫 TicketDetailPage mounted');
        console.log('📋 ticketId:', ticketId);
        fetchTicketData();
    }, [ticketId]);

    const getStatusChip = (status: Ticket['status']) => {
        switch (status) {
            case 'answered':
                return <span className="px-3 py-1 text-xs font-bold rounded-full bg-green-500/10 text-green-500 border border-green-500/20">پاسخ داده شده</span>;
            case 'pending':
                return <span className="px-3 py-1 text-xs font-bold rounded-full bg-custom-orange/10 text-custom-orange border border-custom-orange/20">در حال بررسی</span>;
            case 'closed':
                return <span className="px-3 py-1 text-xs font-bold rounded-full bg-gray-500/10 text-gray-500 dark:bg-gray-400/10 dark:text-gray-400 border border-gray-500/20">بسته شده</span>;
        }
    };

    // 🔄 حالت بارگذاری
    if (loading) {
        return (
            <div className="flex flex-col space-y-6">
                <div className="flex items-center justify-between">
                    <div>
                        <div className="h-8 w-64 bg-gray-300 dark:bg-gray-600 rounded animate-pulse"></div>
                        <div className="h-4 w-48 bg-gray-300 dark:bg-gray-600 rounded mt-2 animate-pulse"></div>
                    </div>
                </div>
                <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm space-y-4">
                    {[...Array(3)].map((_, i) => (
                        <div key={i} className="flex gap-4">
                            <div className="w-10 h-10 bg-gray-300 dark:bg-gray-600 rounded-full animate-pulse"></div>
                            <div className="flex-1 space-y-2">
                                <div className="h-4 bg-gray-300 dark:bg-gray-600 rounded w-3/4 animate-pulse"></div>
                                <div className="h-4 bg-gray-300 dark:bg-gray-600 rounded w-1/2 animate-pulse"></div>
                            </div>
                        </div>
                    ))}
                </div>
            </div>
        );
    }

    // ❌ حالت خطا
    if (error) {
        return (
            <div className="flex flex-col space-y-6">
                <div>
                    <h1 className="text-3xl font-bold text-gray-800 dark:text-white">خطا</h1>
                </div>
                <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-2xl p-6">
                    <div className="flex items-center gap-4">
                        <AlertCircleIcon className="w-8 h-8 text-red-500 flex-shrink-0" />
                        <div>
                            <h3 className="font-semibold text-red-800 dark:text-red-300 mb-2">
                                {error === 'session_expired' ? 'زمان احراز هویت به پایان رسیده' : 'خطا در دریافت اطلاعات'}
                            </h3>
                            <p className="text-red-700 dark:text-red-400 mb-4">{error}</p>
                            <button
                                onClick={() => onNavigate('tickets')}
                                className="bg-red-500 hover:bg-red-600 text-white font-bold py-2 px-6 rounded-lg transition-colors duration-300"
                            >
                                بازگشت به لیست تیکت‌ها
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        );
    }

    // ❌ تیکت یافت نشد
    if (!ticket) {
        return (
            <div className="text-center p-8">
                <h2 className="text-xl font-bold">تیکت یافت نشد</h2>
                <button
                    onClick={() => onNavigate('tickets')}
                    className="mt-4 text-custom-purple hover:underline"
                >
                    بازگشت به لیست تیکت‌ها
                </button>
            </div>
        );
    }

    // ✅ نمایش تیکت
    return (
        <div className="flex flex-col space-y-6">
            <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
                <div>
                    <h1 className="text-2xl font-bold text-gray-800 dark:text-white flex items-center gap-3">
                        {ticket.subject}
                        {getStatusChip(ticket.status)}
                    </h1>
                    <p className="text-gray-500 dark:text-gray-400 mt-1">جزئیات و گفتگوی مربوط به تیکت #{ticket.id}</p>
                </div>
                <button
                    onClick={() => onNavigate('tickets')}
                    className="flex items-center gap-2 py-2 px-4 rounded-lg text-gray-600 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-custom-dark-blue transition-colors self-start sm:self-center"
                >
                    <ArrowRightIcon className="w-5 h-5" />
                    <span>بازگشت به لیست</span>
                </button>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                <div className="lg:col-span-2 flex flex-col space-y-6">
                    {/* تاریخچه گفتگو */}
                    <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm space-y-6">
                        {ticket.messages.length > 0 ? (
                            ticket.messages.map((msg) => (
                                <div
                                    key={msg.id}
                                    className={`flex items-start gap-4 ${msg.sender === 'user' ? 'flex-row-reverse' : ''}`}
                                >
                                    <div
                                        className={`flex-shrink-0 w-10 h-10 rounded-full flex items-center justify-center ${msg.sender === 'user' ? 'bg-custom-purple' : 'bg-custom-cyan'
                                            }`}
                                    >
                                        {msg.sender === 'user' ? (
                                            <UserIcon className="w-6 h-6 text-white" />
                                        ) : (
                                            <HeadphonesIcon className="w-6 h-6 text-white" />
                                        )}
                                    </div>
                                    <div className="flex-1 max-w-xl">
                                        <div
                                            className={`p-4 rounded-xl ${msg.sender === 'user'
                                                ? 'bg-custom-purple/10 dark:bg-custom-purple/20 rounded-tr-none'
                                                : 'bg-gray-100 dark:bg-custom-dark-blue rounded-tl-none'
                                                }`}
                                        >
                                            <p className="text-gray-800 dark:text-gray-200">{msg.text}</p>
                                            <p className="text-xs text-gray-400 dark:text-gray-500 mt-2 text-left">
                                                {msg.timestamp}
                                            </p>
                                        </div>

                                        {/* نمایش فایل‌های پیوست */}
                                        {msg.attachments && msg.attachments.length > 0 && (
                                            <div className="mt-2 space-y-2">
                                                {msg.attachments.map((attachment) => (
                                                    <a
                                                        key={attachment.id}
                                                        href={attachment.url}
                                                        target="_blank"
                                                        rel="noopener noreferrer"
                                                        className="flex items-center gap-2 p-2 bg-gray-50 dark:bg-custom-dark-blue rounded-lg hover:bg-gray-100 dark:hover:bg-custom-dark-blue/80 transition-colors"
                                                    >
                                                        <PaperclipIcon className="w-4 h-4 text-gray-500" />
                                                        <span className="text-sm text-gray-700 dark:text-gray-300 flex-1">
                                                            {attachment.file_name}
                                                        </span>
                                                        <span className="text-xs text-gray-500">
                                                            {formatFileSize(attachment.file_size)}
                                                        </span>
                                                    </a>
                                                ))}
                                            </div>
                                        )}
                                    </div>
                                </div>
                            ))
                        ) : (
                            <p className="text-center text-gray-500">هنوز پاسخی ثبت نشده است</p>
                        )}
                    </div>

                    {/* فرم پاسخ */}
                    {ticket.status !== 'closed' ? (
                        <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm">
                            <h3 className="text-lg font-semibold mb-4 text-gray-800 dark:text-gray-200">
                                ارسال پاسخ جدید
                            </h3>
                            <form onSubmit={handleReplySubmit} className="space-y-4">
                                <textarea
                                    value={replyText}
                                    onChange={(e) => setReplyText(e.target.value)}
                                    placeholder="پاسخ خود را اینجا بنویسید..."
                                    className="w-full pr-4 pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition min-h-[120px] resize-y"
                                    disabled={sending}
                                ></textarea>

                                {/* نمایش فایل‌های انتخاب شده */}
                                {selectedFiles.length > 0 && (
                                    <div className="space-y-2">
                                        <p className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                            فایل‌های انتخاب شده ({selectedFiles.length}/5):
                                        </p>
                                        {selectedFiles.map((file, index) => (
                                            <div
                                                key={index}
                                                className="flex items-center justify-between p-3 bg-gray-50 dark:bg-custom-dark-blue rounded-lg"
                                            >
                                                <div className="flex items-center gap-2 flex-1">
                                                    <PaperclipIcon className="w-4 h-4 text-gray-500" />
                                                    <span className="text-sm text-gray-700 dark:text-gray-300">
                                                        {file.name}
                                                    </span>
                                                    <span className="text-xs text-gray-500">
                                                        ({formatFileSize(file.size)})
                                                    </span>
                                                </div>
                                                <button
                                                    type="button"
                                                    onClick={() => handleRemoveFile(index)}
                                                    className="p-1 hover:bg-red-100 dark:hover:bg-red-900/20 rounded transition-colors"
                                                >
                                                    <XIcon className="w-4 h-4 text-red-500" />
                                                </button>
                                            </div>
                                        ))}
                                    </div>
                                )}

                                <div className="flex items-center justify-between">
                                    <div>
                                        <input
                                            ref={fileInputRef}
                                            type="file"
                                            multiple
                                            accept=".jpg,.jpeg,.png,.pdf,.doc,.docx"
                                            onChange={handleFileSelect}
                                            className="hidden"
                                            id="file-upload"
                                            disabled={sending || selectedFiles.length >= 5}
                                        />
                                        <label
                                            htmlFor="file-upload"
                                            className={`flex items-center gap-2 py-2 px-4 rounded-lg transition-colors cursor-pointer ${selectedFiles.length >= 5
                                                ? 'text-gray-400 dark:text-gray-600 cursor-not-allowed'
                                                : 'text-gray-600 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-custom-dark-blue'
                                                }`}
                                        >
                                            <PaperclipIcon className="w-5 h-5" />
                                            <span>پیوست فایل</span>
                                            {selectedFiles.length > 0 && (
                                                <span className="text-xs bg-custom-purple text-white px-2 py-0.5 rounded-full">
                                                    {selectedFiles.length}
                                                </span>
                                            )}
                                        </label>
                                    </div>
                                    <button
                                        type="submit"
                                        disabled={sending || !replyText.trim()}
                                        className="bg-custom-purple hover:bg-custom-light-purple text-white font-bold py-2.5 px-6 rounded-lg transition-colors duration-300 shadow-lg hover:shadow-xl focus:outline-none focus:ring-4 focus:ring-custom-purple/50 flex items-center justify-center gap-2 disabled:opacity-50 disabled:cursor-not-allowed"
                                    >
                                        <SendIcon className="w-5 h-5" />
                                        <span>{sending ? 'در حال ارسال...' : 'ارسال پاسخ'}</span>
                                    </button>
                                </div>
                            </form>
                        </div>
                    ) : (
                        <div className="bg-yellow-500/10 text-yellow-700 dark:text-yellow-400 p-4 rounded-xl text-center">
                            <p className="font-semibold">
                                این تیکت بسته شده است و امکان ارسال پاسخ جدید وجود ندارد.
                            </p>
                        </div>
                    )}
                </div>

                {/* کارت اطلاعات تیکت */}
                <div className="lg:col-span-1">
                    <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm space-y-4 sticky top-28">
                        <h3 className="text-lg font-semibold mb-2 pb-3 border-b border-gray-200 dark:border-gray-700">
                            اطلاعات تیکت
                        </h3>
                        <div className="flex items-center justify-between">
                            <div className="flex items-center gap-2 text-gray-500 dark:text-gray-400">
                                <ListIcon className="w-5 h-5" />
                                <span>شناسه تیکت</span>
                            </div>
                            <span className="font-mono font-semibold text-gray-800 dark:text-gray-200">
                                #{ticket.id}
                            </span>
                        </div>
                        <div className="flex items-center justify-between">
                            <div className="flex items-center gap-2 text-gray-500 dark:text-gray-400">
                                <TagIcon className="w-5 h-5" />
                                <span>دپارتمان</span>
                            </div>
                            <span className="font-semibold text-gray-800 dark:text-gray-200">
                                {ticket.department}
                            </span>
                        </div>
                        <div className="flex items-center justify-between">
                            <div className="flex items-center gap-2 text-gray-500 dark:text-gray-400">
                                <ClockIcon className="w-5 h-5" />
                                <span>آخرین بروزرسانی</span>
                            </div>
                            <span className="font-semibold text-gray-800 dark:text-gray-200">
                                {ticket.lastUpdate}
                            </span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );
};

export default TicketDetailPage;
