import React, { useState, useEffect } from 'react';
import { PlusCircleIcon, SearchIcon, ChevronDownIcon, AlertCircleIcon } from '../Icons';
import { authFetch } from '../../App';
import { API_BASE_URL } from '@/src/config.js';
interface Ticket {
    id: string;
    subject: string;
    department: string;
    lastUpdate: string;
    status: 'answered' | 'pending' | 'closed';
    priority?: string;
    created_at?: string;
    replies_count?: number;
}

interface TicketListPageProps {
    onNavigate: (page: string, payload?: any) => void;
}

// 🔄 تابع تبدیل داده‌های API به فرمت کلاینت
const transformTicketData = (apiTicket: any): Ticket => {
    const departmentMap: Record<string, string> = {
        'technical': 'پشتیبانی فنی',
        'financial': 'مالی',
        'sales': 'فروش',
        'suggestions': 'پیشنهادات',
        'general': 'عمومی'
    };

    // بک‌اند ما status های: open, pending, answered, closed ارسال می‌کند
    const statusMap: Record<string, 'answered' | 'pending' | 'closed'> = {
        'open': 'pending',
        'pending': 'pending',
        'in_progress': 'pending',
        'answered': 'answered',
        'closed': 'closed'
    };

    const formatDate = (dateString: string): string => {
        const date = new Date(dateString);
        const now = new Date();
        const diffMs = now.getTime() - date.getTime();
        const diffMins = Math.floor(diffMs / 60000);
        const diffHours = Math.floor(diffMs / 3600000);
        const diffDays = Math.floor(diffMs / 86400000);

        if (diffMins < 1) return 'الان';
        if (diffMins < 60) return `${diffMins} دقیقه پیش`;
        if (diffHours < 24) return `${diffHours} ساعت پیش`;
        if (diffDays < 7) return `${diffDays} روز پیش`;

        return new Intl.DateTimeFormat('fa-IR', {
            year: 'numeric',
            month: 'long',
            day: 'numeric'
        }).format(date);
    };

    return {
        id: String(apiTicket.id),
        subject: apiTicket.subject || apiTicket.title, // هر دو را چک می‌کنیم
        department: departmentMap[apiTicket.department] || apiTicket.department,
        lastUpdate: formatDate(apiTicket.updated_at || apiTicket.created_at),
        status: statusMap[apiTicket.status] || 'pending',
        priority: apiTicket.priority || 'normal',
        created_at: apiTicket.created_at,
        replies_count: apiTicket.replies_count || 0
    };
};

// 🌐 سرویس API
const apiService = {
    async request(endpoint: string, options: RequestInit = {}) {
        try {
            const response = await authFetch(`${API_BASE_URL}${endpoint}`, options);

            // بررسی Content-Type
            const contentType = response.headers.get('content-type');
            if (!contentType || !contentType.includes('application/json')) {
                console.error('Response is not JSON:', await response.text());
                throw new Error('پاسخ سرور به فرمت JSON نیست');
            }

            const data = await response.json();

            // بررسی ساختار پاسخ
            if (!data.success && data.message) {
                throw new Error(data.message);
            }

            return data;
        } catch (error) {
            console.error('API Error:', error);

            if (error instanceof Error) {
                if (error.message === 'Token expired or invalid' || error.message === 'SESSION_EXPIRED') {
                    throw new Error('SESSION_EXPIRED');
                }
                throw error;
            }

            throw new Error('خطای ناشناخته در ارتباط با سرور');
        }
    },

    async getTickets() {
        return this.request('/tickets');
    },
};

const TicketListPage: React.FC<TicketListPageProps> = ({ onNavigate }) => {
    const [searchQuery, setSearchQuery] = useState('');
    const [statusFilter, setStatusFilter] = useState('all');
    const [departmentFilter, setDepartmentFilter] = useState('all');
    const [tickets, setTickets] = useState<Ticket[]>([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState<string | null>(null);

    const fetchTickets = async () => {
        try {
            setLoading(true);
            setError(null);

            console.log('Fetching tickets from: /api/tickets');
            const result = await apiService.getTickets();
            console.log('API Response:', result);

            if (result.success && Array.isArray(result.data)) {
                const transformedTickets = result.data.map(transformTicketData);
                console.log('Transformed tickets:', transformedTickets);
                setTickets(transformedTickets);
            } else {
                throw new Error(result.message || 'فرمت پاسخ سرور نامعتبر است');
            }

        } catch (err: any) {
            console.error('Error fetching tickets:', err);

            if (err.message === 'SESSION_EXPIRED') {
                setError('session_expired');
            } else {
                //setError(err.message || 'خطا در دریافت تیکت‌ها');
            }
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchTickets();
    }, []);

    // فیلتر کردن تیکت‌ها
    const filteredTickets = tickets.filter(ticket => {
        const lowercasedQuery = searchQuery.toLowerCase();

        const matchesSearch = ticket.subject.toLowerCase().includes(lowercasedQuery) ||
            ticket.id.toLowerCase().includes(lowercasedQuery);
        const matchesStatus = statusFilter === 'all' || ticket.status === statusFilter;
        const matchesDepartment = departmentFilter === 'all' || ticket.department === departmentFilter;

        return matchesSearch && matchesStatus && matchesDepartment;
    });

    const getStatusChip = (status: Ticket['status']) => {
        switch (status) {
            case 'answered':
                return <span className="px-3 py-1 text-xs font-bold rounded-full bg-green-500/10 text-green-500 border border-green-500/20">پاسخ داده شده</span>;
            case 'pending':
                return <span className="px-3 py-1 text-xs font-bold rounded-full bg-custom-orange/10 text-custom-orange border border-custom-orange/20">در حال بررسی</span>;
            case 'closed':
                return <span className="px-3 py-1 text-xs font-bold rounded-full bg-gray-500/10 text-gray-500 dark:bg-gray-400/10 dark:text-gray-400 border border-gray-500/20">بسته شده</span>;
        }
    };

    const getPriorityBadge = (priority: string) => {
        switch (priority) {
            case 'high':
                return <span className="px-2 py-1 text-xs font-bold rounded-full bg-red-500/10 text-red-500 border border-red-500/20">بالا</span>;
            case 'normal':
                return <span className="px-2 py-1 text-xs font-bold rounded-full bg-blue-500/10 text-blue-500 border border-blue-500/20">متوسط</span>;
            case 'low':
                return <span className="px-2 py-1 text-xs font-bold rounded-full bg-gray-500/10 text-gray-500 border border-gray-500/20">پایین</span>;
            default:
                return <span className="px-2 py-1 text-xs font-bold rounded-full bg-blue-500/10 text-blue-500 border border-blue-500/20">متوسط</span>;
        }
    };

    const handleRetry = () => {
        fetchTickets();
    };

    const handleLoginRedirect = () => {
        localStorage.removeItem('auth_token');
        localStorage.removeItem('auth_phone');
        onNavigate('dashboard');
        window.location.reload();
    };

    // کامپوننت خطا
    const ErrorDisplay = () => {
        if (error === 'session_expired') {
            return (
                <div className="bg-amber-50 dark:bg-amber-900/20 border border-amber-200 dark:border-amber-800 rounded-2xl p-6">
                    <div className="flex items-center gap-4">
                        <AlertCircleIcon className="w-8 h-8 text-amber-500 flex-shrink-0" />
                        <div className="flex-grow">
                            <h3 className="font-semibold text-amber-800 dark:text-amber-300 mb-2">
                                زمان احراز هویت به پایان رسیده
                            </h3>
                            <p className="text-amber-700 dark:text-amber-400 mb-4">
                                برای ادامه کار، لطفاً مجدداً وارد حساب کاربری خود شوید.
                            </p>
                            <button
                                onClick={handleLoginRedirect}
                                className="bg-amber-500 hover:bg-amber-600 text-white font-bold py-2 px-6 rounded-lg transition-colors duration-300"
                            >
                                ورود مجدد
                            </button>
                        </div>
                    </div>
                </div>
            );
        }

        return (
            <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-2xl p-6">
                <div className="flex items-center gap-4">
                    <AlertCircleIcon className="w-8 h-8 text-red-500 flex-shrink-0" />
                    <div className="flex-grow">
                        <h3 className="font-semibold text-red-800 dark:text-red-300 mb-2">
                            خطا در دریافت اطلاعات
                        </h3>
                        <p className="text-red-700 dark:text-red-400 mb-4">
                            {error}
                        </p>
                        <button
                            onClick={handleRetry}
                            className="bg-red-500 hover:bg-red-600 text-white font-bold py-2 px-6 rounded-lg transition-colors duration-300"
                        >
                            تلاش مجدد
                        </button>
                    </div>
                </div>
            </div>
        );
    };

    if (loading) {
        return (
            <div className="flex flex-col space-y-8">
                <div>
                    <h1 className="text-3xl font-bold text-gray-800 dark:text-white">لیست تیکت‌ها</h1>
                    <p className="text-gray-500 dark:text-gray-400 mt-1">در حال دریافت اطلاعات...</p>
                </div>

                {/* اسکلت‌های بارگذاری */}
                <div className="space-y-4">
                    {[...Array(5)].map((_, index) => (
                        <div key={index} className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm animate-pulse">
                            <div className="flex items-center gap-4">
                                <div className="w-12 h-12 bg-gray-300 dark:bg-gray-600 rounded-lg"></div>
                                <div className="flex-grow space-y-2">
                                    <div className="h-4 bg-gray-300 dark:bg-gray-600 rounded w-3/4"></div>
                                    <div className="h-3 bg-gray-300 dark:bg-gray-600 rounded w-1/2"></div>
                                </div>
                                <div className="w-20 h-8 bg-gray-300 dark:bg-gray-600 rounded-full"></div>
                            </div>
                        </div>
                    ))}
                </div>
            </div>
        );
    }

    if (error) {
        return (
            <div className="flex flex-col space-y-8">
                <div>
                    <h1 className="text-3xl font-bold text-gray-800 dark:text-white">لیست تیکت‌ها</h1>
                    <p className="text-gray-500 dark:text-gray-400 mt-1">خطا در دریافت اطلاعات</p>
                </div>
                <ErrorDisplay />
            </div>
        );
    }

    return (
        <div className="flex flex-col space-y-8">
            <div>
                <h1 className="text-3xl font-bold text-gray-800 dark:text-white">لیست تیکت‌ها</h1>
                <p className="text-gray-500 dark:text-gray-400 mt-1">تیکت‌های پشتیبانی خود را مدیریت کنید.</p>
            </div>

            {/* بخش آمار */}
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-4 rounded-2xl shadow-sm border border-gray-200 dark:border-gray-700">
                    <div className="flex items-center justify-between">
                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">کل تیکت‌ها</p>
                            <p className="text-2xl font-bold text-gray-800 dark:text-white">{tickets.length}</p>
                        </div>
                        <div className="w-12 h-12 bg-blue-500/10 rounded-xl flex items-center justify-center">
                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={2} stroke="currentColor" className="w-6 h-6 text-blue-500">
                                <path strokeLinecap="round" strokeLinejoin="round" d="M9 12h3.75M9 15h3.75M9 18h3.75m3 .75H18a2.25 2.25 0 002.25-2.25V6.108c0-1.135-.845-2.098-1.976-2.192a48.424 48.424 0 00-1.123-.08m-5.801 0c-.065.21-.1.433-.1.664 0 .414.336.75.75.75h4.5a.75.75 0 00.75-.75 2.25 2.25 0 00-.1-.664m-5.8 0A2.251 2.251 0 0113.5 2.25H15c1.012 0 1.867.668 2.15 1.586m-5.8 0c-.376.023-.75.05-1.124.08C9.095 4.01 8.25 4.973 8.25 6.108V8.25m0 0H4.875c-.621 0-1.125.504-1.125 1.125v11.25c0 .621.504 1.125 1.125 1.125h9.75c.621 0 1.125-.504 1.125-1.125V9.375c0-.621-.504-1.125-1.125-1.125H8.25zM6.75 12h.008v.008H6.75V12zm0 3h.008v.008H6.75V15zm0 3h.008v.008H6.75V18z" />
                            </svg>
                        </div>
                    </div>
                </div>

                <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-4 rounded-2xl shadow-sm border border-gray-200 dark:border-gray-700">
                    <div className="flex items-center justify-between">
                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">در حال بررسی</p>
                            <p className="text-2xl font-bold text-gray-800 dark:text-white">
                                {tickets.filter(t => t.status === 'pending').length}
                            </p>
                        </div>
                        <div className="w-12 h-12 bg-custom-orange/10 rounded-xl flex items-center justify-center">
                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={2} stroke="currentColor" className="w-6 h-6 text-custom-orange">
                                <path strokeLinecap="round" strokeLinejoin="round" d="M12 6v6h4.5m4.5 0a9 9 0 11-18 0 9 9 0 0118 0z" />
                            </svg>
                        </div>
                    </div>
                </div>

                <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-4 rounded-2xl shadow-sm border border-gray-200 dark:border-gray-700">
                    <div className="flex items-center justify-between">
                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">پاسخ داده شده</p>
                            <p className="text-2xl font-bold text-gray-800 dark:text-white">
                                {tickets.filter(t => t.status === 'answered').length}
                            </p>
                        </div>
                        <div className="w-12 h-12 bg-green-500/10 rounded-xl flex items-center justify-center">
                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={2} stroke="currentColor" className="w-6 h-6 text-green-500">
                                <path strokeLinecap="round" strokeLinejoin="round" d="M9 12.75L11.25 15 15 9.75M21 12c0 1.268-.63 2.39-1.593 3.068a3.745 3.745 0 01-1.043 3.296 3.745 3.745 0 01-3.296 1.043A3.745 3.745 0 0112 21c-1.268 0-2.39-.63-3.068-1.593a3.746 3.746 0 01-3.296-1.043 3.745 3.745 0 01-1.043-3.296A3.745 3.745 0 013 12c0-1.268.63-2.39 1.593-3.068a3.745 3.745 0 011.043-3.296 3.746 3.746 0 013.296-1.043A3.746 3.746 0 0112 3c1.268 0 2.39.63 3.068 1.593a3.746 3.746 0 013.296 1.043 3.746 3.746 0 011.043 3.296A3.745 3.745 0 0121 12z" />
                            </svg>
                        </div>
                    </div>
                </div>

                <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-4 rounded-2xl shadow-sm border border-gray-200 dark:border-gray-700">
                    <div className="flex items-center justify-between">
                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">بسته شده</p>
                            <p className="text-2xl font-bold text-gray-800 dark:text-white">
                                {tickets.filter(t => t.status === 'closed').length}
                            </p>
                        </div>
                        <div className="w-12 h-12 bg-gray-500/10 rounded-xl flex items-center justify-center">
                            <svg xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" strokeWidth={2} stroke="currentColor" className="w-6 h-6 text-gray-500 dark:text-gray-400">
                                <path strokeLinecap="round" strokeLinejoin="round" d="M16.5 10.5V6.75a4.5 4.5 0 10-9 0v3.75m-.75 11.25h10.5a2.25 2.25 0 002.25-2.25v-6.75a2.25 2.25 0 00-2.25-2.25H6.75a2.25 2.25 0 00-2.25 2.25v6.75a2.25 2.25 0 002.25 2.25z" />
                            </svg>
                        </div>
                    </div>
                </div>
            </div>

            {/* فیلترها و جستجو */}
            <div className="bg-custom-light-card dark:bg-custom-dark-blue-light p-6 rounded-2xl shadow-sm border border-gray-200 dark:border-gray-700">
                <div className="flex flex-col lg:flex-row items-center justify-between gap-4">
                    <div className="w-full lg:w-auto flex-grow grid grid-cols-1 md:grid-cols-3 gap-4">
                        <div className="relative">
                            <input
                                type="text"
                                placeholder="جستجو در موضوع یا شناسه تیکت..."
                                className="w-full pr-10 pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition"
                                value={searchQuery}
                                onChange={(e) => setSearchQuery(e.target.value)}
                            />
                            <SearchIcon className="absolute right-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
                        </div>
                        <div className="relative">
                            <select
                                className="w-full appearance-none pr-10 pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition"
                                value={statusFilter}
                                onChange={(e) => setStatusFilter(e.target.value)}
                            >
                                <option value="all">همه وضعیت‌ها</option>
                                <option value="answered">پاسخ داده شده</option>
                                <option value="pending">در حال بررسی</option>
                                <option value="closed">بسته شده</option>
                            </select>
                            <ChevronDownIcon className="absolute right-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400 pointer-events-none" />
                        </div>
                        <div className="relative">
                            <select
                                className="w-full appearance-none pr-10 pl-4 py-2.5 border border-gray-300 dark:border-gray-600 rounded-lg bg-custom-light-bg dark:bg-custom-dark-blue focus:ring-2 focus:ring-custom-purple focus:border-transparent outline-none transition"
                                value={departmentFilter}
                                onChange={(e) => setDepartmentFilter(e.target.value)}
                            >
                                <option value="all">همه دپارتمان‌ها</option>
                                <option value="پشتیبانی فنی">پشتیبانی فنی</option>
                                <option value="مالی">مالی</option>
                                <option value="فروش">فروش</option>
                                <option value="پیشنهادات">پیشنهادات</option>
                                <option value="عمومی">عمومی</option>
                            </select>
                            <ChevronDownIcon className="absolute right-3 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400 pointer-events-none" />
                        </div>
                    </div>
                    <button
                        onClick={() => onNavigate('new-ticket')}
                        className="w-full lg:w-auto flex-shrink-0 bg-custom-purple hover:bg-custom-light-purple text-white font-bold py-2.5 px-6 rounded-lg transition-colors duration-300 shadow-lg hover:shadow-xl focus:outline-none focus:ring-4 focus:ring-custom-purple/50 flex items-center justify-center gap-2"
                    >
                        <PlusCircleIcon className="w-5 h-5" />
                        <span>تیکت جدید</span>
                    </button>
                </div>
            </div>

            {/* جدول تیکت‌ها */}
            <div className="bg-custom-light-card dark:bg-custom-dark-blue-light rounded-2xl shadow-sm border border-gray-200 dark:border-gray-700 overflow-hidden">
                <div className="overflow-x-auto">
                    <table className="w-full text-sm text-right">
                        <thead className="bg-custom-light-bg dark:bg-custom-dark-blue text-gray-500 dark:text-gray-400">
                            <tr>
                                <th className="p-4 font-semibold">شناسه</th>
                                <th className="p-4 font-semibold">موضوع</th>
                                <th className="p-4 font-semibold hidden lg:table-cell">دپارتمان</th>
                                <th className="p-4 font-semibold hidden md:table-cell">اولویت</th>
                                <th className="p-4 font-semibold hidden sm:table-cell">آخرین بروزرسانی</th>
                                <th className="p-4 font-semibold">وضعیت</th>
                                <th className="p-4 font-semibold">عملیات</th>
                            </tr>
                        </thead>
                        <tbody className="divide-y divide-gray-200 dark:divide-gray-700">
                            {filteredTickets.length > 0 ? (
                                filteredTickets.map(ticket => (
                                    <tr key={ticket.id} className="hover:bg-gray-50 dark:hover:bg-custom-dark-blue transition-colors">
                                        <td className="p-4 font-mono text-custom-purple font-bold">#{ticket.id}</td>
                                        <td className="p-4">
                                            <div className="font-semibold text-gray-800 dark:text-gray-200">{ticket.subject}</div>
                                            <div className="text-xs text-gray-500 dark:text-gray-400 lg:hidden mt-1">
                                                {ticket.department} {ticket.replies_count ? `• ${ticket.replies_count} پاسخ` : ''}
                                            </div>
                                        </td>
                                        <td className="p-4 hidden lg:table-cell">
                                            <span className="px-2 py-1 text-xs bg-custom-purple/10 text-custom-purple rounded-full border border-custom-purple/20">
                                                {ticket.department}
                                            </span>
                                        </td>
                                        <td className="p-4 hidden md:table-cell">
                                            {getPriorityBadge(ticket.priority || 'normal')}
                                        </td>
                                        <td className="p-4 hidden sm:table-cell text-gray-500 dark:text-gray-400">
                                            {ticket.lastUpdate}
                                        </td>
                                        <td className="p-4">
                                            {getStatusChip(ticket.status)}
                                        </td>
                                        <td className="p-4">
                                            <button
                                                onClick={() => {
                                                    console.log('🔘 Button clicked!');
                                                    console.log('🎫 Ticket:', ticket);
                                                    console.log('🆔 Ticket ID:', ticket.id);
                                                    onNavigate('ticket-detail', ticket.id);
                                                }}
                                                className="bg-custom-purple hover:bg-custom-light-purple text-white font-bold py-2 px-4 rounded-lg transition-colors duration-300 text-xs"
                                            >
                                                مشاهده
                                            </button>

                                        </td>
                                    </tr>
                                ))
                            ) : (
                                <tr>
                                    <td colSpan={7} className="text-center p-12">
                                        <div className="flex flex-col items-center justify-center">
                                            <div className="w-16 h-16 bg-gray-100 dark:bg-gray-700 rounded-full flex items-center justify-center mb-4">
                                                <SearchIcon className="w-8 h-8 text-gray-400" />
                                            </div>
                                            <h3 className="text-lg font-semibold text-gray-800 dark:text-gray-200 mb-2">
                                                {tickets.length === 0 ? 'هنوز هیچ تیکتی ندارید' : 'تیکتی یافت نشد'}
                                            </h3>
                                            <p className="text-gray-500 dark:text-gray-400 mb-4 text-center">
                                                {tickets.length === 0
                                                    ? 'برای شروع اولین تیکت خود را ایجاد کنید.'
                                                    : 'هیچ تیکتی با فیلترهای انتخاب شده مطابقت ندارد.'
                                                }
                                            </p>
                                            {tickets.length === 0 && (
                                                <button
                                                    onClick={() => onNavigate('new-ticket')}
                                                    className="bg-custom-purple hover:bg-custom-light-purple text-white font-bold py-2 px-6 rounded-lg transition-colors duration-300"
                                                >
                                                    ایجاد اولین تیکت
                                                </button>
                                            )}
                                        </div>
                                    </td>
                                </tr>
                            )}
                        </tbody>
                    </table>
                </div>

                {/* Pagination */}
                {filteredTickets.length > 0 && (
                    <div className="bg-custom-light-bg dark:bg-custom-dark-blue px-6 py-4 border-t border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between">
                            <div className="text-sm text-gray-500 dark:text-gray-400">
                                نمایش {filteredTickets.length} از {tickets.length} تیکت
                            </div>
                            <div className="flex items-center gap-2">
                                <button disabled className="px-3 py-1 text-sm border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 disabled:opacity-50 disabled:cursor-not-allowed">
                                    قبلی
                                </button>
                                <span className="px-3 py-1 text-sm bg-custom-purple text-white rounded-lg">1</span>
                                <button disabled className="px-3 py-1 text-sm border border-gray-300 dark:border-gray-600 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 disabled:opacity-50 disabled:cursor-not-allowed">
                                    بعدی
                                </button>
                            </div>
                        </div>
                    </div>
                )}
            </div>

            {/* راهنما */}
            <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-2xl p-6">
                <div className="flex items-start gap-4">
                    <div className="w-8 h-8 bg-blue-500 rounded-full flex items-center justify-center flex-shrink-0">
                        <span className="text-white text-sm">💡</span>
                    </div>
                    <div>
                        <h4 className="font-semibold text-blue-800 dark:text-blue-300 mb-2">راهنمای تیکت‌ها</h4>
                        <ul className="text-sm text-blue-700 dark:text-blue-400 space-y-1">
                            <li>• تیکت‌های <span className="font-semibold">در حال بررسی</span> توسط پشتیبانی در دست اقدام هستند</li>
                            <li>• تیکت‌های <span className="font-semibold">پاسخ داده شده</span> نیاز به پاسخ‌گویی شما دارند</li>
                            <li>• می‌توانید تیکت‌های حل شده را <span className="font-semibold">ببندید</span></li>
                            <li>• برای مسائل فوری از اولویت <span className="font-semibold">بالا</span> استفاده کنید</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    );
};

export default TicketListPage;
