<?php
// app/Http/Controllers/Api/OnlinePurchaseController.php

namespace App\Http\Controllers\Api;




use App\Http\Controllers\Controller;
use App\Models\OnlinePurchaseImage;
use App\Models\OnlinePurchaseRequest;
use App\Models\OnlinePurchaseShopper;
use App\Models\OnlinePurchaseShopperImage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class OnlinePurchaseController extends Controller
{
    /**
     * ثبت درخواست خرید جدید
     */
    public function storeRequest(Request $request)
    {
        // اعتبارسنجی داده‌ها
        $validator = Validator::make($request->all(), [
            'title'            => 'required|string|max:255',
            'link'             => 'nullable|url|max:500',
            'description'      => 'nullable|string|max:1000',
            'delivery_lat'     => 'nullable|numeric|between:-90,90',
            'delivery_lng'     => 'nullable|numeric|between:-180,180',
            'delivery_country' => 'nullable|string|max:100',
            'delivery_city'    => 'nullable|string|max:100',
            'delivery_address' => 'nullable|string|max:500',
            'item_price'       => 'required|numeric|min:0',
            'proposed_fee'     => 'required|numeric|min:0',
            'currency'         => 'required|in:IRT,USD,CAD,IQD',
            'show_phone'       => 'boolean',
            'images'           => 'nullable|array',
            'images.*'         => 'image|mimes:jpeg,png,jpg,gif|max:2048',
        ], [
            'title.required'        => 'عنوان کالا الزامی است',
            'item_price.required'   => 'قیمت کالا الزامی است',
            'proposed_fee.required' => 'هزینه پیشنهادی الزامی است',
            'item_price.min'        => 'قیمت کالا باید بیشتر از صفر باشد',
            'proposed_fee.min'      => 'هزینه پیشنهادی باید بیشتر از صفر باشد',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در اعتبارسنجی',
                'errors'  => $validator->errors(),
            ], 422);
        }

        try {
            DB::beginTransaction();

            // ایجاد درخواست خرید
            $purchaseRequest = OnlinePurchaseRequest::create([
                'user_id'          => 1,
                'title'            => $request->title,
                'link'             => $request->link,
                'description'      => $request->description,
                'delivery_lat'     => $request->delivery_lat,
                'delivery_lng'     => $request->delivery_lng,
                'delivery_country' => $request->delivery_country,
                'delivery_city'    => $request->delivery_city,
                'delivery_address' => $request->delivery_address,
                'item_price'       => $request->item_price,
                'proposed_fee'     => $request->proposed_fee,
                'currency'         => $request->currency,
                'show_phone'       => filter_var($request->show_phone, FILTER_VALIDATE_BOOLEAN),
                'status'           => 'active',
            ]);

            // ذخیره تصاویر
            if ($request->hasFile('images')) {
                foreach ($request->file('images') as $image) {
                    $filename = time() . '_' . uniqid() . '.' . $image->extension();
                    $image->move(public_path('uploads/online-purchase-requests'), $filename);

                    $purchaseRequest->images()->create([
                        'path' => 'uploads/online-purchase-requests/' . $filename,
                    ]);
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'درخواست خرید با موفقیت ثبت شد',
                'data'    => $purchaseRequest->load('images', 'user'),
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Purchase request registration error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'خطا در ثبت درخواست خرید',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * ثبت اطلاعات خریدار
     */
    public function storeShopper(Request $request)
    {
        // اعتبارسنجی داده‌ها
        $validator = Validator::make($request->all(), [
            'country'           => 'required|string|max:100',
            'city'              => 'required|string|max:100',
            'service_title'     => 'nullable|string|max:255',
            'categories'        => 'nullable|array',
            'categories.*'      => 'string|max:100',
            'supported_sites'   => 'nullable|array',
            'supported_sites.*' => 'string|max:255',
            'fee_description'   => 'nullable|string|max:1000',
            'capabilities'      => 'nullable|string|max:1000',
            'show_phone'        => 'boolean',
            'images'            => 'nullable|array',
            'images.*'          => 'image|mimes:jpeg,png,jpg,gif|max:2048',
        ], [
            'country.required' => 'کشور الزامی است',
            'city.required'    => 'شهر الزامی است',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در اعتبارسنجی',
                'errors'  => $validator->errors(),
            ], 422);
        }

        try {
            DB::beginTransaction();

            // بررسی اینکه آیا کاربر قبلاً پروفایل خریدار دارد
            $existingShopper = OnlinePurchaseShopper::where('user_id', 1)->first();

            if ($existingShopper) {
                // آپدیت پروفایل موجود
                $existingShopper->update([
                    'country'         => $request->country,
                    'city'            => $request->city,
                    'service_title'   => $request->service_title,
                    'categories'      => $request->categories,
                    'supported_sites' => $request->supported_sites,
                    'fee_description' => $request->fee_description,
                    'capabilities'    => $request->capabilities,
                    'show_phone'      => filter_var($request->show_phone, FILTER_VALIDATE_BOOLEAN),
                    'status'          => 'active',
                ]);

                $shopper = $existingShopper;
            } else {
                // ایجاد خریدار جدید
                $shopper = OnlinePurchaseShopper::create([
                    'user_id'         => 1,
                    'country'         => $request->country,
                    'city'            => $request->city,
                    'service_title'   => $request->service_title,
                    'categories'      => $request->categories,
                    'supported_sites' => $request->supported_sites,
                    'fee_description' => $request->fee_description,
                    'capabilities'    => $request->capabilities,
                    'show_phone'      => filter_var($request->show_phone, FILTER_VALIDATE_BOOLEAN),
                    'status'          => 'active',
                ]);
            }

            // ذخیره تصاویر
            if ($request->hasFile('images')) {
                // حذف تصاویر قبلی
                $shopper->images()->delete();

                foreach ($request->file('images') as $image) {
                    $filename = time() . '_' . uniqid() . '.' . $image->extension();
                    $image->move(public_path('uploads/online-purchase-shoppers'), $filename);

                    $shopper->images()->create([
                        'path' => 'uploads/online-purchase-shoppers/' . $filename,
                    ]);
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => $existingShopper ? 'پروفایل خریدار با موفقیت به‌روزرسانی شد' : 'اطلاعات خریدار با موفقیت ثبت شد',
                'data'    => $shopper->load('images', 'user'),
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Shopper registration error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'خطا در ثبت اطلاعات خریدار',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * دریافت لیست درخواست‌های خرید
     */
    public function getRequests(Request $request)
    {
        try {
            $requests = OnlinePurchaseRequest::where('status', 'active')
                ->with(['user:id,name,phone', 'images'])
                ->orderBy('created_at', 'desc')
                ->get();

            // فرمت‌دهی تصاویر مشابه ShipmentController
            $formattedRequests = $requests->map(function ($request) {
                $formattedImages = $request->images->map(function ($image) {
                    return [
                        'id'  => $image->id,
                        'url' => asset($image->path),
                    ];
                });

                $requestArray           = $request->toArray();
                $requestArray['images'] = $formattedImages;

                return $requestArray;
            });

            return response()->json([
                'success' => true,
                'data'    => $formattedRequests,
            ]);

        } catch (\Exception $e) {
            Log::error('Get purchase requests error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت اطلاعات',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * دریافت لیست خریداران
     */
    public function getShoppers(Request $request)
    {
        try {
            $shoppers = OnlinePurchaseShopper::where('status', 'active')
                ->with(['user:id,name,phone', 'images'])
                ->orderBy('created_at', 'desc')
                ->get();

            // فرمت‌دهی تصاویر مشابه ShipmentController
            $formattedShoppers = $shoppers->map(function ($shopper) {
                $formattedImages = $shopper->images->map(function ($image) {
                    return [
                        'id'  => $image->id,
                        'url' => asset($image->path),
                    ];
                });

                $shopperArray           = $shopper->toArray();
                $shopperArray['images'] = $formattedImages;

                return $shopperArray;
            });

            return response()->json([
                'success' => true,
                'data'    => $formattedShoppers,
            ]);

        } catch (\Exception $e) {
            Log::error('Get shoppers error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت اطلاعات',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * دریافت درخواست‌های خرید من
     */
    public function getMyRequests(Request $request)
    {
        try {
            $requests = OnlinePurchaseRequest::where('user_id', 1)
                ->with(['user:id,name,phone', 'images'])
                ->orderBy('created_at', 'desc')
                ->get();

            // فرمت‌دهی تصاویر مشابه ShipmentController
            $formattedRequests = $requests->map(function ($request) {
                $formattedImages = $request->images->map(function ($image) {
                    return [
                        'id'  => $image->id,
                        'url' => asset($image->path),
                    ];
                });

                $requestArray           = $request->toArray();
                $requestArray['images'] = $formattedImages;

                return $requestArray;
            });

            return response()->json([
                'success' => true,
                'data'    => $formattedRequests,
            ]);

        } catch (\Exception $e) {
            Log::error('Get my purchase requests error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت اطلاعات',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * دریافت خریدار من (فقط یک پروفایل)
     */
    public function getMyShopper(Request $request)
    {
        try {
            $shopper = OnlinePurchaseShopper::where('user_id', 1)
                ->with(['user:id,name,phone', 'images'])
                ->first();

            if (! $shopper) {
                return response()->json([
                    'success' => true,
                    'data'    => null,
                ]);
            }

            // فرمت‌دهی تصاویر مشابه ShipmentController
            $formattedImages = $shopper->images->map(function ($image) {
                return [
                    'id'  => $image->id,
                    'url' => asset($image->path),
                ];
            });

            $shopperArray           = $shopper->toArray();
            $shopperArray['images'] = $formattedImages;

            return response()->json([
                'success' => true,
                'data'    => $shopperArray,
            ]);

        } catch (\Exception $e) {
            Log::error('Get my shopper error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت اطلاعات',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * دریافت اطلاعات یک درخواست خرید خاص
     */
    public function getRequest($id)
    {
        try {
            $request = OnlinePurchaseRequest::with(['user:id,name,phone', 'images'])->findOrFail($id);

            // فرمت‌دهی تصاویر مشابه ShipmentController
            $formattedImages = $request->images->map(function ($image) {
                return [
                    'id'  => $image->id,
                    'url' => asset($image->path),
                ];
            });

            $requestArray           = $request->toArray();
            $requestArray['images'] = $formattedImages;

            return response()->json([
                'success' => true,
                'data'    => $requestArray,
            ]);

        } catch (\Exception $e) {
            Log::error('Get purchase request error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'درخواست خرید مورد نظر یافت نشد',
            ], 404);
        }
    }

    /**
     * دریافت اطلاعات یک خریدار خاص
     */
    public function getShopper($id)
    {
        try {
            $shopper = OnlinePurchaseShopper::with(['user:id,name,phone', 'images'])->findOrFail($id);

            // فرمت‌دهی تصاویر مشابه ShipmentController
            $formattedImages = $shopper->images->map(function ($image) {
                return [
                    'id'  => $image->id,
                    'url' => asset($image->path),
                ];
            });

            $shopperArray           = $shopper->toArray();
            $shopperArray['images'] = $formattedImages;

            return response()->json([
                'success' => true,
                'data'    => $shopperArray,
            ]);

        } catch (\Exception $e) {
            Log::error('Get shopper error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'خریدار مورد نظر یافت نشد',
            ], 404);
        }
    }

    /**
     * حذف درخواست خرید
     */
    public function deleteRequest($id)
    {
        try {
            $request = OnlinePurchaseRequest::findOrFail($id);

            // چک کردن مالکیت
            if ($request->user_id !== auth()->id()) {
                return response()->json([
                    'success' => false,
                    'message' => 'شما اجازه حذف این درخواست را ندارید',
                ], 403);
            }

            // حذف تصاویر
            foreach ($request->images as $image) {
                // حذف فایل از پوشه public
                if (file_exists(public_path($image->path))) {
                    unlink(public_path($image->path));
                }
                $image->delete();
            }

            $request->delete();

            return response()->json([
                'success' => true,
                'message' => 'درخواست خرید با موفقیت حذف شد',
            ]);

        } catch (\Exception $e) {
            Log::error('Delete purchase request error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'خطا در حذف درخواست خرید',
            ], 500);
        }
    }

    /**
     * حذف خریدار
     */
    public function deleteShopper($id)
    {
        try {
            $shopper = OnlinePurchaseShopper::findOrFail($id);

            // چک کردن مالکیت
            if ($shopper->user_id !== auth()->id()) {
                return response()->json([
                    'success' => false,
                    'message' => 'شما اجازه حذف این خریدار را ندارید',
                ], 403);
            }

            // حذف تصاویر
            foreach ($shopper->images as $image) {
                // حذف فایل از پوشه public
                if (file_exists(public_path($image->path))) {
                    unlink(public_path($image->path));
                }
                $image->delete();
            }

            $shopper->delete();

            return response()->json([
                'success' => true,
                'message' => 'خریدار با موفقیت حذف شد',
            ]);

        } catch (\Exception $e) {
            Log::error('Delete shopper error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'خطا در حذف خریدار',
            ], 500);
        }
    }

    /**
     * غیرفعال کردن خریدار (به جای حذف)
     */
    public function deactivateShopper()
    {
        try {
            $shopper = OnlinePurchaseShopper::where('user_id', auth()->id())->first();

            if (! $shopper) {
                return response()->json([
                    'success' => false,
                    'message' => 'پروفایل خریداری یافت نشد',
                ], 404);
            }

            $shopper->update(['status' => 'inactive']);

            return response()->json([
                'success' => true,
                'message' => 'پروفایل خریدار با موفقیت غیرفعال شد',
            ]);

        } catch (\Exception $e) {
            Log::error('Deactivate shopper error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'خطا در غیرفعال کردن خریدار',
            ], 500);
        }
    }

    /**
     * فعال کردن خریدار
     */
    public function activateShopper()
    {
        try {
            $shopper = OnlinePurchaseShopper::where('user_id', auth()->id())->first();

            if (! $shopper) {
                return response()->json([
                    'success' => false,
                    'message' => 'پروفایل خریداری یافت نشد',
                ], 404);
            }

            $shopper->update(['status' => 'active']);

            return response()->json([
                'success' => true,
                'message' => 'پروفایل خریدار با موفقیت فعال شد',
            ]);

        } catch (\Exception $e) {
            Log::error('Activate shopper error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'خطا در فعال کردن خریدار',
            ], 500);
        }
    }

    public function updateRequest(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'title'            => 'required|string|max:255',
            'link'             => 'nullable|url|max:500',
            'description'      => 'nullable|string|max:1000',
            'delivery_lat'     => 'nullable|numeric|between:-90,90',
            'delivery_lng'     => 'nullable|numeric|between:-180,180',
            'delivery_country' => 'nullable|string|max:100',
            'delivery_city'    => 'nullable|string|max:100',
            'delivery_address' => 'nullable|string|max:500',
            'item_price'       => 'required|numeric|min:0',
            'proposed_fee'     => 'required|numeric|min:0',
            'currency'         => 'required|in:IRT,USD,CAD,IQD',
            'show_phone'       => 'boolean',
            'images'           => 'nullable|array',
            'images.*'         => 'image|mimes:jpeg,png,jpg,gif|max:2048',
            'deleted_images'   => 'nullable|array',
            'deleted_images.*' => 'integer',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در اعتبارسنجی',
                'errors'  => $validator->errors(),
            ], 422);
        }

        try {
            DB::beginTransaction();

            $purchaseRequest = OnlinePurchaseRequest::where('user_id', 1)
                ->findOrFail($id);

            // به‌روزرسانی اطلاعات
            $purchaseRequest->update([
                'title'            => $request->title,
                'link'             => $request->link,
                'description'      => $request->description,
                'delivery_lat'     => $request->delivery_lat,
                'delivery_lng'     => $request->delivery_lng,
                'delivery_country' => $request->delivery_country,
                'delivery_city'    => $request->delivery_city,
                'delivery_address' => $request->delivery_address,
                'item_price'       => $request->item_price,
                'proposed_fee'     => $request->proposed_fee,
                'currency'         => $request->currency,
                'show_phone'       => $request->show_phone ?? false,
            ]);

            // حذف تصاویر انتخاب شده
            if ($request->has('deleted_images')) {
                foreach ($request->deleted_images as $imageId) {
                    $image = OnlinePurchaseImage::where('purchase_request_id', $id)
                        ->where('id', $imageId)
                        ->first();
                    if ($image) {
                        Storage::disk('public')->delete($image->path);
                        $image->delete();
                    }
                }
            }

            // آپلود تصاویر جدید
            if ($request->hasFile('images')) {
                foreach ($request->file('images') as $image) {
                    $filename = time() . '_' . uniqid() . '.' . $image->extension();
                    $image->move(public_path('uploads/online-purchase-requests'), $filename);
                    $purchaseRequest->images()->create([
                        'path' => 'uploads/online-purchase-requests/' . $filename,
                    ]);
                }
            }

            DB::commit();

            $purchaseRequest->load('images');

            return response()->json([
                'success' => true,
                'message' => 'درخواست خرید با موفقیت ویرایش شد',
                'data'    => $purchaseRequest,
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('خطا در ویرایش درخواست خرید: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'خطا در ویرایش درخواست خرید',
            ], 500);
        }
    }

/**
 * ویرایش پروفایل خریدار
 */
    public function updateShopper(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'service_title'     => 'nullable|string|max:255',
            'country'           => 'required|string|max:100',
            'city'              => 'nullable|string|max:100',
            'categories'        => 'nullable|array',
            'categories.*'      => 'string|max:100',
            'supported_sites'   => 'nullable|array',
            'supported_sites.*' => 'url|max:255',
            'fee_description'   => 'nullable|string|max:500',
            'capabilities'      => 'nullable|string|max:1000',
            'show_phone'        => 'boolean',
            'images'            => 'nullable|array',
            'images.*'          => 'image|mimes:jpeg,png,jpg,gif|max:2048',
            'deleted_images'    => 'nullable|array',
            'deleted_images.*'  => 'integer',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در اعتبارسنجی',
                'errors'  => $validator->errors(),
            ], 422);
        }

        try {
            DB::beginTransaction();

            $shopper = OnlinePurchaseShopper::where('user_id', 1)
                ->findOrFail($id);

            $shopper->update([
                'service_title'   => $request->service_title,
                'country'         => $request->country,
                'city'            => $request->city,
                'categories'      => $request->categories ? json_encode($request->categories) : null,
                'supported_sites' => $request->supported_sites ? json_encode($request->supported_sites) : null,
                'fee_description' => $request->fee_description,
                'capabilities'    => $request->capabilities,
                'show_phone'      => $request->show_phone ?? false,
            ]);

            // حذف تصاویر
            if ($request->has('deleted_images')) {
                foreach ($request->deleted_images as $imageId) {
                    $image = OnlinePurchaseShopperImage::where('shopper_id', $id)
                        ->where('id', $imageId)
                        ->first();
                    if ($image) {
                        Storage::disk('public')->delete($image->path);
                        $image->delete();
                    }
                }
            }

            // آپلود تصاویر جدید
            if ($request->hasFile('images')) {
                foreach ($request->file('images') as $image) {
                     $filename = time() . '_' . uniqid() . '.' . $image->extension();
                    $image->move(public_path('uploads/online-purchase-shoppers'), $filename);
                    $shopper->images()->create([
                        'path' => 'uploads/online-purchase-shoppers/' . $filename,
                    ]);
                }
            }

            DB::commit();

            $shopper->load('images');

            return response()->json([
                'success' => true,
                'message' => 'پروفایل خریدار با موفقیت ویرایش شد',
                'data'    => $shopper,
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('خطا در ویرایش پروفایل خریدار: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'خطا در ویرایش پروفایل خریدار',
            ], 500);
        }
    }
}
