<?php
namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Ticket;
use App\Models\TicketAttachment;
use App\Models\TicketReply;  
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class TicketController extends Controller
{
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title'         => 'required|string|max:255',
            'message'       => 'required|string',
            'department'    => 'required|in:technical,financial,suggestions',
            'priority'      => 'required|in:low,medium,high',
            'attachments'   => 'array|max:5',
            'attachments.*' => 'file|mimes:jpg,jpeg,png,pdf,doc,docx|max:10240',
        ], [
            'title.required'      => 'عنوان تیکت الزامی است',
            'message.required'    => 'متن پیام الزامی است',
            'department.required' => 'انتخاب دپارتمان الزامی است',
            'priority.required'   => 'انتخاب اولویت الزامی است',
            'attachments.max'     => 'حداکثر ۵ فایل می‌توانید آپلود کنید',
            'attachments.*.max'   => 'حجم هر فایل نباید بیشتر از ۱۰ مگابایت باشد',
            'attachments.*.mimes' => 'فرمت فایل باید از نوع: jpg, jpeg, png, pdf, doc, docx باشد',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در اعتبارسنجی',
                'errors'  => $validator->errors(),
            ], 422);
        }

        try {
            DB::beginTransaction();

            // ایجاد تیکت
            $ticket = Ticket::create([
                'user_id'    => $request->user()->id,
                'title'      => $request->title,
                'message'    => $request->message,
                'department' => $request->department,
                'priority'   => $request->priority,
                'status'     => 'open',
            ]);

            // آپلود فایل‌های پیوست
            if ($request->hasFile('attachments')) {
                // ایجاد پوشه tickets در صورت عدم وجود
                $uploadPath = public_path('uploads/tickets');
                if (! file_exists($uploadPath)) {
                    mkdir($uploadPath, 0755, true);
                }

                foreach ($request->file('attachments') as $file) {
                    // ⭐ گرفتن اطلاعات قبل از move کردن
                    $originalName = $file->getClientOriginalName();
                    $fileSize     = $file->getSize();
                    $mimeType     = $file->getMimeType();
                    $extension    = $file->getClientOriginalExtension();

                    // تولید نام یونیک برای فایل
                    $fileName = time() . '_' . uniqid() . '.' . $extension;

                    // انتقال فایل به پوشه public/uploads/tickets
                    $file->move($uploadPath, $fileName);

                    // مسیر نسبی برای ذخیره در دیتابیس
                    $relativePath = 'uploads/tickets/' . $fileName;

                    // ⭐ استفاده از اطلاعات ذخیره شده
                    TicketAttachment::create([
                        'ticket_id' => $ticket->id,
                        'file_path' => $relativePath,
                        'file_name' => $originalName,
                        'file_size' => $fileSize,
                        'mime_type' => $mimeType,
                    ]);
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'تیکت با موفقیت ایجاد شد',
                'data'    => $ticket->load(['attachments', 'user']),
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();

            Log::error('Ticket creation error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'خطا در ایجاد تیکت',
                'error'   => config('app.debug') ? $e->getMessage() : 'خطای سرور',
            ], 500);
        }
    }

    /**
     * دریافت لیست تیکت‌های کاربر
     */
    public function index(Request $request)
    {
        try {
            $tickets = Ticket::where('user_id', $request->user()->id)
                ->with(['user', 'attachments'])
                ->orderBy('created_at', 'desc')
                ->get();

            return response()->json([
                'success' => true,
                'data'    => $tickets,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت اطلاعات',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * دریافت اطلاعات یک تیکت خاص
     */
    public function show($id)
    {
        try {
            $ticket = Ticket::with(['user', 'attachments', 'replies.user', 'replies.attachments'])
                ->findOrFail($id);

            // چک کردن مالکیت تیکت
            if ($ticket->user_id !== auth()->id()) {
                return response()->json([
                    'success' => false,
                    'message' => 'شما اجازه دسترسی به این تیکت را ندارید',
                ], 403);
            }

            return response()->json([
                'success' => true,
                'data'    => $ticket,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'تیکت مورد نظر یافت نشد',
            ], 404);
        }
    }

    public function reply(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'message'       => 'required_without:attachments|string',
            'attachments'   => 'array|max:5',
            'attachments.*' => 'file|mimes:jpg,jpeg,png,pdf,doc,docx|max:10240',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در اعتبارسنجی',
                'errors'  => $validator->errors(),
            ], 422);
        }

        try {
            $ticket = Ticket::findOrFail($id);

            // چک کردن مالکیت تیکت
            if ($ticket->user_id !== $request->user()->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'شما اجازه پاسخ به این تیکت را ندارید',
                ], 403);
            }

            DB::beginTransaction();

            // ایجاد پاسخ
            $reply = TicketReply::create([
                'ticket_id' => $ticket->id,
                'user_id'   => $request->user()->id,
                'message'   => $request->message,
                'is_admin'  => false,
            ]);

            // آپلود فایل‌های پیوست
            if ($request->hasFile('attachments')) {
                $uploadPath = public_path('uploads/tickets');
                if (! file_exists($uploadPath)) {
                    mkdir($uploadPath, 0755, true);
                }

                foreach ($request->file('attachments') as $file) {
                    // ⭐ گرفتن اطلاعات قبل از move کردن
                    $originalName = $file->getClientOriginalName();
                    $fileSize     = $file->getSize();
                    $mimeType     = $file->getMimeType();
                    $extension    = $file->getClientOriginalExtension();

                    // تولید نام یونیک برای فایل
                    $fileName = time() . '_' . uniqid() . '.' . $extension;

                    // انتقال فایل به پوشه public/uploads/tickets
                    $file->move($uploadPath, $fileName);

                    // مسیر نسبی برای ذخیره در دیتابیس
                    $relativePath = 'uploads/tickets/' . $fileName;

                    // ⭐ استفاده از اطلاعات ذخیره شده
                    TicketAttachment::create([
                        'ticket_id'       => $ticket->id,
                        'ticket_reply_id' => $reply->id,
                        'file_path'       => $relativePath,
                        'file_name'       => $originalName,
                        'file_size'       => $fileSize,
                        'mime_type'       => $mimeType,
                    ]);
                }
            }

            // آپدیت وضعیت تیکت
            $ticket->update(['status' => 'in_progress']);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'پاسخ با موفقیت ارسال شد',
                'data'    => $reply->load('attachments'),
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            Log::error('Reply creation error: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'خطا در ارسال پاسخ',
                'error'   => config('app.debug') ? $e->getMessage() : 'خطای سرور',
            ], 500);
        }
    }

    /**
     * بستن تیکت
     */
    public function close($id)
    {
        try {
            $ticket = Ticket::findOrFail($id);

            // چک کردن مالکیت تیکت
            if ($ticket->user_id !== auth()->id()) {
                return response()->json([
                    'success' => false,
                    'message' => 'شما اجازهبستناینتیکتراندارید',
                ], 403);
            }

            $ticket->update(['status' => 'closed']);

            return response()->json([
                'success' => true,
                'message' => 'تیکت باموفقیتبستهشد',
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطا دربستنتیکت',
            ], 500);
        }
    }
}
