<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Trip;
use App\Models\TripCompanion;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;

class TripController extends Controller
{
    /**
     * ثبت سفر جدید
     */
    public function registerTrip(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'origin.country' => 'required|string|max:100',
            'origin.city' => 'required|string|max:100',
            'destination.country' => 'required|string|max:100',
            'destination.city' => 'required|string|max:100',
            'vehicle_type' => 'required|string|in:private_car,bus,train,airplane',
            'departure_date' => 'required|date',
            'capacity' => 'required|numeric|min:0.1',
        ], [
            'origin.country.required' => 'کشور مبدأ الزامی است',
            'origin.city.required' => 'شهر مبدأ الزامی است',
            'destination.country.required' => 'کشور مقصد الزامی است',
            'destination.city.required' => 'شهر مقصد الزامی است',
            'vehicle_type.required' => 'نوع وسیله نقلیه الزامی است',
            'departure_date.required' => 'تاریخ حرکت الزامی است',
            'capacity.required' => 'ظرفیت حمل الزامی است',
            'capacity.min' => 'ظرفیت حمل باید بیشتر از صفر باشد',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در اعتبارسنجی',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $trip = Trip::create([
                'user_id' => $request->user()->id,
                'origin_country' => $request->origin['country'],
                'origin_city' => $request->origin['city'],
                'origin_address' => $request->origin['address'] ?? null,
                'origin_coordinates' => $request->origin['coordinates'] ?? null,
                'destination_country' => $request->destination['country'],
                'destination_city' => $request->destination['city'],
                'destination_address' => $request->destination['address'] ?? null,
                'destination_coordinates' => $request->destination['coordinates'] ?? null,
                'vehicle_type' => $request->vehicle_type,
                'departure_date' => $request->departure_date,
                'arrival_date' => $request->arrival_date,
                'capacity' => $request->capacity,
                'description' => $request->description,
                'show_phone' => $request->show_phone ?? false,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'سفر با موفقیت ثبت شد',
                'data' => $trip->load('user')
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در ثبت سفر',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * ثبت آمادگی برای همسفری
     */
    public function registerCompanion(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'origin.country' => 'required|string|max:100',
            'origin.city' => 'required|string|max:100',
            'destination.country' => 'required|string|max:100',
            'destination.city' => 'required|string|max:100',
            'vehicle_types' => 'required|array|min:1',
            'vehicle_types.*' => 'in:private_car,bus,train,airplane',
            'available_from' => 'required|date',
            'capacity' => 'required|numeric|min:0.1',
        ], [
            'origin.country.required' => 'کشور مبدأ الزامی است',
            'origin.city.required' => 'شهر مبدأ الزامی است',
            'destination.country.required' => 'کشور مقصد الزامی است',
            'destination.city.required' => 'شهر مقصد الزامی است',
            'vehicle_types.required' => 'حداقل یک وسیله نقلیه باید انتخاب شود',
            'vehicle_types.min' => 'حداقل یک وسیله نقلیه باید انتخاب شود',
            'available_from.required' => 'تاریخ شروع الزامی است',
            'capacity.required' => 'ظرفیت حمل الزامی است',
            'capacity.min' => 'ظرفیت حمل باید بیشتر از صفر باشد',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در اعتبارسنجی',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $companion = TripCompanion::create([
                'user_id' => $request->user()->id,
                'origin_country' => $request->origin['country'],
                'origin_city' => $request->origin['city'],
                'origin_address' => $request->origin['address'] ?? null,
                'origin_coordinates' => $request->origin['coordinates'] ?? null,
                'destination_country' => $request->destination['country'],
                'destination_city' => $request->destination['city'],
                'destination_address' => $request->destination['address'] ?? null,
                'destination_coordinates' => $request->destination['coordinates'] ?? null,
                'vehicle_types' => $request->vehicle_types,
                'available_from' => $request->available_from,
                'available_to' => $request->available_to,
                'capacity' => $request->capacity,
                'description' => $request->description,
                'show_phone' => $request->show_phone ?? false,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'آمادگی برای همسفری با موفقیت ثبت شد',
                'data' => $companion->load('user')
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در ثبت آمادگی',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * دریافت لیست سفرهای من
     */
    public function getMyTrips(Request $request)
    {
        try {
            $trips = Trip::where('user_id', $request->user()->id)
                ->with('user', 'images')
                ->orderBy('created_at', 'desc')
                ->get();

            return response()->json([
                'success' => true,
                'data' => $trips
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت اطلاعات',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * دریافت لیست همسفرهای من
     */
    public function getMyCompanions(Request $request)
    {
        try {
            $companions = TripCompanion::where('user_id', $request->user()->id)
                ->with('user')
                ->orderBy('created_at', 'desc')
                ->get();

            return response()->json([
                'success' => true,
                'data' => $companions
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت اطلاعات',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * دریافت لیست همه سفرهای فعال
     */
    public function getTrips(Request $request)
    {
        try {
            $trips = Trip::where('status', 'active')
                ->with('user', 'images')
                ->orderBy('created_at', 'desc')
                ->get();

            return response()->json([
                'success' => true,
                'data' => $trips
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت اطلاعات',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * دریافت لیست همه همسفرهای فعال
     */
    public function getCompanions(Request $request)
    {
        try {
            $companions = TripCompanion::where('status', 'active')
                ->with('user')
                ->orderBy('created_at', 'desc')
                ->get();

            return response()->json([
                'success' => true,
                'data' => $companions
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در دریافت اطلاعات',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * دریافت اطلاعات یک سفر خاص
     */
    public function getTrip($id)
    {
        try {
            $trip = Trip::with('user', 'images')->findOrFail($id);

            return response()->json([
                'success' => true,
                'data' => $trip
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'سفر مورد نظر یافت نشد'
            ], 404);
        }
    }

    /**
     * دریافت اطلاعات یک همسفر خاص
     */
    public function getCompanion($id)
    {
        try {
            $companion = TripCompanion::with('user')->findOrFail($id);

            return response()->json([
                'success' => true,
                'data' => $companion
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'همسفر مورد نظر یافت نشد'
            ], 404);
        }
    }

    /**
     * حذف سفر
     */
    public function deleteTrip(Request $request, $id)
    {
        try {
            $trip = Trip::find($id);

            if (!$trip) {
                return response()->json([
                    'success' => false,
                    'message' => 'سفر مورد نظر یافت نشد'
                ], 404);
            }

            // بررسی مالکیت
            if ($trip->user_id !== $request->user()->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'شما مجاز به حذف این سفر نیستید'
                ], 403);
            }

            $trip->delete();

            return response()->json([
                'success' => true,
                'message' => 'سفر با موفقیت حذف شد'
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در حذف سفر',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * حذف همسفر
     */
    public function deleteCompanion(Request $request, $id)
    {
        try {
            $companion = TripCompanion::find($id);

            if (!$companion) {
                return response()->json([
                    'success' => false,
                    'message' => 'همسفر مورد نظر یافت نشد'
                ], 404);
            }

            // بررسی مالکیت
            if ($companion->user_id !== $request->user()->id) {
                return response()->json([
                    'success' => false,
                    'message' => 'شما مجاز به حذف این همسفر نیستید'
                ], 403);
            }

            $companion->delete();

            return response()->json([
                'success' => true,
                'message' => 'همسفر با موفقیت حذف شد'
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در حذف همسفر',
                'error' => $e->getMessage()
            ], 500);
        }
    }


      public function updateTrip(Request $request, $id)
    {
        $trip = Trip::find($id);

        if (!$trip) {
            return response()->json([
                'success' => false,
                'message' => 'سفر مورد نظر یافت نشد'
            ], 404);
        }

        // بررسی مالکیت
        if ($trip->user_id !== Auth::id()) {
            return response()->json([
                'success' => false,
                'message' => 'شما مجاز به ویرایش این سفر نیستید'
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'origin.country' => 'required|string|max:255',
            'origin.city' => 'required|string|max:255',
            'origin.address' => 'nullable|string',
            'origin.coordinates' => 'nullable|string',
            'destination.country' => 'required|string|max:255',
            'destination.city' => 'required|string|max:255',
            'destination.address' => 'nullable|string',
            'destination.coordinates' => 'nullable|string',
            'vehicle_type' => 'required|string|in:private_car,bus,train,airplane',
            'departure_date' => 'required|date',
            'arrival_date' => 'nullable|date|after:departure_date',
            'capacity' => 'required|numeric|min:0.1',
            'description' => 'nullable|string',
            'show_phone' => 'boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در اعتبارسنجی',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $trip->update([
                'origin_country' => $request->origin['country'],
                'origin_city' => $request->origin['city'],
                'origin_address' => $request->origin['address'] ?? null,
                'origin_coordinates' => $request->origin['coordinates'] ?? null,
                'destination_country' => $request->destination['country'],
                'destination_city' => $request->destination['city'],
                'destination_address' => $request->destination['address'] ?? null,
                'destination_coordinates' => $request->destination['coordinates'] ?? null,
                'vehicle_type' => $request->vehicle_type,
                'departure_date' => $request->departure_date,
                'arrival_date' => $request->arrival_date,
                'capacity' => $request->capacity,
                'description' => $request->description,
                'show_phone' => $request->show_phone ?? false,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'سفر با موفقیت به‌روزرسانی شد',
                'data' => $trip->load('user', 'images')
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در به‌روزرسانی سفر',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * به‌روزرسانی همسفری
     */
    public function updateCompanion(Request $request, $id)
    {
        $companion = TripCompanion::find($id);

        if (!$companion) {
            return response()->json([
                'success' => false,
                'message' => 'همسفر مورد نظر یافت نشد'
            ], 404);
        }

        // بررسی مالکیت
        if ($companion->user_id !== Auth::id()) {
            return response()->json([
                'success' => false,
                'message' => 'شما مجاز به ویرایش این همسفری نیستید'
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'origin.country' => 'required|string|max:255',
            'origin.city' => 'required|string|max:255',
            'origin.address' => 'nullable|string',
            'origin.coordinates' => 'nullable|string',
            'destination.country' => 'required|string|max:255',
            'destination.city' => 'required|string|max:255',
            'destination.address' => 'nullable|string',
            'destination.coordinates' => 'nullable|string',
            'vehicle_types' => 'required|array|min:1',
            'vehicle_types.*' => 'in:private_car,bus,train,airplane',
            'available_from' => 'required|date',
            'available_to' => 'nullable|date|after:available_from',
            'capacity' => 'required|numeric|min:0.1',
            'description' => 'nullable|string',
            'show_phone' => 'boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در اعتبارسنجی',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $companion->update([
                'origin_country' => $request->origin['country'],
                'origin_city' => $request->origin['city'],
                'origin_address' => $request->origin['address'] ?? null,
                'origin_coordinates' => $request->origin['coordinates'] ?? null,
                'destination_country' => $request->destination['country'],
                'destination_city' => $request->destination['city'],
                'destination_address' => $request->destination['address'] ?? null,
                'destination_coordinates' => $request->destination['coordinates'] ?? null,
                'vehicle_types' => $request->vehicle_types,
                'available_from' => $request->available_from,
                'available_to' => $request->available_to,
                'capacity' => $request->capacity,
                'description' => $request->description,
                'show_phone' => $request->show_phone ?? false,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'همسفری با موفقیت به‌روزرسانی شد',
                'data' => $companion->load('user')
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'خطا در به‌روزرسانی همسفری',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
